import os
import textwrap
import subprocess

from . import base
from . import unified_agent


class Service(base.Base):
    # Use the same settings as Sandbox Server
    _CONFIG = base.SandboxLauncher._CONFIG

    _CONFIG_PATCHES = base.unfold_list([{
        ("sandbox_server", "sandbox1_server"): textwrap.dedent("""
            server:
              services:
                log:
                  level: "DEBUG"
            common:
              unified_agent:
                %(unified_agent)s
        """ % dict(unified_agent=unified_agent.ua_sockets_config_for_server)),
        "sandbox1_server": textwrap.dedent("""
            server:
              services:
                statistics_processor:
                  logbroker:
                    topic_dir: "/sandbox/preprod/statistics/topics"
                    consumer: "/sandbox/preprod/statistics/consumers/statistics_processor"
            common:
              unified_agent:
                %(unified_agent)s
        """ % dict(unified_agent=unified_agent.ua_sockets_config_for_server)),
    }], base=base.SandboxLauncher._CONFIG_PATCHES)

    # Port for service API
    api_port = None

    def packages(self):
        return ["services.tgz"]

    def user(self):
        return self._service_user._asdict()

    @property
    def binary_path(self):
        return os.path.join(self.get_pack("services.tgz"), "sandbox-services")

    def start(self):
        self._write_to_config("server.services.api", {"port": self.api_port})
        self._write_solomon_descr_to_config("server.solomon.push")
        self.change_permissions(self.binary_path, chmod=0o755)
        self.create(
            [self.binary_path, "run", "--name", self.__name__],
            env={"SANDBOX_CONFIG": self.config_as_file()}
        )

    def ping(self):
        # Perform log rotation. Don't notify, service will detect file changes automatically.
        self._rotate_logs(move=True, notify=False)
        # Don't forget default `ping` logic
        return super(Service, self).ping()

    def logfiles(self):
        return ["/var/log/sandbox/core/{}.log".format(self.__name__)]

    @property
    def _ROTATE_LOGS(self):
        return self.logfiles()


class MongoMonitor(Service):
    # Should match service name in sandbox.services.modules
    __name__ = "mongo_monitor"
    api_port = 9900

    def _mongod_instances(self):
        if self._key > 0:
            group = "sandbox1_mongo"
            ports = 10
        else:
            group = "sandbox_mongo"
            ports = 4

        for host in subprocess.check_output(["sky", "list", "k@" + group]).splitlines():
            for port in range(ports):
                yield "{}:{}".format(host, 37001 + port)

    def start(self):
        values = {
            "mongod": list(self._mongod_instances()),
            "mongos": "localhost:22222",
        }
        self._write_to_config("server.services.mongo_monitor", values)

        return super(MongoMonitor, self).start()


class TelegramBot(Service):
    __name__ = "telegram_bot"
    api_port = 9901


class StatisticsProcessor(Service):
    __name__ = "statistics_processor"
    api_port = 9902


class TasksStatistics(Service):
    __name__ = "tasks_statistics"
    api_port = 9903


class ClientSlotsMonitor(Service):
    __name__ = "client_slots_monitor"
    api_port = 9904


class ClientAvailabilityChecker(Service):
    __name__ = "client_availability_checker"
    api_port = 9905


class TaskStateSwitcher(Service):
    __name__ = "task_state_switcher"
    api_port = 9906


class Cleaner(Service):
    __name__ = "cleaner"
    api_port = 9907


class Scheduler(Service):
    __name__ = "scheduler"
    api_port = 9908


class GroupSynchronizer(Service):
    __name__ = "group_synchronizer"
    api_port = 9910


class ClientAvailabilityManager(Service):
    __name__ = "client_availability_manager"
    api_port = 9911


class HostsCacheUpdater(Service):
    __name__ = "hosts_cache_updater"
    api_port = 9912


class ClientProcessor(Service):
    __name__ = "client_processor"
    api_port = 9913


class TaskStatusChecker(Service):
    __name__ = "task_status_checker"
    api_port = 9914


class TaskStatusNotifier(Service):
    __name__ = "task_status_notifier"
    api_port = 9915


class Mailman(Service):
    __name__ = "mailman"
    api_port = 9916


class MessengerQBot(Service):
    __name__ = "messengerq_bot"
    api_port = 9917


class Juggler(Service):
    __name__ = "juggler"
    api_port = 9918


class MongoChecker(Service):
    __name__ = "mongo_checker"
    api_port = 9919


class CpuBalancer(Service):
    __name__ = "cpu_balancer"
    api_port = 9920


class AutoRestartTasks(Service):
    __name__ = "auto_restart_tasks"
    api_port = 9921


class UpdateRobotOwners(Service):
    __name__ = "update_robot_owners"
    api_port = 9922


class MDSCleaner(Service):
    __name__ = "mdscleaner"
    api_port = 9925


class NetworkMacrosResolver(Service):
    __name__ = "network_macros_resolver"
    api_port = 9926


class ResourcesStatistics(Service):
    __name__ = "resources_statistics"
    api_port = 9927


class MetricsReporter(Service):
    __name__ = "metrics_reporter"
    api_port = 9928


class BackupResources(Service):
    __name__ = "backup_resources"
    api_port = 9929


class ReplicateResources(Service):
    __name__ = "replicate_resources"
    api_port = 9930


class CleanResources(Service):
    __name__ = "clean_resources"
    api_port = 9931


class BackupResourcesToMds(Service):
    __name__ = "backup_resources_to_mds"
    api_port = 9932


class TaskTagsChecker(Service):
    __name__ = "task_tags_checker"
    api_port = 9933


class CopyResourcesToWarehouse(Service):
    __name__ = "copy_resources_to_warehouse"
    api_port = 9934


class CheckSemaphores(Service):
    __name__ = "check_semaphores"
    api_port = 9935


class TaskQueueValidator(Service):
    __name__ = "task_queue_validator"
    api_port = 9936


class TasksEnqueuer(Service):
    __name__ = "tasks_enqueuer"
    api_port = 9937


class MdsLruCleaner(Service):
    __name__ = "mds_lru_cleaner"
    api_port = 9938


class LogbrokerPublisher(Service):
    __name__ = "logbroker_publisher"
    api_port = 9939
