import os
import copy
import json
import logging
import requests

from . import base


class SolomonAgent(base.Base):
    """ Solomon agent for mac """
    SERVICE_CONFIG_NAME = "solomon_service.json"
    PORT = 30991
    PING_TIMEOUT = 15

    _CONFIG = {
        "HttpServer": {
            "BindAddress": "::",
            "BindPort": PORT,
            "MaxConnections": 100,
            "OutputBufferSize": 8192
        },

        "ConfigLoader": {
            "FileLoader": {
                "UpdateInterval": "30s",
                "ConfigsFiles": [
                    "{srv}/" + SERVICE_CONFIG_NAME
                ]
            }
        }
    }

    _SOLOMON_SERVICE_CONF = {
        "Project": "",
        "Service": "",

        "PullInterval": "15s",

        "Modules": [
            {
                "System": {
                    "Cpu": "ADVANCED",
                    "Memory": "ADVANCED",
                    "Network": "ADVANCED",
                    "Storage": "ADVANCED",
                    "Io": "ADVANCED",
                    "Kernel": "ADVANCED"
                }
            }
        ]
    }

    __name__ = __servant_name__ = "solomon_agent"
    __cfg_fmt__ = "json"

    def __init__(self, key, host_platform, groups, layout, dc, solomon_descr, tvminfo=None):
        self._key = key
        self._host_platform = host_platform
        self._special_groups = groups
        self._layout = layout
        self._solomon_descr = solomon_descr
        self._tvminfo = tvminfo
        self._clsname = self.__class__.__name__
        self._modname = self.__class__.__module__
        self.__cfg__ = self._CONFIG
        self._solomon_service_config = copy.deepcopy(self._SOLOMON_SERVICE_CONF)
        self._solomon_service_config["Project"] = self._solomon_descr["project"]
        self._solomon_service_config["Service"] = "_".join((self._solomon_descr["project"], "mac", str(self._key % 2)))

    def ping(self):
        try:
            requests.get(self.sensors_url(), timeout=self.PING_TIMEOUT).raise_for_status()
        except:
            return False
        return True

    @classmethod
    def packages(self):
        return [
            {
                "type": "naked",
                "source": "sbr:SANDBOX_SOLOMON_AGENT_BINARY",
                "sb_resource_filter": {
                    "attrs": {
                        "released": "stable"
                    },
                    "owner": "SANDBOX"
                },
                "alias": "solomon_binary",
                "unpack_it": False,
            },
        ]

    @property
    def solomon_config_path(self):
        return os.path.join(self.substval("{srv}/sandbox.json"))

    def sensors_url(self):
        return "http://[::1]:{}/storage/read?project={}&service={}".format(
            self.PORT, self._solomon_service_config["Project"], self._solomon_service_config["Service"]
        )

    def postinstall(self):
        self.writetofile(json.dumps(self._solomon_service_config, indent=4), "{srv}/" + self.SERVICE_CONFIG_NAME)
        self._make_symlink(self.config_as_file(), self.solomon_config_path, force=True, ignore_errors=True)

    def user(self):
        return self._service_user._asdict()

    def start(self):
        path = os.path.join(self.get_package("solomon_binary"), "solomon-agent")
        self.change_permissions(path, chmod=0o755)
        logging.warning("Found package %s with solomon binary agent.", path)
        cmd = [
            path,
            "--config",
            self.solomon_config_path
        ]
        self.create(cmd, cwd="{srv}")
