import os
import socket
import logging
import textwrap
import subprocess

from . import base


class STEP(base.Base):
    __name__ = "step"

    _SENTINEL_HOST = "redis.sandbox.yandex-team.ru"
    _SENTINEL_PORT = 26379
    _SENTINEL_NAMESPACE = "redis_prod"

    _REDIS_DB_BROKER = {
        0: 8,
        1: 0
    }
    _REDIS_DB_RESULT = {
        0: 9,
        1: 1
    }
    _REDIS_DB_BEATCOP = {
        0: 10,
        1: 2
    }

    _MONGO_DB_NAME = "events2"
    _MONGO_HOST = "localhost"
    _MONGO_PORT = 22222

    _HTTP_PORT = 9989

    _LOG_DIR = "/var/log/sandbox/step"
    _UWSGI_LOG_DIR = os.path.join(_LOG_DIR, "uwsgi")
    _RUN_DIR = "/opt/sandbox/run/step"

    _BEATCOP_CONF_FILE = "statinfra_api_celerybeat_beatcop.ini"
    _UWSGI_CONF_FILE = "statinfra_api_uwsgi.ini"

    _STATBOX_LOG_DIR = "/var/log/statbox"
    _STATBOX_MASTER_ADDR = {
        0: "logbroker.yandex.net",
        1: "logbroker-pre.yandex.net"
    }
    _STATBOX_TVM_CLIENT_ID = {
        0: 2002826,
        1: 2011100,
    }
    _STATBOX_TVM_SERVER_ID = {
        0: 2001059,
        1: 2001147,
    }
    _STATBOX_PUSH_CLIENT_PID_FILE = "/var/run/statbox/push-client.pid"

    # ############### #

    _UWSGI_LOGFILE = os.path.join(_UWSGI_LOG_DIR, "uwsgi.log")
    _UWSGI_REQ_LOGFILE = os.path.join(_UWSGI_LOG_DIR, "uwsgi-req.log")

    _UWSGI_PID = os.path.join(_RUN_DIR, "uwsgi.pid")
    _CELERY_PID = os.path.join(_RUN_DIR, "celery.pid")
    _BEATCOP_PID = os.path.join(_RUN_DIR, "beatcop.pid")

    _BEATCOP_SCHEDULE = os.path.join(_RUN_DIR, "celerybeat-schedule")

    _STEP_PKG = "step.tgz"
    _STEP_VENV_PKG = "venv_step.tgz"

    _UWSGI_CONFIG = textwrap.dedent("""
        [uwsgi]

        autoload = true
        pidfile = {pidfile}

        uid = zomb-sandbox
        http = {socket}
        chmod-socket = 660
        chown-socket = {user}

        procname-prefix-spaced = [STEP]

        master = true  # ??

        listen = 1024

        cheaper = 16
        cheaper-initial = 4
        cheaper-step = 4
        workers = 32

        module = sandbox.step.statinfra_api.wsgi
        env = DJANGO_SETTINGS_MODULE=sandbox.step.statinfra_api.settings
        virtualenv = {venv}
        static-map = /api/static={static_dir}

        die-on-term = true
        buffer-size = 16384

        req-logger = file:{req_logfile}
        logger = file:{logfile}
    """)

    _RSYSLOG_CONFIG = textwrap.dedent(r"""
        # Syslog config for logging from the statinfra-api

        # WARN: partially-global
        $FileCreateMode 0644
        $EscapeControlCharactersOnReceive off

        $template LogFormatSIA,"%msg:2:$%\n"
        $template LogDynFileSIA,"/var/log/%syslogtag:R,ERE,1,DFLT:file__(sandbox/step/[a-zA-Z0-9_/-]*)--end%.log"

        # NOTE: this can be generalised to multiple projects.
        # The root directory creation will have to be done outside syslog though,
        # as syslog normally doesn't have access to `mkdir /var/log/something`.
        :syslogtag, regex, "file__sandbox/step/[a-zA-Z0-9_/-]*" ?LogDynFileSIA;LogFormatSIA
        # 'skip other handling of the matched message'
        & ~

    """)

    _LOGROTATE_CONFIG = textwrap.dedent("""
        {log_dir}/*.log {{
            daily
            rotate 31

            compress
            delaycompress

            notifempty
            missingok

            postrotate
                service rsyslog reload
            endscript
        }}

        {uwsgi_log_dir}/*.log {{
            copytruncate

            daily
            rotate 7

            compress

            notifempty
            missingok
        }}
    """)

    _STATBOX_PUSH_CLIENT_CONFIG = textwrap.dedent("""
        ---
        ident: step

        logger:
            mode: [file, stderr]
            file: {log_dir}/watcher.log
            level: 7

        watcher:
            state: /var/lib/push-client

        network:
            proto: pq
            master-addr: {master_addr}
            master-port: 2135
            tvm-client-id: {tvm_client_id}
            tvm-server-id: {tvm_server_id}
            tvm-secret-file: /etc/yandex/statbox-push-client/.tvm_secret
        files:
            - name: {step_log_dir}/django.log
              log_type: statinfra-event-processor-django-log
            - name: {step_log_dir}/celery.log
              log_type: statinfra-event-processor-celery-log
    """)

    _STATBOX_PUSH_CLIENT_LOGROTATE_CONFIG = textwrap.dedent("""
    {log_dir}/watcher.log
    {{
        rotate 1
        compress
        missingok
        daily
        size 1000M
        sharedscripts
        prerotate
            ( test -x /etc/init.d/statbox-push-client && /etc/init.d/statbox-push-client stop >/dev/null ) || exit 0
        endscript
        postrotate
            ( test -x /etc/init.d/statbox-push-client && /etc/init.d/statbox-push-client start >/dev/null ) || exit 0
        endscript
    }}
    """)

    def packages(self):
        return [self._STEP_PKG, self._STEP_VENV_PKG]

    def user(self):
        return self._service_user.name

    def group(self):
        return self._service_user.group

    def balancer(self):
        return {
            "domain": "step",
            "location": "/",
            "port": self._HTTP_PORT
        }

    @staticmethod
    def logfiles():
        """ List of logs sends to Elasticsearch """
        return [
            STEP._UWSGI_LOGFILE,
            STEP._UWSGI_REQ_LOGFILE,

            # Django log files. Defined in "070-logging.conf"
            os.path.join(STEP._LOG_DIR, "debug.log"),
            os.path.join(STEP._LOG_DIR, "error_requests.log"),
            os.path.join(STEP._LOG_DIR, "django.log"),
            os.path.join(STEP._LOG_DIR, "celery.log"),
        ]

    def step_django_settings(self, conf):
        return os.path.join(
            self.get_pack(self._STEP_PKG),
            "sandbox",
            "step",
            "statinfra_api",
            "settings",
            conf
        )

    def step_etc(self, conf):
        return os.path.join(
            self.get_pack(self._STEP_PKG),
            "sandbox",
            "step",
            "etc",
            "statinfra_api",
            conf
        )

    def _venv_bin(self, executable=''):
        return os.path.join(self.get_pack(self._STEP_VENV_PKG), "bin", executable)

    def write_django_config(self, fname, *args, **kwargs):
        self._write_config(self.step_django_settings(fname), 0o644, self.user(), *args, **kwargs)

    def write_etc_config(self, fname, *args, **kwargs):
        self._write_config(self.step_etc(fname), 0o644, self.user(), *args, **kwargs)

    def postinstall(self):
        for path in (self._LOG_DIR, self._RUN_DIR, self._UWSGI_LOG_DIR, self._STATBOX_LOG_DIR):
            if not os.path.exists(path):
                os.makedirs(path)
            user = "syslog" if path == self._LOG_DIR else self.user()
            self.change_permissions(path, chmod=0o755, chown=user)
        self.change_permissions(self._STATBOX_LOG_DIR, chmod=0o775)

        # rsyslog config
        self._write_config(
            "/etc/rsyslog.d/42-step.conf",
            0o644,
            "root",
            self._RSYSLOG_CONFIG,
            owner=self.user(),
            group=self.group()
        )

        # restart rsyslog
        p = subprocess.Popen(["service", "rsyslog", "restart"], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        if not base.check_subprocess_status(p):
            logging.info("Successfully restart rsyslog")

        # logrotate config
        self._write_config(
            "/etc/logrotate.d/step.logrotate",
            0o644,
            "root",
            self._LOGROTATE_CONFIG,
            log_dir=self._LOG_DIR,
            uwsgi_log_dir=self._UWSGI_LOG_DIR
        )

        # statbox push client config
        config_dir = "/etc/yandex/statbox-push-client"
        if not os.path.exists(config_dir):
            os.makedirs(config_dir)
        self._write_config(
            os.path.join(config_dir, "push-client.yaml"),
            0o644,
            "root",
            self._STATBOX_PUSH_CLIENT_CONFIG,
            log_dir=self._STATBOX_LOG_DIR,
            step_log_dir=self._LOG_DIR,
            master_addr=self._STATBOX_MASTER_ADDR[self._key],
            tvm_client_id=self._STATBOX_TVM_CLIENT_ID[self._key],
            tvm_server_id=self._STATBOX_TVM_SERVER_ID[self._key],
        )

        # statbox push client logrotate config
        self._write_config(
            "/etc/logrotate.d/statbox-push-client.logrotate",
            0o644,
            "root",
            self._STATBOX_PUSH_CLIENT_LOGROTATE_CONFIG,
            log_dir=self._STATBOX_LOG_DIR
        )

        self._restart_service("statbox-push-client")

        # Run Django migrations - init DB
        p = subprocess.Popen(
            [
                "/usr/bin/env", "sudo", "-u", self.user(),
                self._venv_bin("python"),
                os.path.join(self.get_pack(self._STEP_PKG), "sandbox", "step", "statinfra_api", "manage.py"),
                "migrate"
            ],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            env={"PYTHONPATH": self.get_pack(self._STEP_PKG)}
        )
        if not base.check_subprocess_status(p):
            logging.info("Successfully mirgate DB")

        # Collect static files
        p = subprocess.Popen(
            [
                "/usr/bin/env", "sudo", "-u", self.user(),
                self._venv_bin("python"),
                os.path.join(self.get_pack(self._STEP_PKG), "sandbox", "step", "statinfra_api", "manage.py"),
                "collectstatic", "--no-input"
            ],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            env={"PYTHONPATH": self.get_pack(self._STEP_PKG)}
        )
        if not base.check_subprocess_status(p):
            logging.info("Successfully collect static")

    def start(self):
        # Django settings
        self.write_django_config(
            "010-deploy.conf",
            textwrap.dedent("""
                SENTINEL_HOST="{}"
                SENTINEL_PORT={}
                SENTINEL_NAMESPACE="{}"

                CELERY_BROKER_DB={}
                CELERY_RESULT_DB={}

                MONGO_DB_NAME="{}"
                MONGO_HOST="{}"
                MONGO_PORT={}

                SQLITE_DB_PATH="{}"
            """),
            self._SENTINEL_HOST,
            self._SENTINEL_PORT,
            self._SENTINEL_NAMESPACE,
            self._REDIS_DB_BROKER[self._key],
            self._REDIS_DB_RESULT[self._key],
            self._MONGO_DB_NAME,
            self._MONGO_HOST,
            self._MONGO_PORT,
            os.path.join(self.data_dir(), "step.sqlite.db")
        )

        # uWSGI settings
        self.write_etc_config(
            self._UWSGI_CONF_FILE, self._UWSGI_CONFIG,
            pidfile=self._UWSGI_PID,
            socket="[::]:{}".format(self._HTTP_PORT),
            user=self.user(),
            venv=self.get_pack(self._STEP_VENV_PKG),
            static_dir=os.path.join(
                self.get_pack(self._STEP_VENV_PKG),
                "lib", "python2.7", "site-packages", "rest_framework", "static"
            ),
            req_logfile=self._UWSGI_REQ_LOGFILE,
            logfile=self._UWSGI_LOGFILE
        )

        self._make_symlink(
            self.get_pack(self._STEP_VENV_PKG),
            os.path.expanduser("~{}/step_venv".format(self.user())),
            force=True
        )

        for pid_fname in (self._CELERY_PID, self._BEATCOP_PID):
            if os.path.exists(pid_fname):
                os.unlink(pid_fname)

        self.create([
            self._venv_bin("python"),
            self._venv_bin("celery"),
            "--app=sandbox.step.statinfra_api.celery:app",
            "worker",
            "-l", "debug",
            "--pidfile={}".format(self._CELERY_PID),
        ], env={"PYTHONPATH": self.get_pack(self._STEP_PKG)})

        self.create([
            self._venv_bin("python"),
            os.path.join(self.get_pack(self._STEP_PKG), "sandbox", "step", "beatcop", "beatcop.py"),
            self._venv_bin("python"),
            self._venv_bin("celery"),
            "--app=sandbox.step.statinfra_api.celery:app",
            "beat",
            "-l", "debug",
            "--pidfile={}".format(self._BEATCOP_PID),
            "--schedule={}".format(self._BEATCOP_SCHEDULE),
        ], env={
            "SINGLE_BEAT_IDENTIFIER": "celerybeat",
            "SENTINEL_HOST": self._SENTINEL_HOST,
            "SENTINEL_PORT": str(self._SENTINEL_PORT),
            "SENTINEL_NAMESPACE": self._SENTINEL_NAMESPACE,
            "REDIS_DB_BEATCOP": str(self._REDIS_DB_BEATCOP[self._key]),
            "PYTHONPATH": self.get_pack(self._STEP_PKG)
        })

        self.create([
            self._venv_bin("uwsgi"),
            self.step_etc(self._UWSGI_CONF_FILE)
        ], env={"PYTHONPATH": self.get_pack(self._STEP_PKG)})

        # Dirty HACK
        for fname in (self._UWSGI_LOGFILE, self._UWSGI_REQ_LOGFILE):
            # touch
            with open(fname, 'a'):
                pass
            self.change_permissions(fname, chmod=0o644, chown=self.user())

    def ping(self):
        return self._check_service("statbox-push-client", self._STATBOX_PUSH_CLIENT_PID_FILE)


class RemToStep(base.Base):
    __name__ = "rem_to_step"

    _CONFIG = textwrap.dedent("""
        rem_to_step:
          distributed:
            node_id: {node_id}
            nodes: {nodes}
            state_filename: /opt/sandbox/run/rem_to_step.state
          log:
            root: /var/log/sandbox
          step:
            token: /home/zomb-sandbox/.rem2step/token
    """)

    _CONFIG_PATCHES = base.unfold_list([{
        "sandbox1_server": textwrap.dedent("""
            rem_to_step:
              step:
                url: https://step-sandbox1.n.yandex-team.ru/api/v1
        """),
        "sandbox_server": textwrap.dedent("""
            rem_to_step:
              step:
                url: https://step.sandbox.yandex-team.ru/api/v1
        """),
    }])

    @property
    def __cfg__(self):
        if self._config is not None:
            return self._config
        from library.sky import hostresolver
        ctag = base.ctag("sandbox_zk" if self._key == 0 else "sandbox1_server")
        self._CONFIG = self._CONFIG.format(
            node_id=self.__node_id(socket.getfqdn()),
            nodes=[
                {"id": self.__node_id(h), "host": h, "port": 11111}
                for h in map(str, hostresolver.Resolver().resolveHosts(ctag))
            ]
        )
        return super(RemToStep, self).__cfg__

    @staticmethod
    def __node_id(host):
        node_id = int(filter(str.isdigit, host) or 0)
        assert node_id
        return node_id

    def packages(self):
        return ["linux_rem_to_step.tgz"]

    def user(self):
        return self._service_user._asdict()

    def start(self):
        binary_path = os.path.join(self.get_pack("linux_rem_to_step.tgz"), "rem_to_step")
        self.change_permissions(binary_path, chmod=0o755)
        self.create([binary_path], env={"SANDBOX_CONFIG": self.config_as_file()})

    @staticmethod
    def logfiles():
        return ["/var/log/sandbox/rem_to_step.log"]
