import os
import pwd
import grp
import stat
import time
import shlex
import base64
import random
import shutil
import socket
import signal
import logging
import tarfile
import textwrap
import platform
import datetime as dt
import itertools as it
import subprocess as sp
import collections

import psutil

from . import base

CPUInfo = collections.namedtuple("CPUInfo", "physical_id core_id processor")
CPUSET_CGROUP_PATH = "/sys/fs/cgroup/cpuset"


def get_address(fqdn):
    return socket.getaddrinfo(fqdn, 0, socket.AF_INET6)[0][-1][0]


def on_linux():
    return os.uname()[0] == "Linux" and "Microsoft" not in platform.platform()


def on_wsl():
    return "Microsoft" in platform.platform()


def on_macos():
    return "Darwin" in platform.platform()


class SystemConfig(object):
    __metaclass__ = base.ServantMeta
    __name__ = "system_config"

    NETWORK_RELOAD_PERIOD = dt.timedelta(days=1)
    NETWORK_ROUTES_UPDATE_PERIOD = dt.timedelta(minutes=5)

    # {name: <number of CPU cores>}
    CPU_SETS = {
        "serviceq": 2,
    }

    # SANDBOX-2651  ulimits for zomb-sandbox, sandbox users.
    _CONF_FILENAME_TEMPLATE = "/etc/security/limits.d/{}.conf"
    _ULIMITS = {
        "zomb-sandbox": textwrap.dedent("""
            zomb-sandbox          soft    memlock          100000
            zomb-sandbox          hard    memlock          100000
            zomb-sandbox          soft    nofile           1048576
            zomb-sandbox          hard    nofile           1048576
            zomb-sandbox          soft    core             unlimited
        """),
        "sandbox": textwrap.dedent("""
            sandbox          soft    memlock          {0}
            sandbox          hard    memlock          {0}
            sandbox          soft    nofile           1048576
            sandbox          hard    nofile           1048576
            sandbox          soft    core             unlimited
        """),
        "root": textwrap.dedent("""
            root          soft    memlock          {0}
            root          hard    memlock          {0}
            root          soft    nofile           1048576
            root          hard    nofile           1048576
            root          soft    core             unlimited
        """)
    }

    _CGROUP_CONFIG = textwrap.dedent("""
        mount {
            cpu = /sys/fs/cgroup/cpu;
            cpuset = /sys/fs/cgroup/cpuset;
            cpuacct = /sys/fs/cgroup/cpu;
            devices = /sys/fs/cgroup/devices;
            memory = /sys/fs/cgroup/memory;
            blkio = /sys/fs/cgroup/blkio;
            freezer = /sys/fs/cgroup/freezer;
        }
    """)

    _CGROUP_REJECTS = [
        "yandex-porto",
        "yandex-search-porto",
        "qloud-porto-router"
    ]

    _CGROUP_PACKAGES = [
        "cgroup-bin",
        "cgroup-lite"
    ]

    _CGROUP_FLAG_FILE = "/var/local/sandbox_cgroup_flag"
    _CGROUP_CONFIG_FILE = "/etc/cgconfig.conf"

    _MEMLOCK = {
        "sandbox": {
            "default": 12582912,
            "rty": 42949672960
        },
        "root": {
            "default": 12582912,
            "rty": 42949672960
        },
    }
    _MEMBORDER = 80  # SANDBOX-2651 uses custom ulimits when RAM is above

    _CONF_SYSCTL_FILENAME = "/etc/sysctl.d/61-net.conf"

    _REDIS_FILENAME = {
        "redis-sentinel": "/etc/redis/sentinel.conf",
        "redis-server": "/etc/redis/redis.conf"
    }

    DEV_KVM = "/dev/kvm"

    _REDIS_MASTER = "sandbox-server07.search.yandex.net"

    class _REDIS_CONF(object):
        class __metaclass__(type):
            __TEMPLATES = {
                "redis-sentinel": textwrap.dedent("""
                    daemonize yes
                    port 26379
                    logfile "/var/log/redis/sentinel.log"
                    sentinel monitor redis_prod {} 6379 1
                    sentinel down-after-milliseconds redis_prod 6000
                    sentinel failover-timeout redis_prod 60000
                    sentinel config-epoch redis_prod 432
                """),
                "redis-server": textwrap.dedent("""
                    daemonize yes
                    pidfile /var/run/redis/redis-server.pid
                    port 6379
                    unixsocket /tmp/redis.sock
                    unixsocketperm 777
                    timeout 0
                    tcp-keepalive 0
                    loglevel notice
                    logfile /var/log/redis/redis.log
                    databases 16
                    save ""
                    stop-writes-on-bgsave-error yes
                    rdbcompression yes
                    rdbchecksum yes
                    dbfilename redis.rdb
                    dir /var/lib/redis/
                    slave-serve-stale-data no
                    slave-read-only yes
                    repl-disable-tcp-nodelay no
                    slave-priority 100
                    #requirepass pass_here
                    maxmemory 32gb
                    maxmemory-policy volatile-lru
                    appendonly no
                    appendfsync everysec
                    no-appendfsync-on-rewrite no
                    auto-aof-rewrite-percentage 100
                    auto-aof-rewrite-min-size 512mb
                    lua-time-limit 5000
                    slowlog-log-slower-than 10000
                    slowlog-max-len 128
                    notify-keyspace-events ""
                    hash-max-ziplist-entries 512
                    hash-max-ziplist-value 64
                    list-max-ziplist-entries 512
                    list-max-ziplist-value 64
                    set-max-intset-entries 512
                    zset-max-ziplist-entries 128
                    zset-max-ziplist-value 64
                    activerehashing yes
                    client-output-buffer-limit normal 0 0 0
                    client-output-buffer-limit slave 256mb 64mb 60
                    client-output-buffer-limit pubsub 32mb 8mb 60
                    hz 10
                    aof-rewrite-incremental-fsync yes
                """)
            }

            def __getitem__(self, item):
                return self.__TEMPLATES[item].format(get_address(SystemConfig._REDIS_MASTER))

    _JUGGLER_CONFIG = textwrap.dedent("""
        [client]
        check_bundles=ps_search2,wall-e-checks-bundle,sandbox-checks-bundle
    """).strip()

    def __init__(self, _, host_platform, special_groups, layout, *__):
        self.groups = special_groups
        self.layout = layout

    def additional_users(self):
        if base.check_tags(self.groups, base.YP_LITE_TAGS):
            return {
                "zomb-sandbox": dict(uid=29684, gid=21710, group="dpt_virtual_robot", create_home=True),
                "sandbox": dict(uid=3831, gid=3831, group="sandbox", create_home=True),
                "hw-watcher": dict(uid=135, gid=135, group="hw-watcher", create_home=True),
                "statbox": dict(uid=10000, create_home=False),
            }
        return {"statbox": dict(uid=10000, create_home=False)}

    @staticmethod
    def totalmem():
        try:
            with open("/proc/meminfo") as fh:
                return next(int(l.split()[1]) << 10 for l in fh if l.startswith("MemTotal:"))
        except Exception:
            return None

    def lxc_configure(self):
        with open(self._CONF_SYSCTL_FILENAME, "w") as f:
            f.write(self._CONF_SYSCTL_FILENAME)
        sp.call(["/sbin/sysctl", "-p", self._CONF_SYSCTL_FILENAME])

    def pkg_is_installed(self, pkgname):
        """
        :param pkgname: str with package name without version
        :return: bool
        """
        try:
            return "install ok installed" in sp.check_output(["dpkg-query", "-Wf'${Status}'", pkgname])
        except sp.CalledProcessError:
            return False

    def apt_get(self, package, install):
        try:
            sp.check_call(["apt-get", "--yes", "--force-yes", "install" if install else "remove", package])
            return True
        except sp.CalledProcessError:
            return False

    def cgroup_check_install(self, cgroup_pkgs, reject_pkgs, c_tags):
        """
        :param cgroup_pkgs: list of packages with cgroups
        :param reject_pkgs: LIst of conflict pkgs
        :param c_tags: tags of current host
        :return: True if pkg installd, False if cgroups can't be installed.
        """

        # Filter hosts with client tag only
        if base.check_tags(
            c_tags, undesired_tags={"sandbox_client", "sandbox1_client", "yp_lite@sandbox2", "yp_lite@sandbox3"}
        ):
            return False
        # Filter hosts with conflicts installed
        for name in reject_pkgs:
            if self.pkg_is_installed(name):
                return False

        if platform.linux_distribution()[1] == "12.04":
            try:
                # only cgroup-bin will suffice
                if self.pkg_is_installed("cgroup-bin"):
                    return True
                if self.pkg_is_installed("cgroup-lite"):
                    self.apt_get("cgroup-lite", install=False)
                return self.apt_get("cgroup-bin", install=True)
            finally:
                if not os.path.exists("/sys/fs/cgroup/cpuacct"):
                    os.symlink("cpu", "/sys/fs/cgroup/cpuacct")
        else:
            for name in cgroup_pkgs:
                if self.pkg_is_installed(name):
                    return True
            # if no rejects packages are installed, install cgroup-lite:
            return self.apt_get("cgroup-lite", install=True)

    def compare_redis_configs(self, redis_conf, redis_conf_on_system):
        """
        :param redis_conf: is list, config file content
        :param redis_conf_on_system: is string, full path to config file on system
        :return: True if there are all strings from etalon config in the config on system
        """
        redis_etalon = redis_conf
        with open(redis_conf_on_system, "r") as f:
            redis_real = f.readlines()
        redis_real = redis_real[:len(redis_etalon)]
        return set(redis_etalon) == set(redis_real)

    def fetch_redis_conf(self, redis_conf, suffix=""):
        fname = self._REDIS_FILENAME[redis_conf]
        content = self._REDIS_CONF[redis_conf] + suffix
        if not os.path.exists(fname) or not self.compare_redis_configs(content, fname):
            with open(fname, "w") as f:
                f.write(content)
            os.chown(fname, pwd.getpwnam("redis").pw_uid, grp.getgrnam("root").gr_gid)

    def _unpack_layout(self):
        if not self.layout:
            return
        layout_path = self.raw_data_as_file(base64.b64decode(self.layout), "tar.bz2", chmod=0o400)
        layout_root = "/"
        layout_tar = tarfile.open(layout_path)

        now = time.time()
        rename_mapping = []
        user_resolver = base.CachingResolver(lambda _: pwd.getpwnam(_).pw_uid)
        group_resolver = base.CachingResolver(lambda _: grp.getgrnam(_).gr_gid)

        def members():
            for tarinfo in layout_tar:
                src = os.path.join(layout_root, tarinfo.name)
                tarinfo.mtime = int(time.time())
                try:
                    tarinfo.uid = user_resolver[tarinfo.uname]
                    tarinfo.gid = group_resolver[tarinfo.gname or tarinfo.uname]
                except KeyError as ex:
                    raise RuntimeError("Unable to resolve user/group for file {!r} owned by {!r}.{!r}: {}".format(
                        src, tarinfo.uname, tarinfo.gname, ex
                    ))
                try:
                    st = os.stat(src)
                    if stat.S_ISREG(st.st_mode):
                        tarinfo.name = ".".join((tarinfo.name, "new", str(now)))
                        rename_mapping.append((os.path.join(layout_root, tarinfo.name), src))
                        yield tarinfo
                    elif (
                        (st.st_mode & 0o777) != (tarinfo.mode & 0o777) or
                        st.st_uid != tarinfo.uid or st.st_gid != tarinfo.gid
                    ):
                        logging.info(
                            "Changing %r owner to %r.%r/%r.%r (was %r.%r), mode 0%o (was 0%o)",
                            src, tarinfo.uname, tarinfo.gname, tarinfo.uid, tarinfo.gid,
                            st.st_uid, st.st_gid, tarinfo.mode, st.st_mode
                        )
                        os.chown(src, tarinfo.uid, tarinfo.gid)
                        os.chmod(src, tarinfo.mode)
                except OSError:
                    yield tarinfo

        layout_tar.extractall(layout_root, members())
        for src, dst in rename_mapping:
            logging.debug("Renaming %r to %r", src, dst)
            os.rename(src, dst)

    def patch_juggler_config(self):
        # JUGGLERSUPPORT-908: patch Juggler config for installing custom checks bundles
        juggler_conf_path = "/home/monitor/juggler/etc/sandbox.conf"
        try:
            dirname = os.path.dirname(juggler_conf_path)
            os.makedirs(dirname)
            os.chown(dirname, pwd.getpwnam("monitor").pw_uid, grp.getgrnam("monitor").gr_gid)
            os.chmod(dirname, 0o775)
        except OSError:
            pass

        if os.path.exists(juggler_conf_path):
            with open(juggler_conf_path, "r") as config:
                if config.read().strip() == self._JUGGLER_CONFIG:
                    return

        try:
            with open(juggler_conf_path, "w") as config:
                config.write(self._JUGGLER_CONFIG)
        except Exception as exc:
            logging.error("Failed to update Sandbox-specific Juggler config: %s", exc)
        else:
            logging.debug("juggler-client config updated, restarting the service")
            sp.call(["service", "juggler-client", "restart"], close_fds=True)

    @staticmethod
    def _cpuinfo():
        cpuinfo = []
        with open("/proc/cpuinfo") as f:
            while True:
                physical_id = None
                core_id = None
                processor = None
                for line in f:
                    if not line.strip():
                        break
                    key, value = map(str.strip, line.split(":", 2))
                    if key == "processor":
                        processor = int(value)
                    elif key == "physical id":
                        physical_id = int(value)
                    elif key == "core id":
                        core_id = int(value)
                else:
                    break
                cpuinfo.append(CPUInfo(physical_id, core_id, processor))
        return sorted(cpuinfo)

    @staticmethod
    def _make_cpu_intervals(cpus):
        """
        Converts list of CPU numbers to string with comma separated continuous intervals
        Example: 1,2,4,6,7,8 -> 1-2,4,6-8
        :param cpus: list of CPU numbers
        """
        intervals = []
        for _, group in it.groupby(enumerate(sorted(cpus)), lambda i: i[1] - i[0]):
            group = list(group)
            b, e = group[0][1], group[-1][1]
            intervals.append(str(b) if b == e else "{}-{}".format(b, e))
        return ",".join(intervals)

    def _create_cpuset(self, cpuset, cpus):
        cpuset_path = os.path.join(CPUSET_CGROUP_PATH, cpuset)
        if not os.path.exists(cpuset_path):
            os.mkdir(cpuset_path)
            with open(os.path.join(cpuset_path, "cpuset.cpus"), "w") as f:
                f.write(self._make_cpu_intervals(cpus))
            with open(os.path.join(CPUSET_CGROUP_PATH, "cpuset.mems")) as f:
                cpuset_mems = f.read()
            with open(os.path.join(cpuset_path, "cpuset.mems"), "w") as f:
                f.write(cpuset_mems)
        return cpuset_path

    def _create_cpusets(self):
        cpuinfo = self._cpuinfo()
        for name, cores in self.CPU_SETS.items():
            cpus = [item.processor for item in cpuinfo[-cores:]]
            self._create_cpuset(name, cpus)
            cpuinfo = cpuinfo[:-cores]

        cpus = [item.processor for item in cpuinfo]
        default_cpuset_path = self._create_cpuset("default", cpus)

        # move all processes to default cpu set
        all_tasks_path = os.path.join(CPUSET_CGROUP_PATH, "tasks")
        default_tasks_path = os.path.join(default_cpuset_path, "tasks")
        with open(all_tasks_path) as f:
            pids = f.readlines()
        for pid in pids:
            try:
                os.readlink("/proc/{}/exe".format(pid.strip()))
                with open(default_tasks_path, "w") as default_tasks:
                    default_tasks.write(pid)
            except (OSError, IOError):
                pass

    def _onetime_linux_tunings(self):
        """ System tunings which should be performed only one time in Linux host's life. """
        host_conductortags = set(self.groups)
        host = socket.getfqdn()

        # create ulimits
        ram = self.totalmem()
        for user, content in self._ULIMITS.iteritems():
            filename = self._CONF_FILENAME_TEMPLATE.format(user)
            filename_tmp = filename + "~"
            with open(filename_tmp, "w") as f:
                memlock = self._MEMLOCK.get(user)
                if memlock:
                    if ram >= self._MEMBORDER * 2 ** 30:
                        content = content.format(memlock["rty"])
                    else:
                        content = content.format(memlock["default"])
                f.write(content)
            os.rename(filename_tmp, filename)

        # setup timezone and ntpd
        sp.check_call(["ln", "-sf", "/usr/share/zoneinfo/Europe/Moscow", "/etc/localtime"])
        if platform.linux_distribution()[1] == "16.04" and base.check_tags(
            self.groups, undesired_tags=base.YP_LITE_TAGS
        ):
            # disable `timesyncd`, we'll be using `ntpd` because of juggler
            sp.check_call(["timedatectl", "set-ntp", "no"])
        if not self.pkg_is_installed("ntp"):
            self.apt_get("ntp", install=True)

        # disable system atop
        sp.call(["/usr/sbin/update-rc.d", "atop", "disable"])
        sp.call(["service", "atop", "stop"])
        # https://ml.yandex-team.ru/thread/2370000002974325314/
        sp.call(["/usr/sbin/update-ca-certificates", "--fresh"])

        # Ensure `kvm` group exists and `sandbox` user in it
        if os.path.exists(self.DEV_KVM) and grp.getgrgid(os.stat(self.DEV_KVM).st_gid).gr_name != "kvm":
            gids = sorted(_.gr_gid for _ in grp.getgrall() if 99 < _.gr_gid < 1000)
            readahead = iter(gids)
            next(readahead)
            gid = next(a for a, b in it.izip(gids, readahead) if a + 1 != b) + 1
            sp.call(["groupadd", "-g", str(gid), "kvm"])
            sp.call(["usermod", "-aG", "kvm", "sandbox"])
            sp.call(["chown", "root:kvm", self.DEV_KVM])
            sp.call(["chmod", "660", self.DEV_KVM])

        if host_conductortags & {"sandbox1_multios", "sandbox_multios"}:
            sp.call(["sysctl", "-p", self._CONF_SYSCTL_FILENAME])

        if host_conductortags & {"sandbox_stg", "sandbox_server"}:
            sp.call(["/usr/sbin/update-rc.d", "ya-slb-tun", "defaults"])

        if "sandbox_zk" in host_conductortags:
            # Redis
            sp.call(["/usr/sbin/update-rc.d", "redis-server", "defaults"])
            sp.call(["/usr/sbin/update-rc.d", "redis-sentinel", "defaults"])
            sp.call(["service", "redis-server", "start"])
            sp.call(["service", "redis-sentinel", "start"])
            if host == self._REDIS_MASTER:
                for key in self._REDIS_FILENAME.keys():
                    self.fetch_redis_conf(key)
                    sp.call(["/usr/sbin/invoke-rc.d", key, "restart"])
            else:
                self.fetch_redis_conf("redis-server", "\nslaveof {} 6379\n".format(self._REDIS_MASTER))
                self.fetch_redis_conf("redis-sentinel")
            # Zookeeper
            zid = int(host.split(".")[0][-2:])
            if not os.path.exists("/etc/zookeeper/myid"):
                with open("/etc/zookeeper/myid", "w") as fh:
                    fh.write(str(zid) + "\n")
                with open("/etc/zookeeper/conf/myid", "w") as fh:
                    fh.write(str(zid) + "\n")
                uid = pwd.getpwnam("zookeeper").pw_uid
                gid = grp.getgrnam("zookeeper").gr_gid
                os.chown("/etc/zookeeper/myid", uid, gid)
                os.chown("/etc/zookeeper/conf/myid", uid, gid)

        usbar_service_conf = "/lib/systemd/system/unclear_shutdown_barrier.service"
        enable_usbar_service = False
        for part in psutil.disk_partitions():
            if "errors=panic" in part.opts:
                enable_usbar_service = (
                    os.path.exists(usbar_service_conf) and
                    os.stat(usbar_service_conf).st_ctime + 600 > time.time()
                )
                break

        if enable_usbar_service:
            sp.check_call(shlex.split("systemctl daemon-reload"))
            sp.check_call(shlex.split("systemctl enable unclear_shutdown_barrier"))
            sp.check_call(shlex.split("systemctl start unclear_shutdown_barrier"))

        # allow to run tcpdump by unprivileged user
        cap = "cap_net_raw+eip"
        tcpdump_binary = "/usr/sbin/tcpdump"
        try:
            sp.check_call(["/sbin/setcap", cap, tcpdump_binary])
        except sp.CalledProcessError as ex:
            logging.error("Error while setting capability %s for %s: %s", cap, tcpdump_binary, ex)

        # DEVTOOLSSUPPORT-10073
        sp.call(["/sbin/sysctl", "kernel.nmi_watchdog=0"])  # disable NMI watchdog to use more pref counters by tasks

    @staticmethod
    def _init_project_quota():
        project_quota_binary = "/usr/bin/project_quota"
        if not os.path.exists(project_quota_binary):
            return
        for partition in ("/place", "/"):
            logging.info("Initializing project quota on %s", partition)
            p = sp.Popen(
                [project_quota_binary, "init", partition], stdout=sp.PIPE, stderr=sp.PIPE
            )
            if not base.check_subprocess_status(p):
                logging.info("Project quota on %s successfully initialized", partition)
            logging.info("Turning on project quota on %s", partition)
            p = sp.Popen(
                [project_quota_binary, "on", partition], stdout=sp.PIPE, stderr=sp.PIPE
            )
            if not base.check_subprocess_status(p):
                logging.info("Project quota on %s successfully turned on", partition)

    def _onboot_linux_tunings(self):
        """ System tunings which should be performed only after each Linux host reboot. """
        host_conductortags = set(self.groups)

        if host_conductortags & {"sandbox1_server", "sandbox_server"}:
            self._create_cpusets()

        self._init_project_quota()

        # creates sysctl for yabs
        # Install cgroups
        cg_result = self.cgroup_check_install(self._CGROUP_PACKAGES, self._CGROUP_REJECTS, host_conductortags)

        if cg_result:
            with open(self._CGROUP_FLAG_FILE, "w") as f:
                f.write("")
            with open(self._CGROUP_CONFIG_FILE, "w") as f:
                f.write(self._CGROUP_CONFIG)
        else:
            try:
                os.remove(self._CGROUP_FLAG_FILE)
            except:
                pass

        # Fix "/var/log/messages" absence - SANDBOX-4503
        if not os.path.exists("/var/log/messages") and os.path.exists("/var/log/messages.1"):
            os.rename("/var/log/messages.1", "/var/log/messages")
        if os.path.exists("/etc/logrotate-hourly.conf"):
            with open("/etc/logrotate-hourly.conf", "r") as fh:
                conf = fh.readlines()
            if not any("create" in l for l in conf):
                with open("/etc/logrotate-hourly.conf", "w") as fh:
                    for l in conf:
                        fh.write(l)
                        if l.strip().startswith("size "):
                            fh.write("\tcreate\n")

        if base.check_tags(self.groups, undesired_tags=base.YP_LITE_TAGS):
            # https://ml.yandex-team.ru/thread/sandbox/164944336352483725/
            open("/proc/sys/kernel/kptr_restrict", "w").write("0")
            # Ensure `nbd` module is loaded into the kernel (SANDBOX-6271)
            sp.call(["/sbin/modprobe", "nbd", "max_part=16"])

    def _windows_postinstall(self):
        # Install imdisk driver
        source_path = self.find("imdiskinst.exe")
        dest_path = "/mnt/c/tools/imdiskinst.exe"
        win_dest_path = "C:\\tools\\imdiskinst.exe"
        if os.path.exists(dest_path):
            os.remove(dest_path)
        shutil.copyfile(source_path, dest_path)
        try:
            # For silent install env added
            sp.check_call(['/mnt/c/Windows/System32/cmd.exe', "/C", "set IMDISK_SILENT_SETUP=1&& {}".format(win_dest_path)], stderr=sp.STDOUT)
        except sp.CalledProcessError as ex:
            if not os.path.exists('/mnt/c/Windows/System32/imdisk.exe'):
                logging.error("Error while installing imdisk retcode: %s stdout: %s", ex.returncode, ex.output)
                raise

    def postinstall(self):
        self._unpack_layout()
        if on_linux():
            self._onetime_linux_tunings()
        if on_wsl():
            self._windows_postinstall()

    def start(self):
        if on_linux() or on_macos():
            self.patch_juggler_config()

        if on_linux():
            self._onboot_linux_tunings()

        self.create(["/bin/sleep", "365d"])

    def packages(self):
        return [
            {
                "type": "naked",
                "source": "sbr:WINDOWS_IMDISK_DRIVER",
                "sb_resource_filter": {
                    "attrs": {
                        "released": "stable"
                    },
                    "owner": "SANDBOX"
                },
                "alias": "imdisk_driver",
                "unpack_it": True,
            }
        ]

    @staticmethod
    def _reload_network():
        try:
            ya_netconfig_path = "/usr/lib/yandex-netconfig/ya-netconfig"
            logging.debug("Reloading network configuration via netconfig, output follows:")
            p = sp.Popen([ya_netconfig_path, "reload"], stdout=sp.PIPE, stderr=sp.STDOUT)
            reload_out = p.communicate()[0]
            logging.debug(reload_out)
            if '(use "start")' in reload_out:
                p = sp.Popen([ya_netconfig_path, "restart"], stdout=sp.PIPE, stderr=sp.STDOUT)
                logging.debug(p.communicate()[0])
        except (sp.CalledProcessError, OSError) as ex:
            logging.error("Error reloading network configuration: %s", ex)
            return False

    @staticmethod
    def _update_routes():
        try:
            logging.debug("Applying routes update, output follows:")
            p = sp.Popen(
                ["/usr/lib/yandex-netconfig/yandex-netconfig-tool", "apply_routes", "--force"],
                stdout=sp.PIPE, stderr=sp.STDOUT,
            )
            logging.debug(p.communicate()[0])
        except (sp.CalledProcessError, OSError) as ex:
            logging.error("Error updating network routes: %s", ex)
            return False

    def ping(self):
        if on_wsl():
            return True
        # On Ubuntu 16.04 systemd doesn't know how to restart `ntp` service in case of failures.
        # https://st.yandex-team.ru/SANDBOX-4835
        if platform.linux_distribution()[1] == "16.04" and base.check_tags(
            self.groups, undesired_tags=base.YP_LITE_TAGS
        ):
            ntp_state = sp.check_output(["/bin/systemctl", "show", "-p", "SubState", "ntp.service"]).strip()
            if ntp_state == "SubState=exited":
                sp.check_call(["/bin/systemctl", "restart", "ntp.service"])

        # TODO: RUNTIMECLOUD-3947
        # TODO: Temporary disable network reload on servers - it leads to MongoDB elections sometimes.
        # TODO: On other hosts regular `netconfig reload` causes hostname resolution errors and `cqudp` crashes.
        # TODO: So it totally disabled. In case of non-working fastbone issues will be detected, this code
        # TODO: should be enabled again and the appropriate ticket should be reopened.
        if not on_linux() or "sandbox_server" in self.groups:
            return True
        if self.persist is None:
            self.persist = {}
        now = dt.datetime.now()
        last_reload = self.persist.setdefault("network_reloaded", now + dt.timedelta(minutes=random.random() * 30))
        last_routes_update = self.persist.get("network_routes_updated", dt.datetime.min)

        if last_reload + self.NETWORK_RELOAD_PERIOD < now:
            self._reload_network()
            last_reload = last_routes_update = now
        elif last_routes_update + self.NETWORK_ROUTES_UPDATE_PERIOD < now:
            self._update_routes()
            last_routes_update = now

        self.persist["network_reloaded"] = last_reload
        self.persist["network_routes_updated"] = last_routes_update
        return True


class Atop(base.Base):
    __name__ = "sandbox_atop"

    _LOGFILE = "/var/log/sandbox/atop.log"
    _ROTATE_LOGS = [_LOGFILE]
    _ROTATE_SIGNAL = signal.SIGTERM  # atop process escapes from _launcher_ process group, but _launcher_ does not forward HUP signal to the child process
    ATOP_BINARY_PATH = "/place/sandbox-data/atop"

    def user(self):
        return "root"

    def packages(self):
        return [
            {
                "type": "naked",
                "source": "sbr:SANDBOX_ATOP_BINARY",
                "sb_resource_filter": {
                    "attrs": {
                        "released": "stable"
                    },
                    "owner": "SANDBOX"
                },
                "alias": "atop_binary",
                "unpack_it": False,
            },
        ]

    def postinstall(self):
        path = os.path.join(self.get_package("atop_binary"), "atop")
        if os.path.exists(self.ATOP_BINARY_PATH):
            os.remove(self.ATOP_BINARY_PATH)
        shutil.copyfile(path, self.ATOP_BINARY_PATH)
        self.change_permissions(self.ATOP_BINARY_PATH, chmod=0o755)

    def start(self):
        path = os.path.join(self.get_package("atop_binary"), "atop")
        self.change_permissions(path, chmod=0o755)
        logging.info("Found package %s with atop binary.", path)
        cmd = [path, "-a", "-w", self._LOGFILE, "15"]
        self.create(cmd, env={"TERM": "xterm"})

    def ping(self):
        self._rotate_logs(move=True, notify=True)
        return True
