import os
import hashlib
import requests
import collections


TVMApp = collections.namedtuple("TVMApp", ("alias", "tvm_id"))


class TVMTool(object):
    """
    We do not use builtin TVMTool servant from Samogon because it:
    * uses `system` environment instead of `user`;
    * does not use HTTP status check;
    * does not support multiple service ticket destinations;
    * does not use destination aliases (e.g. `yav`);
    * adds port number into the servant's name, which does not look nice :)
    """

    SOURCE = {
        0: TVMApp("sandbox", 2002826),  # sandbox-production
        1: TVMApp("sandbox", 2011100),  # sandbox-testing
    }
    for k in (2, 3):
        SOURCE[k] = SOURCE[k % 2]

    # Aliases and client IDs of all TVM apps we want to access
    TARGETS = (
        TVMApp("yav", 2001357),
        TVMApp("mds-testing", 2000272),
        TVMApp("mds-production", 2000273),
        TVMApp("arcanum", 2000683),
        TVMApp("solomon", 2010242),
        TVMApp("blackbox", 223),  # blackbox.yandex-team.ru
        TVMApp("logbroker", 2001059),
        TVMApp("skyboned", 2021848),
        TVMApp("d-testing", 2023015),
        TVMApp("d-production", 2023017),
    )

    BLACKBOX = 2  # blackbox.yandex-team.ru

    PORT = 13570
    PING_TIMEOUT = 15.

    SERVICE_USER = "zomb-sandbox"

    # Secret file with tvmtool access token located at the service user home directory.
    # Must be equal to `common.tvm.access_token` from settings
    TOKEN_FILENAME = ".tvmtool_access_token"

    # Resources with tvmtool binary for all Sandbox platforms
    PACKAGES = {
        "linux": "sbr:1102159831",
        "macos": "sbr:1102159864",
    }

    __name__ = __servant_name__ = "tvmtool"
    __cfg_fmt__ = "json"

    def __init__(self, key, tvm_secret, platform):
        self.platform = platform or ""
        self.__tvm_secret = tvm_secret

        source_app = self.SOURCE[key]

        h = hashlib.sha256()
        h.update(tvm_secret.encode("utf-8", errors="replace") if isinstance(tvm_secret, unicode) else tvm_secret)
        self._auth = h.hexdigest()[:32]  # tvmtool requires a token of exactly this length

        self.__cfg__ = {
            "BbEnvType": self.BLACKBOX,
            "clients": {
                source_app.alias: {
                    "self_tvm_id": source_app.tvm_id,
                    "secret": tvm_secret,
                    "dsts": {
                        app.alias: {"dst_id": app.tvm_id}
                        for app in self.TARGETS
                    }
                }
            }
        }

    def ping(self):
        url = "http://127.0.0.1:{}/tvm/ping".format(self.PORT)
        try:
            requests.get(url, timeout=self.PING_TIMEOUT).raise_for_status()
        except:
            return False
        return True

    @property
    def info(self):
        return {"port": self.PORT, "auth": self._auth}

    def packages(self):
        arch = "linux"
        if self.platform.startswith("darwin"):
            arch = "macos"
        return [{
            "type": "naked",
            "source": self.PACKAGES[arch],
            'unpack_it': False,
        }]

    def start(self):
        cmd = [
            self.find("tvmtool"),
            "-e",
            "--port", str(self.PORT),
            "-c", self.config_as_file(),
        ]

        self.create(cmd, env={"QLOUD_TVM_TOKEN": self._auth}, cwd="{srv}")

    def postinstall(self):
        token_path = os.path.join(os.path.expanduser("~" + self.SERVICE_USER), self.TOKEN_FILENAME)
        with open(token_path, "w") as f:
            f.write(self._auth)
        self.change_permissions(token_path, chmod=0o400, chown=self.SERVICE_USER)

        # SANDBOX-8823: remove after Statinfra stopped using STEP
        tvm_secret_path = "/etc/yandex/statbox-push-client/.tvm_secret"
        if os.path.exists(os.path.dirname(tvm_secret_path)):
            with open(tvm_secret_path, "wb") as f:
                f.write(self.__tvm_secret)
            self.change_permissions(tvm_secret_path, chmod=0o400, chown="statbox")
