from __future__ import absolute_import

import os
import json
import copy
import yaml
import resource
import urlparse

import requests

from . import tvm


class UnifiedAgent(object):
    __name__ = __servant_name__ = "unified_agent"
    __cfg_fmt__ = "yaml"

    TVM_SECRET_PATH = "{srv}/.tvm_secret"
    GRPC_PATH = "unix:///opt/sandbox/run/unified_agent/{}.sock"
    STORAGES_PATH = "/opt/sandbox/run/unified_agent_storages"
    STATUS_PORT = 16301
    PING_TIMEOUT = 5
    LEGACY_WEBSERVER_METRIC = "legacy_webserver"
    ZOOKEEPER_METRIC = "zookeeper"

    __default_pipe = "default"
    __nobatch_pipe = "nobatch"
    __autocheck = "autocheck"
    __agentr_statistics = "agentr_statistics"
    __serviceq_statistics = "serviceq_statistics"
    __infrequent_statistics = "infrequent_statistics"
    __serviceapi_statistics = "serviceapi_statistics"
    __serviceapi_calls_statistics = "serviceapi_calls_statistics"
    __resource_audit_statistics = "resource_audit_statistics"
    __billing_metrics = "billing_metrics"
    __proxy_statistics = "proxy_statistics"

    TOPICS = {
        __autocheck: {
            0: "//logbroker.yandex.net/autocheck_sandbox_tasks/stages",
            1: "//logbroker.yandex.net/autocheck_sandbox_tasks/test/stages"
        },
        __agentr_statistics: {
            0: "//logbroker.yandex.net/sandbox/statistics/topics/agentr",
            1: "//logbroker.yandex.net/sandbox/preprod/statistics/topics/agentr",
        },
    }

    SERVER_TOPICS = {
        __serviceq_statistics: {
            0: "//logbroker.yandex.net/sandbox/statistics/topics/serviceq",
            1: "//logbroker.yandex.net/sandbox/preprod/statistics/topics/serviceq",
        },
        __infrequent_statistics: {
            0: "//logbroker.yandex.net/sandbox/statistics/topics/infrequent",
            1: "//logbroker.yandex.net/sandbox/preprod/statistics/topics/infrequent",
        },
        __serviceapi_statistics: {
            0: "//logbroker.yandex.net/sandbox/statistics/topics/serviceapi",
            1: "//logbroker.yandex.net/sandbox/preprod/statistics/topics/serviceapi",
        },
        __serviceapi_calls_statistics: {
            0: "//logbroker.yandex.net/sandbox/statistics/topics/serviceapi_calls",
            1: "//logbroker.yandex.net/sandbox/preprod/statistics/topics/serviceapi_calls",
        },
        __resource_audit_statistics: {
            0: "//logbroker.yandex.net/sandbox/statistics/topics/resource_audit",
            1: "//logbroker.yandex.net/sandbox/preprod/statistics/topics/resource_audit",
        },
        __billing_metrics: {
            0: "//lbkx.logbroker.yandex.net/yc/yandex/billing-sandbox",
        },
    }

    PROXY_TOPICS = {
        __proxy_statistics: {
            0: "//logbroker.yandex.net/sandbox/statistics/topics/proxy",
            1: "//logbroker.yandex.net/sandbox/preprod/statistics/topics/proxy",
        },
    }

    SOLOMON_METRICS = {
        LEGACY_WEBSERVER_METRIC: {
            "input": {
                "plugin": "metrics_pull",
                "config": {
                    "url": "http://localhost:8080/api/v1.0/service/status/server_solomon",
                    "format": {"solomon_json": {}},
                    "metric_name_label": "name",
                    "poll_period": "15s",
                    "project": "sandbox",
                    "service": None
                },
            },
            "pipe": [
                {
                    "storage_ref":
                        {
                            "name": "legacy_webserver"
                        }
                }
            ],
            "port": 30577
        },
        # subprocess filter is a hack which replaces the whole content of the generated metrics with subprocess output
        # so agent_metrics is needed only as a periodic metrics producer. Replace after UNIFIEDAGENT-335
        ZOOKEEPER_METRIC: {
            "input": {
                "plugin": "agent_metrics",
                "config": {
                    "service": "zookeeper"
                }
            },
            "pipe": [
                {
                    "filter": {
                        "plugin": "subprocess",
                        "config": {
                            "command": "/home/zomb-sandbox/unified_agent/zookeeper_status.py"
                        }
                    }
                },
                {
                    "storage_ref": {
                        "name": "zookeeper"
                    }
                },
                {
                    "filter": {
                        "plugin": "assign",
                        "config": {
                            "session": [
                                {"_metrics_service": "zookeeper"},
                                {"_metrics_project": "sandbox"},
                                {"_metrics_format": "solomon_json"}
                            ]
                        }
                    }
                }
            ],
            "port": 30578
        }
    }

    for topics in (TOPICS, SERVER_TOPICS, PROXY_TOPICS):
        for d in topics.values():
            for k in (2, 3):
                if k % 2 in d:
                    d[k] = d[k % 2]

    CONFIG = {
        "status": {
            "host": "localhost",
            "port": STATUS_PORT
        },
        "pipes": [
            {
                "name": __default_pipe,
                "pipe": [{
                    "filter": {
                        "plugin": "batch",
                        "config": {
                            "delimiter": "\n",
                            "flush_period": "100ms",
                            "limit": {
                                "bytes": "256kb"
                            }
                        }
                    }
                }]
            },
            {
                "name": __nobatch_pipe,
                "pipe": []
            }
        ],
        "storages": [
            {
                "name": None,
                "plugin": "fs",
                "config": {
                    "directory": None,
                    "max_partition_size": "100mb"
                }
            }
        ],
        "routes": [{
            "input": {
                "plugin": "grpc",
                "config": {
                    "uri": None,
                    "grpc_memory_quota": "100mb"
                },
                "flow_control": {
                    "new_sessions_rate_limit": 25
                }
            },
            "channel": {
                "pipe": [
                    {
                        "pipe_ref": {
                            "name": "default"
                        }
                    },
                    {
                        "storage_ref": {
                            "name": None
                        }
                    }
                ],
                "output": {
                    "plugin": "logbroker",
                    "config": {
                        "endpoint": None,
                        "topic": None,
                        "tvm": {
                            "client_id": None,
                            "secret": {
                                "file": None
                            }
                        }
                    }
                }
            }
        }]
    }

    CUSTOM_PIPES = {__autocheck: __nobatch_pipe}
    RLIMIT_NOFILE = (1048576, 1048576)
    YP_RLIMIT_NOFILE = (102400, 102400)

    def __init__(self, key, tvm_secret, run_on_server, run_on_proxy, solomon_metrics):
        self.__key = key
        self.__tvm_id = tvm.TVMTool.SOURCE[self.__key].tvm_id
        self.__tvm_secret = tvm_secret
        self.__run_on_server = run_on_server
        self.__run_on_proxy = run_on_proxy
        self.__solomon_metrics = solomon_metrics
        self.__cfg__ = {}

    def packages(self):
        return [
            {
                "type": "naked",
                "source": "sbr:UNIFIED_AGENT_BIN",
                "sb_resource_filter": {
                    "attrs": {
                        "released": "stable"
                    },
                    "owner": "LOGBROKER"
                },
                "alias": "unified_agent",
                "unpack_it": False,
            },
        ]

    def _ensure_storage_dirs(self):
        sock_dirpath = os.path.dirname(urlparse.urlparse(self.GRPC_PATH).path)
        for path in (sock_dirpath, self.STORAGES_PATH):
            if not os.path.exists(path):
                os.mkdir(path)
            self.change_permissions(path, chmod=0o755)
            self.change_permissions(path, recursive=True)

    def start(self):
        secret_path = self.substval(self.TVM_SECRET_PATH)

        self.__cfg__ = self.get_config(secret_path)

        def __is_zookeeper_route(route):  # type: (dict) -> bool
            return route.get("input", {}).get("config", {}).get("service", "").startswith("zookeeper")
        routes_zookeeper = [r for r in self.__cfg__["routes"] if __is_zookeeper_route(r)]
        if routes_zookeeper:
            route_zookeeper = routes_zookeeper[0]
            path_script_zookeeper_status = route_zookeeper["channel"]["pipe"][0]["filter"]["config"]["command"].strip()
            self.change_permissions(path_script_zookeeper_status, chmod=0o755)

        path = os.path.join(self.get_package("unified_agent"), "unified_agent")
        self.change_permissions(path, chmod=0o755)
        cmd = [path, "--config", self.config_as_file()]
        resources = {
            resource.RLIMIT_NOFILE: self.YP_RLIMIT_NOFILE if self.__key // 2 else self.RLIMIT_NOFILE,
        }
        self._ensure_storage_dirs()
        self.create(cmd, resources=resources)

    def postinstall(self):
        self.writetofile(self.__tvm_secret, self.TVM_SECRET_PATH, chmod=0o400)
        self._ensure_storage_dirs()

    def ping(self):
        r = requests.get("http://localhost:{}/status".format(self.STATUS_PORT), timeout=self.PING_TIMEOUT)
        r.raise_for_status()
        return yaml.load(r.text)["_status"]["ready"]

    def get_config(self, secret_path):
        cfg = copy.deepcopy(self.CONFIG)
        routes = cfg["routes"]
        route_tmpl = routes.pop()
        storages = cfg["storages"]
        storage_tmpl = storages.pop()
        topics = dict(self.TOPICS)
        if self.__run_on_server:
            topics.update(self.SERVER_TOPICS)
        if self.__run_on_proxy:
            topics.update(self.PROXY_TOPICS)

        for project, topic in topics.items():
            if self.__key not in topic:
                continue
            p = urlparse.urlparse(topic[self.__key])
            logbroker_endpoint, topic = p.netloc, p.path
            route = copy.deepcopy(route_tmpl)
            route["channel"]["pipe"][1]["storage_ref"]["name"] = project
            route["input"]["config"]["uri"] = self.GRPC_PATH.format(project)
            output_config = route["channel"]["output"]["config"]
            pipe_ref = route["channel"]["pipe"][0]["pipe_ref"]
            tvm_settings = output_config["tvm"]
            tvm_settings["client_id"] = self.__tvm_id
            tvm_settings["secret"]["file"] = secret_path
            output_config["endpoint"] = logbroker_endpoint
            output_config["topic"] = topic
            pipe_ref["name"] = self.CUSTOM_PIPES.get(project, self.__default_pipe)
            routes.append(route)
            storage = copy.deepcopy(storage_tmpl)
            storage["name"] = project
            storage["config"]["directory"] = os.path.join(self.STORAGES_PATH, project)
            storages.append(storage)

        for project in self.__solomon_metrics:
            metric = self.SOLOMON_METRICS.get(project)
            if not metric:
                continue
            storage = copy.deepcopy(storage_tmpl)
            storage["name"] = project
            storage["config"]["directory"] = os.path.join(self.STORAGES_PATH, project)
            storages.append(storage)
            route = copy.deepcopy(route_tmpl)
            route["input"] = metric["input"]
            if project == self.LEGACY_WEBSERVER_METRIC:
                route["input"]["config"]["service"] = project + "1" if self.__key % 2 > 0 else project
            route["channel"]["pipe"] = metric["pipe"]
            route["channel"]["output"]["config"] = {"port": metric["port"]}
            route["channel"]["output"]["plugin"] = "metrics_pull"
            routes.append(route)

        return cfg


def ua_projects_dumped_config(*project_groups):
    return json.dumps({
        project: {"uri": UnifiedAgent.GRPC_PATH.format(project)}
        for group in project_groups
        for project in group
    })


ua_sockets_config_for_client = ua_projects_dumped_config(UnifiedAgent.TOPICS)
ua_sockets_config_for_server = ua_projects_dumped_config(UnifiedAgent.TOPICS, UnifiedAgent.SERVER_TOPICS)
ua_sockets_config_for_proxy = ua_projects_dumped_config(UnifiedAgent.TOPICS, UnifiedAgent.PROXY_TOPICS)
