import os
import logging
import xml.dom.minidom
import distutils.version
import subprocess as sp

import requests

HOST_TAGS_URL_BASE = "https://c.yandex-team.ru/api/get_host_tags"
TAG2HOSTS_URL_BASE = "https://c.yandex-team.ru/api/tag2hosts"
ISS_SOCKET = "/run/iss/pod.socket"
ISS_POD_SPEC_REQ = b"GET /pod_spec HTTP/1.0\r\nHost: localhost\r\n\r\n"


def prefix_for_key(key):
    replacement = key % 2 or ""
    return "sandbox{}".format(replacement)


def resolve_ctag(ctag):
    return requests.get(os.path.join(TAG2HOSTS_URL_BASE, ctag)).text.splitlines()


def host_tags(fqdn):
    return requests.get(os.path.join(HOST_TAGS_URL_BASE, fqdn)).text.splitlines()


def sort_hosts(hosts):
    return sorted(hosts, key=distutils.version.LooseVersion)


def prettify_xml(config):
    indented = xml.dom.minidom.parseString(config).toprettyxml(indent=" " * 4, encoding="ascii")
    return "\n".join(line for line in indented.splitlines() if line.strip())


def check_output(cmd):
    try:
        with open(os.devnull, "w") as devnull:
            for line in sp.check_output(cmd, stderr=devnull).split("\n"):
                yield line
    except (sp.CalledProcessError, OSError):
        pass


class EscapeeMixin(object):
    """
    Samogon runs servants' (sub)processes in a couple of control groups to have a handy way to kill them.
    In order to avoid having been killed way too early, a servant needs to escape these cgroups by
    moving its processes' identifiers from each cgroup into the top level one.
    """

    CGROUP_BASE = "/sys/fs/cgroup/"
    CGROUP_TASKS_PATTERN = CGROUP_BASE + "{ctl}/key_{key}/user/{name}_{tag}/tasks"

    def move_from_cgroups(self):
        for ctl in ("cpu", "cpu,cpuacct", "memory", "blkio", "freezer"):
            cgroup_base = self.CGROUP_BASE + ctl
            curr_tasks = self.CGROUP_TASKS_PATTERN.format(
                ctl=ctl, key=self._key, name=self.__name__, tag=self.uniq_tag()
            )
            logging.debug("Checking cgroup %r", curr_tasks)
            if not os.path.exists(curr_tasks):
                continue
            default_tasks = os.path.join(cgroup_base, "tasks")
            with open(curr_tasks, "r") as infh:
                tasks = list(infh)
            logging.info(
                "Moving following tasks from %r cgroup %r to root: %r",
                os.path.dirname(curr_tasks), ctl, map(str.strip, tasks)
            )
            for l in tasks:
                try:
                    with open(default_tasks, "a") as outfh:
                        outfh.write(l)
                except IOError as ex:
                    logging.warning("Error moving pid %s: %s", l.strip(), ex)
