import hashlib
import textwrap


REPLICA_BLOCK_TEMPLATE = textwrap.dedent(
    """
    <replica>
        <host>{}</host>
        <port>{}</port>
        <user>{}</user>
        <password>{}</password>
    </replica>
    """
).strip()

SHARD_BLOCK_TEMPLATE = textwrap.dedent(
    """
    <shard>
        <internal_replication>true</internal_replication>
        {}
    </shard>
    """
).strip()

ZK_BLOCK_TEMPLATE = textwrap.dedent(
    """
    <node index="{}">
        <host>{}</host>
        <port>2181</port>
    </node>
    """
).strip()

USER_BLOCK_TEMPLATE = textwrap.dedent(
    """
    <{user}>
        <profile>{profile}</profile>
        <quota>{quota}</quota>
        {password_tag}
        <networks>
            <ip>::/0</ip>
        </networks>
        <access_management>{access_management}</access_management>
    </{user}>
    """
)


def replica_block(fqdn, port, login, password):
    return REPLICA_BLOCK_TEMPLATE.format(fqdn, port, login, password)


def shard_block(hosts, port, login, password):
    replica_blocks = [
        replica_block(fqdn, port, login, password)
        for fqdn in hosts
    ]
    return SHARD_BLOCK_TEMPLATE.format("\n".join(replica_blocks))


def zk_block(index, fqdn):
    return ZK_BLOCK_TEMPLATE.format(index, fqdn)


def user_block(user, profile="default", quota="default", password=None, access_management=0):
    if password is not None:
        password_tag = "<password_sha256_hex>{}</password_sha256_hex>".format(
            hashlib.sha256(password).hexdigest()
        )
    else:
        password_tag = "<password/>"

    return USER_BLOCK_TEMPLATE.format(user=user, profile=profile, quota=quota, password_tag=password_tag,
                                      access_management=access_management)
