#!/skynet/python/bin/python -W ignore::UserWarning
# -*- coding: utf-8 -*-
from __future__ import print_function, unicode_literals

import os
import abc
import sys
import json
import time
import shutil
import logging
import tarfile
import tempfile
import textwrap
import platform
import requests
import subprocess as sp
import progressbar as pb

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr

from sandbox.devbox import utils

logger = logging.getLogger('sandbox')

UPDATE_ON_WHITELIST = (
    "start", "start_server", "start_client", "check_updates",
)


class BootstrapError(common.errors.SandboxException):
    pass


class ResourcesVersion(object):
    # Increment this number to force Local Sandbox to perform `check_updates`
    VERSION = 1

    def __init__(self, path):
        self.path = os.path.join(path, ".res_version")

    @property
    def outdated(self):
        if not os.path.exists(self.path):
            return True
        try:
            version = int(open(self.path).read().strip())
        except ValueError:
            return True

        return version < self.VERSION

    def set_updated(self):
        with open(self.path, "wb") as f:
            f.write(str(self.VERSION))


class LocalEnvironment(object):
    """
    View over local environment. Provides necessary properties to boot and update environment.
    """
    __metaclass__ = abc.ABCMeta

    CHECK_PERIOD = 24 * 60 * 60  # Delay between the two runs, check for updates. In seconds
    is_archive = True

    def __init__(self, path, sandbox_revision):
        self.path = path
        self.update_file = os.path.join(self.path, '.updated')
        self._sandbox_revision = sandbox_revision

    @abc.abstractproperty
    def exe_path(self):
        pass

    @abc.abstractproperty
    def resource(self):
        """
        :return: Sandbox resource meta
        :rtype: `SandboxResource`
        """

    @property
    def exists(self):
        return os.path.exists(self.exe_path)

    @property
    def revision_file_path(self):
        return os.path.join(self.path, '.revision')

    @property
    def revision_file_exists(self):
        return os.path.exists(self.revision_file_path)

    def get_revision(self):
        if self.revision_file_exists:
            with open(self.revision_file_path) as f:
                return f.readline()

    def get_local_version(self):
        """
        :return: Sandbox revision or resource version
        """
        try:
            with open(self.update_file, 'rb') as f:
                return f.readline().strip()
        except IOError:
            return ''

    @property
    def actual_version(self):
        return self._sandbox_revision

    @property
    def was_updated(self):
        if os.path.exists(self.update_file):
            with open(self.update_file, "rb") as f:
                return len(f.readlines()) == 2
        return False

    def refresh_update_stamp(self, update_happened=False):
        prev_rev = (self.get_local_version() or "Fresh") if update_happened else None
        with open(self.update_file, "wb") as f:
            f.write("\n".join((self.actual_version, prev_rev)) if prev_rev is not None else self.actual_version)

    def version_changed(self):
        """
        :return: `True`, if svn revision of repo differs from that of a resource
        """
        return self.get_local_version() != self._sandbox_revision

    def need_to_check(self):
        try:
            updated_last_time = os.stat(self.update_file).st_mtime
        except OSError:
            updated_last_time = None
        update_outdated = updated_last_time is None or time.time() > (updated_last_time + self.CHECK_PERIOD)
        return update_outdated or self.version_changed()


class SandboxResource(object):
    """
    Class for interacting with Sandbox resources
    """

    API_TOTAL_WAIT = 20
    attrs = None
    check_platform = False

    def __init__(self):
        self._resource = None
        self.version = None
        self.url = None
        self.id = None

    def initialize(self):
        """
        Initiate `_resource` variable with compatible resource information via Sandox API
        """

        self._resource = self._compatible_released_resource_info()
        if not self._resource:
            return
        self.version = self._resource['attributes'].get('version')
        self.url = self._resource['http']['proxy']
        self.id = self._resource['id']

    def __str__(self):
        if self.version:
            return "resource #{}, version {}".format(self.id, self.version)
        else:
            return "resource #{}".format(self.id)

    @property
    def exists(self):
        return self._resource is not None

    @classmethod
    def _compatible_released_resource_info(cls):
        """
        Determines latest releases resource for given object, based on `resource_type` and current platform.
        In case of no compatible resource found, `None` will be returned.

        :return: :class:`dict` with the resource information or `None`
        """

        api = common.rest.Client(common.rest.Client.DEFAULT_BASE_URL, total_wait=cls.API_TOTAL_WAIT)

        res = api.resource.read(**cls.attrs)
        logger.debug('Retrieve information about last released resource with sandbox dependencies.')

        for res in res['items']:
            if cls.check_platform:
                if cls._check_resource_platform(res):
                    logger.debug('Resource information has been successfully retrieved.')
                    return res
            else:
                return res
        logger.debug('Information about compatible resource not found.')

    @staticmethod
    def _check_resource_platform(resource):
        """
        Checks, whether given resource is compatible with the current platform.

        :param resource: Resource object to be checked.
        :return: `True`, if the given resource is compatible with the current platform, `False` otherwise.
        :rtype: bool
        """
        resource_platform = resource['attributes'].get('platform', '')
        if resource_platform == 'any':
            return True
        current_platform = platform.platform()
        resource_platforms = resource_platform.split(',')
        for resource_platform in resource_platforms:
            if common.platform.compare_platforms(resource_platform, current_platform):
                return True
        return False


class SandboxResourceVenv(SandboxResource):
    check_platform = True
    attrs = {
        "type": "SANDBOX_DEPS",
        "attrs": json.dumps({ctr.ServiceAttributes.RELEASED: ctt.ReleaseStatus.STABLE}),
        "state": ctr.State.READY,
        "arch": common.config.Registry().this.system.family,
        "order": "-id",
        "limit": 100,
    }


class PreexecutorBinary(SandboxResource):
    check_platform = False
    attrs = {
        "type": "SANDBOX_ARCHIVE",
        "attrs": json.dumps({ctr.ServiceAttributes.RELEASED: ctt.ReleaseStatus.STABLE, "type": "preexecutor-linux"}),
        "state": ctr.State.READY,
        "order": "-id",
        "limit": 1,
    }


class ServiceApiBinary(SandboxResource):
    check_platform = False
    attrs = {
        "type": "SANDBOX_ARCHIVE",
        "attrs": json.dumps({ctr.ServiceAttributes.RELEASED: ctt.ReleaseStatus.STABLE, "type": "serviceapi-linux"}),
        "state": ctr.State.READY,
        "order": "-id",
        "limit": 1,
    }


class TaskboxBinary(SandboxResource):
    check_platform = False
    attrs = {
        "type": "SANDBOX_ARCHIVE",
        "attrs": json.dumps({ctr.ServiceAttributes.RELEASED: ctt.ReleaseStatus.STABLE, "type": "taskbox-linux"}),
        "state": ctr.State.READY,
        "order": "-id",
        "limit": 1,
    }


class TVMToolBinary(SandboxResource):
    check_platform = False
    attrs = {
        "type": "TVM_TOOL_BINARY",
        "state": ctr.State.READY,
        "arch": "linux",
        "order": "-id",
        "limit": 1,
    }

    def initialize(self):
        super(TVMToolBinary, self).initialize()
        self.version = str(self._resource["task"]["id"])


class Py3SourcesBinary(SandboxResource):
    check_platform = False
    attrs = {
        "type": "SANDBOX_PY_3_MODULES_PARSER",
        "attrs": json.dumps({ctr.ServiceAttributes.RELEASED: ctt.ReleaseStatus.STABLE}),
        "state": ctr.State.READY,
        "order": "-id",
        "limit": 1,
    }


class LocalVenv(LocalEnvironment):
    @property
    def exe_path(self):
        return os.path.join(self.path, "bin", "python")

    @common.utils.singleton_property
    def resource(self):
        return SandboxResourceVenv()


class ServiceApiEnv(LocalEnvironment):
    @property
    def exe_path(self):
        return os.path.join(self.path, "serviceapi")

    @common.utils.singleton_property
    def resource(self):
        return ServiceApiBinary()


class PreexecutorEnv(LocalEnvironment):
    @property
    def exe_path(self):
        return os.path.join(self.path, "preexecutor")

    @common.utils.singleton_property
    def resource(self):
        return PreexecutorBinary()


class TaskboxEnv(LocalEnvironment):
    @property
    def exe_path(self):
        return os.path.join(self.path, "taskbox")

    @common.utils.singleton_property
    def resource(self):
        return TaskboxBinary()


class TVMToolEnv(LocalEnvironment):
    is_archive = False

    @property
    def exe_path(self):
        return os.path.join(self.path, "tvmtool")

    @common.utils.singleton_property
    def resource(self):
        return TVMToolBinary()

    @property
    def actual_version(self):
        return self.resource.version

    def need_to_check(self):
        # we can't make assumptions based on comparison of saved version and local repo revision
        return True


class Py3SourcesEnv(LocalEnvironment):
    is_archive = False

    @property
    def exe_path(self):
        return os.path.join(self.path, "py3_sources")

    @common.utils.singleton_property
    def resource(self):
        return Py3SourcesBinary()


class BootEnvironment(object):
    """
    Bootstrapper for sandbox virtual environment. Aggregates LocalEnvironment and SandboxResource.
    Checks environments for required packages, loads necessary resources, maintains environment.
    """

    LOAD_RESOURCE_TIMEOUT = 20
    SKYNET_PREFIX = '/skynet'

    def __init__(self, runtime_dir, sandbox_dir):
        self._runtime_dir = runtime_dir
        self._sandbox_dir = sandbox_dir
        self._sandbox_revision = utils.get_path_revision(sandbox_dir, last_changed_rev=True)
        self._local_venv = LocalVenv(os.path.join(runtime_dir, "venv"), self._sandbox_revision)
        self._local_resources = ResourcesVersion(runtime_dir)

    @staticmethod
    def need_to_update(env, res):
        current_rev = env.get_revision()
        return not current_rev or (current_rev != res.version)

    @property
    def should_check_for_updates(self):
        if len(sys.argv) <= 1 or sys.argv[1] not in UPDATE_ON_WHITELIST:
            return False
        return self._local_venv.was_updated or self._local_resources.outdated

    def check_requirements(self, force_env_setup):
        executable = sys.executable
        real_prefix = getattr(sys, "real_prefix", "")
        if not executable.startswith(self.SKYNET_PREFIX) and not real_prefix.startswith(self.SKYNET_PREFIX):
            raise BootstrapError("Sandbox server can run only by skynet python interpreter..")
        logger.debug("Checking third-party packages requirements...")
        import pkg_resources
        not_satisfied_requirements = []
        reqs_file = os.path.join(self._sandbox_dir, "requirements.txt")
        with open(reqs_file) as requirements:
            for line in requirements:
                req = line.strip()
                try:
                    pkg_resources.require(req)
                except pkg_resources.DistributionNotFound:
                    not_satisfied_requirements.append("'{}' not installed.".format(req))
                except pkg_resources.VersionConflict as ex:
                    not_satisfied_requirements.append("'{}' version mismatch: {}".format(req, str(ex.args[0])))
        if not_satisfied_requirements:
            print("Following requirements are not satisfied:" + "\n\t".join([''] + not_satisfied_requirements))
            if force_env_setup or common.console.confirm("\rDo you want to run pip tool ([y]/n)? "):
                cmd = [
                    sys.executable,
                    os.path.join(os.path.dirname(sys.executable), 'pip'),
                    'install', '--upgrade', '--index=https://pypi.yandex-team.ru/simple/', '--requirement', reqs_file
                ]
                env = os.environ.copy()
                env.update({'CFLAGS': '-I/usr/local/include', 'LDFLAGS': '-L/skynet/python/lib -L/usr/local/lib'})
                sp.check_call(cmd, env=env)

    @staticmethod
    def _extract_tar_file(tar_file, destination, with_backup=False):
        """
        Extracts given tarball file into the destination.

        :param tar_file:    tarball file name with full path.
        :param destination: directory name with full path to extract the given tarball to.
        :return:            given directory name.
        """
        logger.debug('Extract virtualenv of the tar-archive.')
        tmp_env = os.path.join(os.path.dirname(destination), '.old_venv')
        if with_backup:
            shutil.move(destination, tmp_env)
        try:
            tarfile.open(tar_file.name, mode='r:gz').extractall(destination)
        except:
            logger.debug('Error while extract archive content.')
            if with_backup:
                shutil.move(tmp_env, destination)
            raise
        finally:
            if with_backup:
                shutil.rmtree(tmp_env)
        logger.debug('Extraction is complete.')
        return destination

    @classmethod
    def update_environment(cls, url, local_venv):
        with tempfile.NamedTemporaryFile('wb') as tar_file:
            if not cls.fetch_resource(url, tar_file):
                return False

            if local_venv.is_archive:
                cls._extract_tar_file(tar_file, local_venv.path, local_venv.exists)
            else:
                if not os.path.exists(local_venv.path):
                    os.makedirs(local_venv.path)
                try:
                    os.unlink(local_venv.exe_path)
                except OSError:
                    pass
                shutil.copy(tar_file.name, local_venv.exe_path)
                os.chmod(local_venv.exe_path, 0o755)

        return True

    @classmethod
    def fetch_resource(cls, url, fileobj):
        """
        Loads virtualenv.tar.gz file from resource_url location and extracts it into environment folder.

        :param url: url source of remote resource
        :param fileobj: file object to store in
        """
        logger.debug('Fetch resource from URL: %s.', url)
        size_label = pb.FormatLabel('%(cursize)s/%(maxsize)s')
        size_label.mapping['cursize'] = ('currval', lambda x: common.utils.size2str(x))
        size_label.mapping['maxsize'] = ('maxval', lambda x: common.utils.size2str(x))
        pbar = pb.ProgressBar(
            widgets=[
                pb.Bar(), ' ',
                pb.Percentage(), ' | ',
                size_label, ' | ',
                pb.Timer(), ' | ',
                pb.ETA(), ' |',
                pb.FileTransferSpeed()
            ],
            maxval=1
        )
        pbar.start()

        try:
            response = requests.get(url, stream=True, timeout=cls.LOAD_RESOURCE_TIMEOUT, verify=False)
            response.raise_for_status()
        except requests.exceptions.Timeout as ex:
            logger.error("Timeout fetch resource: %s", ex)
            return False
        except requests.exceptions.HTTPError as ex:
            logger.error("Error fetch resource: %s", ex)
            return False
        except:
            raise

        pbar.maxval = int(response.headers.get('content-length'))
        pbar.start()
        for chunk in response.iter_content(chunk_size=1024):
            pbar.update(min(pbar.currval + len(chunk), pbar.maxval))
            if chunk:
                fileobj.write(chunk)
                fileobj.flush()
        pbar.finish()
        logger.debug('Resource has been successfully fetched.')
        return True

    @property
    def running_in_venv(self):
        executable = sys.executable
        real_prefix = getattr(sys, 'real_prefix', '')
        return not executable.startswith(self.SKYNET_PREFIX) and real_prefix.startswith(self.SKYNET_PREFIX)

    def self_prepared_environment(self, local_exe_path, force_env_setup):
        """
        Checks packages in current or expected environment and returns path to suitable one.

        :param force_env_setup: try setting the environment up even if there's no need
        :param local_exe_path: path to virtual environment executable (if exists)
        :return: path to environment executable
        """
        env = os.environ.copy()
        env["PYTHONPATH"] = ":".join((self.SKYNET_PREFIX, os.path.dirname(self._sandbox_dir)))
        sp.check_call(
            [
                local_exe_path,
                "-c",
                textwrap.dedent("""
                    from sandbox.devbox import bootstrap
                    bootstrap.BootEnvironment({!r}, {!r}).check_requirements({})
                    """.format(self._runtime_dir, self._sandbox_dir, force_env_setup)
                )
            ],
            cwd=self._sandbox_dir,
            env=env,
        )
        return local_exe_path

    def _prepare_executable_component(self, env):
        """
        :param env: environment to prepare
        :type env: `LocalEnvironment`
        :return: path to executable
        """
        if not env.need_to_check():
            env.refresh_update_stamp()
            return env.exe_path

        cz = common.console.AnsiColorizer()
        local_version = env.get_local_version()
        resource = env.resource
        resource_type = resource.__class__.__name__
        try:
            with common.console.LongOperation("Checking Sandbox storage for recent {}".format(resource_type)):
                resource.initialize()
        except Exception as ex:
            print(cz.red("Error occurred while checking {}: {}".format(resource_type, ex)))
            return env.exe_path

        if resource.version == local_version:
            env.refresh_update_stamp()
            return env.exe_path
        logger.debug("Found new resource of version=%s local version=%s", env.resource.version, local_version)
        logger.info("Downloading %s via %s", resource, resource.url)
        update_is_done = self.update_environment(resource.url, env)

        if update_is_done:
            env.refresh_update_stamp(update_happened=update_is_done)
            return env.exe_path

        print(cz.red("{} was not prepared. Exiting.".format(env.__class__.__name__)))
        sys.exit(1)

    def prepare_serviceapi(self):
        self._prepare_executable_component(
            ServiceApiEnv(
                os.path.join(self._runtime_dir, "serviceapi"),
                self._sandbox_revision,
            ),
        )

    def prepare_preexecutor(self):
        self._prepare_executable_component(
            PreexecutorEnv(
                os.path.join(self._runtime_dir, "preexecutor"),
                self._sandbox_revision,
            ),
        )

    def prepare_taskbox(self):
        self._prepare_executable_component(
            TaskboxEnv(
                os.path.join(self._runtime_dir, "taskbox"),
                self._sandbox_revision,
            ),
        )

    def prepare_tvmtool(self):
        self._prepare_executable_component(
            TVMToolEnv(
                os.path.join(self._runtime_dir, "tvmtool"),
                self._sandbox_revision,
            ),
        )

    def prepare_py3_sources(self):
        self._prepare_executable_component(
            Py3SourcesEnv(
                os.path.join(self._runtime_dir, "py3_sources"),
                self._sandbox_revision,
            )
        )

    def prepare_virtual_environment(self, force_env_setup):
        """
        Chief executor managing all branches of execution.

        :return: path to executable in suitable environment.
        """
        env_exists = self.running_in_venv or self._local_venv.exists
        cz = common.console.AnsiColorizer()
        local_version = self._local_venv.get_revision()
        if env_exists and local_version is None:
            print(cz.blue("Detected manually managed virtual environment."))
            if self.running_in_venv:
                if force_env_setup or sys.stdout.isatty():
                    self.check_requirements(force_env_setup)
                return sys.executable
            return self.self_prepared_environment(self._local_venv.exe_path, force_env_setup)

        if not (sys.stdout.isatty() or force_env_setup) or not self._local_venv.need_to_check():
            self._local_venv.refresh_update_stamp()
            return self._local_venv.exe_path

        try:
            with common.console.LongOperation("Checking Sandbox storage for recent virtual environment resource"):
                sandbox_resource = self._local_venv.resource
                sandbox_resource.initialize()
        except Exception as ex:
            print(cz.red("Error occurred while checking virtual environment resource: {}".format(ex)))
            return self._local_venv.exe_path
        if not sandbox_resource.exists:
            print(cz.red("Could not find virtual environment bundle for current platform."))
        elif sandbox_resource.version == local_version:
            self._local_venv.refresh_update_stamp()
            return self._local_venv.exe_path

        if not env_exists:
            print(cz.red("No local virtual environment found."))
        else:
            print(cz.yellow("Local virtual environment is outdated."))

        update_is_done = False
        if sandbox_resource.exists:
            logger.info("Downloading {} via {}".format(sandbox_resource, sandbox_resource.url))
            update_is_done = self.update_environment(sandbox_resource.url, self._local_venv)

        if update_is_done or env_exists:
            self._local_venv.refresh_update_stamp(update_happened=update_is_done)
            return self._local_venv.exe_path

        print("Virtual environment was not prepared. Exiting.")
        sys.exit(1)

    @classmethod
    def get_exec_environment(cls, sandbox_dir):
        """
        Makes copy of environ with updating PYTHONPATH value to enable search libraries
        in Skynet folder.

        :return: dict with environ mappings
        """
        skynet_path = cls.SKYNET_PREFIX
        python_path_key = "PYTHONPATH"
        env_copy = os.environ.copy()
        python_path = env_copy.get(python_path_key)
        paths = python_path.split(os.pathsep) if python_path else []
        for extra_path in (sandbox_dir, skynet_path):
            if extra_path not in paths:
                paths = [extra_path] + paths
        env_copy[python_path_key] = os.pathsep.join(paths)
        env_copy["PYTHONUSERBASE"] = os.path.expanduser("~/.sandbox/local")
        return env_copy
