import os
import py
import sys
import pwd
import json
import logging
import textwrap
import platform
import subprocess as sp

import requests

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.devbox import utils

logger = logging.getLogger("sandbox")


def print_and_exit(message, exit_code=1):
    logger.error(message)
    sys.exit(exit_code)


def get_arc_repo(sandbox_dir):
    parent_sandbox_dir = os.path.dirname(sandbox_dir)
    try:
        mounts = sp.check_output(
            ["findmnt", "-nl", "-o", "fstype,target"],
            stderr=sp.STDOUT
        ).strip()
    except sp.CalledProcessError as exc:
        print_and_exit("Unable to find mounts: {}".format(exc))
    arc_repo = ""
    for line in mounts.split("\n"):
        fstype, mount = line.split(None, 1)
        if fstype == "fuse.arc" and mount == parent_sandbox_dir:
            arc_repo = parent_sandbox_dir
            break
    return arc_repo


def make_runtime_dir(sandbox_dir, arc_repo):
    if arc_repo:
        runtime_dir = os.path.join(os.path.dirname(os.path.dirname(sandbox_dir)), "runtime_data")
    else:
        runtime_dir = os.path.join(os.path.dirname(sandbox_dir), "runtime_data")
    return runtime_dir


class SandboxConfig(object):
    """
    Class to retrieve the configuration file for Sandbox server and client.
    """

    class LXCNetworkPicker(object):
        PLATFORMNETS_CONTENTS_URL = "https://racktables.yandex.net/export/expand-fw-macro.php?macro=_PLATFORMNETS_"

        VENDOR_FILE = "/sys/class/dmi/id/sys_vendor"
        QEMU_MARKER = "QEMU"

        @common.utils.classproperty
        def preferred_network_type(cls):
            import netaddr

            cz = common.console.AnsiColorizer()

            # SANDBOX-7187
            try:
                with open(cls.VENDOR_FILE, "r") as fileobj:
                    if fileobj.read().strip() == cls.QEMU_MARKER:
                        print(cz.blue("We're inside a QEMU virtual machine; using NAT schema for LXC"))
                        return ctm.Network.Type.NAT
            except IOError:
                pass

            try:
                response = requests.get(cls.PLATFORMNETS_CONTENTS_URL)
                response.raise_for_status()
                networks = response.text.splitlines()

            except requests.RequestException:
                command = " ".join((
                    os.path.join(os.path.dirname(os.path.dirname(__file__)), "sandbox"),
                    "set", "--key", "client.lxc.network.type", "--value", ctm.Network.Type.NAT
                ))

                print(cz.yellow("Can't determine proper network settings for LXC containers"))
                print(
                    cz.yellow("If the host is allocated in QYP, execute \"") +
                    cz.green(command) +
                    cz.yellow("\" before starting Sandbox")
                )
                networks = []

            if networks:
                ipaddrs = sp.check_output("ip -o -6 addr sh scope global".split()).splitlines()
                for iface in ipaddrs:
                    iface = iface.split()
                    if "lxc" not in iface[1] and "vlan" not in iface[1] and not iface[3].startswith("f"):
                        ip = netaddr.IPNetwork(iface[3])
                        if any(
                            ip in network
                            for network in map(netaddr.IPNetwork, networks)
                        ):
                            print(
                                cz.blue("The host belongs to _PLATFORMNETS_ firewall macro, using NAT schema for LXC")
                            )
                            return ctm.Network.Type.NAT

            print(cz.blue("Using MACVLAN schema for LXC"))
            return ctm.Network.Type.MACVLAN

    def __init__(
        self, sandbox_dir=None, runtime_data_dir=None, sandbox_tasks_dir=None,
        serviceapi_port=None, web_server_port=None, client_port=None, tvmtool_port=None,
        fileserver_port=None, serviceq_port=None, taskbox_port=None,
        mongo_uri=None, username=None, porto=False, arc_repo=""
    ):
        self.node_id = common.config.Registry().this.id
        self.sandbox_dir = sandbox_dir
        self.runtime_data_dir = runtime_data_dir
        self.sandbox_tasks_dir = sandbox_tasks_dir if sandbox_tasks_dir else "sandbox_tasks"
        self.arc_src_dir = os.path.join(self.runtime_data_dir, "srcdir")
        self.arc_test_data_dir = os.path.join(self.runtime_data_dir, "testdata")
        self.tasks_dir = os.path.join(self.runtime_data_dir, "tasks")
        self.ramdrive = os.path.join(self.runtime_data_dir, "ramdrive")
        self.environment_dir = os.path.join(self.runtime_data_dir, "environment")
        self.container = os.path.join(self.runtime_data_dir, "privileged")
        self.logs_dir = os.path.join(self.runtime_data_dir, "logs")
        self.tests_dir = os.path.join(self.runtime_data_dir, "tests")
        self.configs_dir = os.path.join(self.runtime_data_dir, "configs")
        self.resources_dir = os.path.join(self.runtime_data_dir, "resources")
        self.run_dir = os.path.join(self.runtime_data_dir, "run")
        self.server_pid_file = os.path.join(self.run_dir, "server.py.pid")
        self.client_pid_file = os.path.join(self.run_dir, "client.py.pid")
        self.fileserver_pid_file = os.path.join(self.run_dir, "fileserver.py.pid")
        self.bin_dir = os.path.join(self.runtime_data_dir, "bin")
        self.local_settings_file = os.path.join(self.configs_dir, "settings.yaml")
        self.tvm_config_file = os.path.join(self.configs_dir, "tvm.conf")
        self.hostname = common.config.Registry().this.fqdn
        self.username = username if username else pwd.getpwuid(os.geteuid())[0]
        self.serviceapi_port = int(serviceapi_port) if serviceapi_port else None
        self.arc_repo = arc_repo

        def choose_port(port, shift):
            if port:
                return int(port)
            else:
                return (self.serviceapi_port + shift) if self.serviceapi_port else None

        self.client_port = choose_port(client_port, 1)
        self.serviceq_port = choose_port(serviceq_port, 2)
        self.web_server_port = choose_port(web_server_port, 3)
        self.taskbox_port = choose_port(taskbox_port, 4)
        self.fileserver_port = choose_port(fileserver_port, 10)
        self.tvmtool_port = choose_port(tvmtool_port, 11)

        self.mongo_uri = mongo_uri or self.default_mongo_uri()

        try:
            with open(os.devnull, "w") as devnull:
                self.root = not sp.call(["sudo", "-n", "/bin/true"], stderr=devnull)
        except OSError:
            self.root = False

        self.lxc = self.root and os.path.exists("/var/lib/lxc") and not self.apparmor_enabled
        if self.apparmor_enabled:
            logger.warning("Detected AppArmor installed and enabled, LXC support is unavailable.")
        self.porto = porto and self.root and self.porto_version_is_ok()

        if self.lxc and self.porto:
            logger.error("Both LXC and yandex-porto are available, choosing LXC")
            self.porto = False

    @common.utils.singleton_classproperty
    def apparmor_enabled(cls):
        try:
            sp.check_call(["apparmor_status", "--enabled"])
            return True
        except (sp.CalledProcessError, OSError):
            return False

    @staticmethod
    def porto_version_is_ok():
        # Check yandex-porto is installed and version is OK
        p = sp.Popen(["dpkg-query", "-W", "yandex-porto"], stdout=sp.PIPE, stderr=sp.PIPE)
        out, err = p.communicate()

        if p.returncode == 0:
            _, version = out.split()
            if version >= "4.12.20":
                return True
            logger.error("yandex-porto is installed, but version >=4.13 is required")

        return False

    @property
    def sandbox_dir(self):
        return self._sandbox_dir

    @sandbox_dir.setter
    def sandbox_dir(self, directory):
        if directory:
            self._sandbox_dir = os.path.abspath(directory)
        else:
            self._sandbox_dir = py.path.local(sys.argv[0]).dirpath().dirname
        self._sandbox_dir = os.path.realpath(self._sandbox_dir)

    @property
    def runtime_data_dir(self):
        return self._runtime_data_dir

    @runtime_data_dir.setter
    def runtime_data_dir(self, directory):
        if directory:
            if os.path.isabs(directory):
                self._runtime_data_dir = directory
            else:
                self._runtime_data_dir = os.path.join(os.path.dirname(self.sandbox_dir), directory)
        else:
            self._runtime_data_dir = os.path.join(os.path.dirname(self.sandbox_dir), 'runtime_data')
        self._runtime_data_dir = os.path.realpath(self._runtime_data_dir)

    @property
    def tasks_dir(self):
        return self._tasks_dir

    @tasks_dir.setter
    def tasks_dir(self, directory):
        self._tasks_dir = os.path.abspath(directory)

    @property
    def sandbox_tasks_dir(self):
        return self._sandbox_tasks_dir

    @sandbox_tasks_dir.setter
    def sandbox_tasks_dir(self, directory):
        if not os.path.isabs(directory):
            directory = os.path.join(os.path.dirname(self.sandbox_dir), directory)
        self._sandbox_tasks_dir = directory

    def get_mongodb_connection_url(self):
        if self.mongo_uri:
            return "{0}".format(self.mongo_uri)
        else:
            return "file://{0}".format(os.path.join(self.runtime_data_dir, 'mongo_default.cnf'))

    def get_version_config(self):
        settings = common.config.Registry()
        return textwrap.dedent(
            """
            version:
              current: {}
            """.format(settings.version.actual)
        )

    @staticmethod
    def get_tvm_config():
        from sandbox import deploy
        tvmtool = deploy.tvm.TVMTool
        source = tvmtool.SOURCE[0]
        config = {
            "BbEnvType": tvmtool.BLACKBOX,
            "clients": {
                source.alias: {
                    "self_tvm_id": source.tvm_id,
                    "secret": "fake_secret",
                    "dsts": {
                        app.alias: {"dst_id": app.tvm_id}
                        for app in tvmtool.TARGETS
                    }
                }
            }
        }
        # Add destinations as clients to check their fake TVM tickets
        for target in tvmtool.TARGETS:
            config["clients"][target.alias] = {
                "self_tvm_id": target.tvm_id,
                "secret": "fake_secret",
                "dsts": {}
            }
        return json.dumps(config, indent=4)

    def default_mongo_uri(self):
        if not utils.is_port_free(27017):
            return "mongodb://localhost/sandbox_{}".format(self.username)

    def get_common_config(self):
        config = textwrap.dedent(
            """
            this:
              id: "{this}"
            common:
              dirs:
                service: "{service_root}"
                data: "{runtime_data_dir}"
                runtime: "${{common.dirs.data}}/run"
              tvm:
                port: {tvmtool_port}
                access_token: {tvm_access_token}
              py3_sources_binary: "${{common.dirs.data}}/py3_sources/py3_sources"
            """.format(
                this=common.config.Registry().this.id,
                service_root=self.sandbox_dir,
                runtime_data_dir=self.runtime_data_dir,
                tvmtool_port=self.tvmtool_port,
                tvm_access_token=common.utils.random_string(32),
            )
        )
        return config

    def get_devbox_config(self):
        config = textwrap.dedent(
            """
            devbox:
              arc_repo: {arc_repo}
            """.format(
                arc_repo=self.arc_repo,
            )
        )
        return config

    def get_server_config(self):
        """
        Retrieve content of configuration file for Sandbox server.
        """
        config = textwrap.dedent(
            """
            server:
              log:
                root: "${{common.dirs.data}}/logs"
              api:
                port: {serviceapi_port}
                workers: 16
                enable_stats: false
              web:
                static:
                  root_path: "${{common.dirs.service}}/web"
                address:
                  host: "{hostname}"
                  port: {web_server_port}
                root_path: "${{server.web.static.root_path}}/templates"

              encryption_key: "file://${{common.dirs.data}}/.vault_key"
              storage_hosts: []

              services:
                log:
                  rotate: true

                mailman:
                  release_subscribers: "{username}@yandex-team.ru"
                serviceq:
                  enabled: true
                serviceapi:
                  enabled: true
                taskbox:
                  enabled: true
                packages_updater:
                  enabled: true
                tvmtool:
                  enabled: true

              mongodb:
                connection_url: "{connection_url}"
            """.format(
                hostname=self.hostname,
                serviceapi_port=self.serviceapi_port,
                web_server_port=self.web_server_port,
                username=self.username,
                connection_url=self.get_mongodb_connection_url(),
            )
        )
        return config

    def get_serviceq_config(self):
        """
        Retrieve content of configuration file for Service Q.
        """
        config = textwrap.dedent(
            """
            serviceq:
              zookeeper:
                enabled: false

              log:
                root: "${{common.dirs.data}}/logs"

              server:
                server:
                  host: ""
                  port: {port}

                mongodb:
                  connection_url: ${{server.mongodb.connection_url}}
            """.format(
                port=self.serviceq_port
            )
        )
        return config

    @staticmethod
    def get_proxy_config():
        config = textwrap.dedent(
            """
            proxy:
              logging:
                handlers:
                  file:
                    filename: "${common.dirs.data}/logs/proxy.log"

            """
        )
        return config

    @staticmethod
    def get_agentr_config():
        """ Retrieve content of configuration file for AgentR. """
        config = textwrap.dedent(
            """
            agentr:
              log:
                root: "${common.dirs.data}/logs"

              daemon:
                server:
                  unix: "${common.dirs.runtime}/agentr.sock"
            """
        )
        return config

    def get_client_config(self):
        """
        Retrieve content of configuration file for Sandbox client.
        """
        from sandbox.deploy import client
        client_launcher = client.ClientLauncher(
            1, platform.system().lower(), ["sandbox1_client"], None, "", self.hostname, None
        )
        client_launcher.get_pack = lambda *_: os.path.dirname(self.sandbox_dir)
        tags = {ctc.Tag.GENERIC, ctc.Tag.POSTEXECUTE} | client_launcher.client_tags
        if self.lxc:
            tags.add(ctc.Tag.LXC)
        config = textwrap.dedent(
            """
            client:
              tags: [{tags}]
              xmlrpc_url: "http://{hostname}:{port}/sandbox/xmlrpc"
              rest_url: "http://{hostname}:{port}/api/v1.0"
              port: {client_port}
              idle_time: 15

              dirs:
                data: "${{common.dirs.data}}"

              tasks:
                code_dir: "{tasks_code_dir}"

              skynet_key:
                path: null

              sdk:
                svn:
                  confdir: "{userhome}/.subversion"
                  arcadia:
                    trunk: "arcadia:/arc/trunk/arcadia"
                    rw:
                      user: "{username}"
                    ro:
                      user: "{username}"

              fileserver:
                port: {fileserver_port}
              auto_cleanup:
                free_space_threshold: 0
                hard_free_space_threshold: 0
              lxc:
                enabled: {lxc_enabled}
                dirs:
                  root: "{userhome}"
                mount_tmp_dir: true
                network:
                  type: "{lxc_network_type}"
              porto:
                enabled: {porto_enabled}
                mount_tmp_dir: true
                network:
                  type: "{porto_network_type}"
            """.format(
                hostname=self.hostname,
                port=self.serviceapi_port,
                tasks_code_dir=self.sandbox_tasks_dir,
                username=self.username,
                userhome=os.path.realpath(os.path.expanduser("~" + self.username)),
                client_port=self.client_port,
                fileserver_port=self.fileserver_port,
                lxc_enabled=str(self.lxc).lower(),
                lxc_network_type=self.LXCNetworkPicker.preferred_network_type,
                porto_enabled=str(self.porto).lower(),
                porto_network_type=ctm.Network.Type.NAT,
                tags=", ".join(map(str, tags))
            )
        )
        return config

    def get_taskbox_config(self):
        """ Retrieve content of configuration file for Taskbox. """
        config = textwrap.dedent(
            """
            taskbox:
              log:
                root: "${{server.log.root}}/taskbox"
              dispatcher:
                server:
                  port: {port}
            """.format(
                port=self.taskbox_port,
            )
        )
        return config

    def make_directories(self):
        """
        Make directories to launch Sandbox.
        """
        for dir_name, path in (
            ("runtime directory", self.runtime_data_dir),
            ("logs directory", self.logs_dir),
            ("configs directory", self.configs_dir),
            ("tasks directory", self.tasks_dir),
            ("directory for arcadia sources", self.arc_src_dir),
            ("directory for arc cache", os.path.join(self.arc_src_dir, "arc_vcs")),
            ("directory for tests", self.tests_dir),
            ("directory for arcadia test data", self.arc_test_data_dir),
            ("directory for resources", self.resources_dir),
            ("run directory", self.run_dir),
            ("ramdrive directory", self.ramdrive),
            ("environment directory", self.environment_dir),
        ):
            if not os.path.exists(path):
                logger.debug("Create %s %s", dir_name, path)
                os.makedirs(path)

    def make_lxc_directories(self):
        if not self.lxc:
            return
        settings = common.config.Registry()
        logger.debug("Ensure required directories for LXC containers support")
        sp.check_call([
            "sudo", "mkdir", "-p",
            settings.client.lxc.rootfs.basedir, "/opt/skynet",
            "/place/berkanavt/supervisor", "/place/vartmp", "/place/coredumps"
        ])
        if not os.path.exists(self.container):
            logger.debug('Create privileged container directory %s' % self.container)
            os.makedirs(self.container)
        if not os.path.exists("/var/lxc/templates"):
            logger.debug("Create templates directory %r", "/var/lxc/templates")
            sp.check_call(["sudo", "mkdir", "-p", "/var/lxc/templates"])

    def create_config(self, server=True):
        """
        Write content into configuration files.
        """
        version_config = self.get_version_config()
        devbox_config = self.get_devbox_config()
        common_config = self.get_common_config()
        server_config = self.get_server_config()
        serviceq_config = self.get_serviceq_config()
        agentr_config = self.get_agentr_config()
        client_config = self.get_client_config()
        taskbox_config = self.get_taskbox_config()
        proxy_config = self.get_proxy_config()
        with open(self.local_settings_file, 'w') as f:
            f.write(version_config)
            f.write(devbox_config)
            f.write("\n")
            f.write(common_config)
            f.write('\n')
            if server:
                f.write(server_config)
                f.write(serviceq_config)
            f.write(agentr_config)
            f.write(taskbox_config)
            f.write(client_config)
            f.write(proxy_config)

        with open(self.tvm_config_file, "w") as f:
            f.write(self.get_tvm_config())

    @staticmethod
    def add_config_file_path_to_environment(config_path):
        """
        Adds config file path to system environment to use it with imports the Sandbox settings.

        :param config_path: directory path with configuration file.
        :return: old configuration file path
        """
        old_value = SandboxConfig.get_config_file_path_from_environment()
        if not os.path.isfile(config_path):
            raise Exception("Incorrect config file path: %s", config_path)
        os.environ['SANDBOX_CONFIG'] = os.path.realpath(config_path)
        return old_value

    @staticmethod
    def get_config_file_path_from_environment():
        """
        Gets path of directory with configuration path from system environment.

        :return: directory path if variable found, else None.
        """
        return os.environ.get('SANDBOX_CONFIG')

    @classmethod
    def get_config_from_file(cls, config_file):
        SandboxConfig.add_config_file_path_to_environment(config_file)
        settings = common.config.Registry()
        settings.reload()
        if settings.version.current < settings.version.actual:
            message = (
                "Config file version is too old (format {version}), please execute:\n\t"
                "rm {config_file}\n\t"
                "{executable} setup -p {port} --mongo-uri {uri} -r {runtime}"
            ).format(
                version=settings.version.current,
                config_file=config_file,
                executable=os.path.join(
                    os.path.dirname(os.path.dirname(sys.argv[0])),  # sandbox directory (argv[0] points to ctl_impl.py)
                    "sandbox"
                ),
                port=settings.server.api.port,
                uri=settings.server.mongodb.connection_url,
                runtime=settings.client.dirs.data,
            )
            print_and_exit(message)

        sandbox_config = SandboxConfig(
            sandbox_tasks_dir=settings.client.tasks.code_dir,
            runtime_data_dir=settings.client.dirs.data,
            serviceapi_port=settings.server.api.port,
            web_server_port=settings.server.web.address.port,
            client_port=settings.client.port,
            fileserver_port=settings.client.fileserver.port,
            tvmtool_port=settings.common.tvm.port,
        )

        sandbox_config.logs_dir = settings.server.log.root
        sandbox_config.environment_dir = settings.client.tasks.env_dir
        sandbox_config.arc_src_dir = settings.client.vcs.dirs.base_cache
        sandbox_config.arc_test_data_dir = settings.client.vcs.dirs.tests_data
        sandbox_config.static_dir = settings.server.web.static.root_path
        sandbox_config.tasks_dir = settings.client.tasks.data_dir
        sandbox_config.ramdrive = settings.client.tasks.ramdrive
        sandbox_config.tests_dir = os.path.join(os.path.dirname(sandbox_config.logs_dir), 'tests')
        sandbox_config.configs_dir = os.path.dirname(config_file)
        sandbox_config.run_dir = settings.client.dirs.run
        sandbox_config.server_pid_file = os.path.join(sandbox_config.run_dir, 'server.py.pid')
        sandbox_config.client_pid_file = os.path.join(sandbox_config.run_dir, 'client.py.pid')
        sandbox_config.fileserver_pid_file = os.path.join(sandbox_config.run_dir, 'fileserver.py.pid')
        sandbox_config.local_settings_file = config_file
        sandbox_config.hostname = settings.server.web.address.host
        sandbox_config.mongo_uri = settings.server.mongodb.connection_url
        sandbox_config.porto = settings.client.porto.enabled
        return sandbox_config


def get_config(options):
    if getattr(options, 'config', None):
        logger.debug('Get Sandbox config path from ./sandbox script options.')
        config_file = os.path.abspath(os.path.expanduser(options.config))
        logger.debug('Sandbox config: %s' % config_file)
        if os.path.isfile(config_file):
            return SandboxConfig.get_config_from_file(config_file)
        else:
            print_and_exit('ERROR: file does not exist.')
    sandbox_config_file = SandboxConfig.get_config_file_path_from_environment()
    if sandbox_config_file:
        logger.debug('Get Sandbox config path from environment variable.')
        logger.debug('Sandbox config file: %s', sandbox_config_file)
        if os.path.isfile(sandbox_config_file):
            return SandboxConfig.get_config_from_file(sandbox_config_file)
        else:
            print_and_exit(
                'ERROR: file does not exist.\n'
                'Change "SANDBOX_CONFIG" environment variable or remove it (command: unset SANDBOX_CONFIG). '
                'You can also specify path to a config file with --config (-c) option'
            )
    sandbox_dir = py.path.local(sys.argv[0]).dirpath().dirname
    if getattr(options, 'runtime', None):
        runtime_dir = os.path.abspath(os.path.expanduser(options.runtime))
    elif os.path.exists(os.path.join(sandbox_dir, 'runtime_data')):
        runtime_dir = os.path.join(sandbox_dir, 'runtime_data')
    else:
        arc_repo = get_arc_repo(sandbox_dir)
        runtime_dir = make_runtime_dir(sandbox_dir, arc_repo)
    runtime_config_file = os.path.join(runtime_dir, 'configs', 'settings.yaml')
    config_file = os.path.join(sandbox_dir, 'etc', 'settings.yaml')
    if os.path.exists(config_file):
        logger.debug('Sandbox config: %s' % config_file)
        return SandboxConfig.get_config_from_file(config_file)
    elif os.path.exists(runtime_config_file):
        logger.debug('Sandbox config: %s' % runtime_config_file)
        return SandboxConfig.get_config_from_file(runtime_config_file)
    else:
        print_and_exit(
            'Sandbox is not configured. '
            'Try to use "./sandbox setup" to setup Sandbox or specify path '
            'to Sandbox config file with --config option.'
        )


def print_config(args):
    config = get_config(args)
    sandbox_config_file = config.local_settings_file
    if os.path.exists(sandbox_config_file):
        logger.debug("Config file: %s" % sandbox_config_file)
        sys.stdout.write(open(sandbox_config_file).read())
    else:
        logger.debug("Cannot find config file %s" % sandbox_config_file)


def set_config_option(args):
    import ast
    import yaml
    import types

    sandbox_config_file = get_config(args).local_settings_file
    if not os.path.exists(sandbox_config_file):
        print_and_exit("Cannot find config file '{}'".format(sandbox_config_file))

    with open(sandbox_config_file, "rb") as f:
        config = yaml.load(f)

    try:
        path, target_key = args.key.rsplit(".", 1)
        target_node = reduce(lambda d, k: d[k], path.split("."), config)
    except ValueError:  # top-level setting
        target_node = config
        target_key = args.key
    except KeyError:
        print_and_exit("Incorrect setting {!r}".format(args.key))

    previous_value = target_node.get(target_key)
    if isinstance(previous_value, (list, dict)):
        print_and_exit("Can't set complex values (edit the config itself if you really mean it)")

    try:
        args.value = ast.literal_eval(args.value)
        if not isinstance(args.value, (basestring, bool, int, long, float, types.NoneType)):
            print_and_exit("Only primitive Python values are allowed (`{}` given)".format(type(args.value).__name__))
    except Exception:
        pass

    target_node[target_key] = args.value
    with open(sandbox_config_file, "wb") as f:
        yaml.dump(config, f, default_flow_style=False)
    print("Set {!r}={!r} (was: {!r})".format(args.key, args.value, previous_value))


def get_runtime_data_path(options):
    _logger_disabled = logger.disabled
    logger.disabled = True
    try:
        config = get_config(options)
        return config.runtime_data_dir
    except SystemExit:
        pass
    finally:
        logger.disabled = _logger_disabled
    sandbox_dir = py.path.local(sys.argv[0]).dirpath().dirname
    if getattr(options, "runtime", None):
        runtime_dir = py.path.local(os.path.expanduser(options.runtime))
    else:
        arc_repo = get_arc_repo(sandbox_dir)
        runtime_dir = make_runtime_dir(sandbox_dir, arc_repo)

    return str(runtime_dir)
