#!/skynet/python/bin/python -W ignore::UserWarning
# coding: utf-8

from __future__ import print_function
from __future__ import absolute_import

import os
import py
import sys
import json
import time
import yaml
import psutil
import shutil
import signal
import socket
import logging
import getpass
import hashlib
import argparse
import dateutil
import pathlib2
import tempfile
import urlparse
import datetime as dt
import textwrap
import distutils
import subprocess

SANDBOX_DIR = reduce(lambda p, _: os.path.dirname(p), xrange(2), os.path.abspath(__file__))
sys.path = ["/skynet", os.path.dirname(SANDBOX_DIR), SANDBOX_DIR] + sys.path

from sandbox import common
common.encoding.setup_default_encoding()

from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.common.types import user as ctu
from sandbox.common.types import client as ctc
from sandbox.common.types import resource as ctr
common.import_hook.setup_sandbox_namespace()

from sandbox import sandboxsdk as sdk1
from sandbox import sdk2

from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox.database import upgrade as db_upgrade

import sandbox.devbox.utils as dev_utils
import sandbox.devbox.config as dev_config
import sandbox.devbox.mapping as dev_mapping


logger = logging.getLogger("sandbox")
logger.propagate = False
h = logging.StreamHandler()
h.setFormatter(logging.Formatter("%(asctime)s %(levelname)-6s (%(module)s) %(message)s"))
logger.addHandler(h)

MAX_CLIENTS_NUMBER = 10000
MONGODB_INSTALLATION_LINK = "https://docs.mongodb.com/manual/tutorial/install-mongodb-on-ubuntu/"
QYP_VM_INSTRUCTIONS = "https://wiki.yandex-team.ru/sandbox/allocate-host/"


def print_and_exit(message, exit_code=1):
    logger.error(message)
    sys.exit(exit_code)


class SandboxAPI(object):

    enable_auth = False

    @classmethod
    def rest_client(cls, source=None, **kws):
        if source:
            u = urlparse.urlparse(source, scheme="http")
            if not u.netloc:
                u = u._replace(netloc=u.path)
            kws["base_url"] = urlparse.urlunparse(u._replace(path="/api/v1.0"))
        else:
            kws["base_url"] = common.rest.Client.DEFAULT_BASE_URL
            if cls.enable_auth:
                kws["auth"] = cls.auth
        return common.rest.Client(**kws)


class RemoteChannel(object):
    def __init__(self):
        # Temporary patch channel.sandbox remote client to search compatible resources in production installation.
        self.remote = SandboxAPI.rest_client()

        self.channel = sdk1.channel
        self.local = self.channel.channel

    def __enter__(self):
        self.channel.channel = type('FakeRemoteLocal', (object,), {
            "rest": sdk1.sandboxapi.SandboxRest(common.rest.Client.DEFAULT_BASE_URL),
            'task': type('FakeTask', (object,), {'platform': None}),
            'sandbox': property(lambda _: self.remote),
        })()
        return self.remote

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.channel.channel = self.local


def _prepare_env(options, sandbox_config, force, target_platform=None):
    common.projects_handler.load_project_types(reuse=True)

    with common.rest.DispatchedClient as dispatch, RemoteChannel() as remote:
        dispatch(SandboxAPI.rest_client)
        try:
            resources = [
                env.compatible_resource
                for env in (
                    sdk1.environments.ArcEnvironment(),
                    sdk1.environments.SvnEnvironment(platform=target_platform),
                    sdk1.environments.GCCEnvironment(),
                    sdk1.environments.GDBEnvironment("8.0.1", platform=target_platform),
                )
            ]
        except Exception as ex:
            print("Cannot find appropriate resource(s): {!s}".format(ex))
            return

    for res in resources:
        if force:
            _add_task(options, res.task_id, remote, sandbox_config.node_id, sync=True, force=True)
        else:
            _add_task(options, res.task_id, remote, None, sync=False)


def _sync_container_related_resources(resource_type, options, sandbox_config, force):
    rest = SandboxAPI.rest_client()
    resources = rest.resource.read(
        type=resource_type, attrs={"released": ctt.ReleaseStatus.STABLE}, owner="SANDBOX", limit=100, order="-id",
    )
    found = {}
    for r in resources["items"]:
        pl = r["attributes"].get("platform")
        if pl and pl not in found:
            found[pl] = r
            _prepare_env(options, sandbox_config, force, target_platform=pl)
    resources = rest.resource.read(
        type="SANDBOX_DEPS", attrs={"released": ctt.ReleaseStatus.STABLE}, owner="SANDBOX", limit=100, order="-id",
    )
    for r in resources["items"]:
        pl = r["attributes"].get("platform")
        if pl in found and pl + "_venv" not in found:
            found[pl + "_venv"] = r
    with RemoteChannel() as remote:
        for tid in sorted(set(_["task"]["id"] for _ in found.itervalues())):
            _add_task(options, tid, remote, None, sync=False)


def _try_generate_encryption_key():
    cz = common.console.AnsiColorizer()
    keyfile_path = common.utils.to_path(common.config.Registry().server.encryption_key)
    if not os.path.exists(keyfile_path):
        mapping.ensure_connection()
        if mapping.Vault.objects.count():
            print(cz.blue("Skip vault key generation (avoid losing existing vault items)"))
            return

        common.crypto.AES.generate_key(keyfile_path)
        print(cz.blue("Vault key written to ") + cz.red(keyfile_path) + cz.blue(" -- make a backup!"))

    else:
        print(cz.blue("Found existing vault key at ") + cz.red(keyfile_path))
    print(cz.blue("Details: https://wiki.yandex-team.ru/sandbox/vault/#local-sandbox"))


def _update_resources(options):
    cz = common.console.AnsiColorizer()
    sandbox_config = dev_config.get_config(options)
    dev_config.SandboxConfig.add_config_file_path_to_environment(sandbox_config.local_settings_file)

    with common.console.LongOperation("Preparing commonly used environment resources") as op:
        op.intermediate("", True)
        _prepare_env(options, sandbox_config, options.force)

    with common.rest.DispatchedClient as dispatch, RemoteChannel() as remote:
        dispatch(SandboxAPI.rest_client)
        res = sdk2.service_resources.ArcadiaApiClient.find(
            attrs={"released": ctt.ReleaseStatus.STABLE}
        ).first()
        _add_task(options, res.task_id, remote, None)

    if sandbox_config.lxc:
        print(cz.blue("LXC support available."))
        with common.console.LongOperation("Post-configuring LXC service") as op:
            op.intermediate("", True)
            subprocess.call(["sudo", "service", "lxc", "start"])
        with common.console.LongOperation("Preparing LXC environment resources") as op:
            op.intermediate("", True)
            _sync_container_related_resources("LXC_CONTAINER", options, sandbox_config, options.force)

    if sandbox_config.porto:
        print(cz.blue("Porto support available."))
        with common.console.LongOperation("Preparing Porto environment resources") as op:
            op.intermediate("", True)
            _sync_container_related_resources("PORTO_LAYER", options, sandbox_config, options.force)

    from sandbox.devbox import bootstrap
    res_version = bootstrap.ResourcesVersion(dev_config.get_runtime_data_path(options))
    res_version.set_updated()


def _mongo_installation_info(config):
    if not config.mongo_uri:
        return (
            "Please create your own MongoDB installation and use --mongo-uri then. You can follow the instructions: {}."
        ).format(MONGODB_INSTALLATION_LINK)


def _verify_setup(config):
    msg = _mongo_installation_info(config)
    if common.os.User.can_root:
        return not msg, msg
    if msg:
        msg = (
            "{message} If you do not have a superuser access, probably you"
            "should get a virtual machine following the instructions {qyp}"
        ).format(message=msg, qyp=QYP_VM_INSTRUCTIONS)
    return True, msg


def setup_sandbox(options, client_only=False):
    """
    Setup Sandbox: create configuration file and all the necessary folders

    :param options: options provided by user via command line, must contain port number
    :param client_only: do not write server-side settings into the configuration file
    :return: 0 if setup is successful
    """

    cz = common.console.AnsiColorizer()
    print(cz.black("Setting up Sandbox..."))
    if not options.port:
        print_and_exit("Error. You need to specify --port (-p) option.")

    # check ipv6 host resolution
    try:
        socket.getaddrinfo("localhost", None, socket.AF_INET6)
    except socket.gaierror:
        print_and_exit("Error: 'localhost' must resolve as ipv6 address. Check /etc/hosts")

    sandbox_dir = py.path.local(sys.argv[0]).dirpath().dirname
    arc_repo = dev_config.get_arc_repo(sandbox_dir)
    logger.info("Sandbox source folder: %s", sandbox_dir)

    try:
        ports = dev_utils.Ports()
        ports.initialize(options.port, dev_utils.get_registered_ports(sandbox_dir))
    except ValueError as er:
        print_and_exit(er.message)

    if not options.runtime:
        runtime_dir = dev_config.make_runtime_dir(sandbox_dir, arc_repo)
    else:
        runtime_dir = os.path.abspath(options.runtime)
    if not os.path.exists(runtime_dir):
        os.makedirs(runtime_dir)
    logger.info("Runtime data folder: %s", runtime_dir)
    tasks_dir = sandbox_dir
    logger.info("Tasks source folder folder: %s", tasks_dir)
    if not os.path.exists(tasks_dir) or not os.path.exists(os.path.join(tasks_dir, "projects")):
        print_and_exit("ERROR: Folder with tasks sources does not exist or there is no projects directory.")
    logger.info("Create Sandbox config and required folders")
    sandbox_config = dev_config.SandboxConfig(
        sandbox_dir=sandbox_dir, runtime_data_dir=runtime_dir, sandbox_tasks_dir=tasks_dir,
        serviceapi_port=ports.serviceapi_port, web_server_port=ports.web_server_port, client_port=ports.client_port,
        fileserver_port=ports.fileserver_port, serviceq_port=ports.serviceq_port, taskbox_port=ports.taskbox_port,
        mongo_uri=options.mongo_uri, porto=options.porto, arc_repo=arc_repo
    )
    continue_setup, verification_message = _verify_setup(sandbox_config)
    if not continue_setup:
        print_and_exit(verification_message)
    if verification_message:
        logger.warning(verification_message)

    conf = sandbox_config.local_settings_file
    if os.path.exists(conf):
        print_and_exit(
            "Configuration file already exists. Cannot setup Sandbox. "
            "Remove file '{conf}' to re-configure it.\n"
            "Execute `{self} info` to determine current host and port or "
            "`{self} clean_all` to completely erase the installation.".format(conf=conf, self="./sandbox")
        )

    sandbox_config.make_directories()
    sandbox_config.create_config(server=not client_only)

    for subpath in ("", "serviceq"):
        conf_link = os.path.join(sandbox_dir, subpath, "etc", "settings.yaml")
        if os.path.lexists(conf_link):
            os.remove(conf_link)
        os.symlink(conf, conf_link)

    sandbox_config = dev_config.get_config(options)
    sandbox_config.make_lxc_directories()
    dev_config.SandboxConfig.add_config_file_path_to_environment(sandbox_config.local_settings_file)
    dev_utils.register_sandbox_install(
        sandbox_dir,
        [
            ports.serviceapi_port, ports.web_server_port, ports.client_port, ports.fileserver_port,
            ports.serviceq_port, ports.taskbox_port,
        ]
    )

    _try_generate_encryption_key()
    _update_resources(options)

    print(cz.green("Setup finished."))
    return 0


def setup_client(options):
    return setup_sandbox(options, client_only=True)


def check_resource_updates(options):
    _update_resources(options)


def is_server_launched(config):
    logger.debug('is_server_launched, config {0}'.format(config))
    import scripts.sandbox_ctl
    return scripts.sandbox_ctl.rc_action('silent_status', 'server') == 0


def is_client_launched(config):
    logger.debug('is_client_launched, config {0}'.format(config))
    import scripts.sandbox_ctl
    return scripts.sandbox_ctl.rc_action('silent_status', 'client') == 0


def clean_logs(args):
    """
    Removes and creates logs directory.
    """
    config = dev_config.get_config(args)
    # delete
    logger.debug('Delete logs folder %s' % config.logs_dir)
    shutil.rmtree(config.logs_dir)
    # create
    logger.debug('Makes logs folder %s' % config.logs_dir)
    os.makedirs(config.logs_dir)
    logger.debug('Done')


def clean_tests(args):
    """
    Removes and creates tests directory
    """
    config = dev_config.get_config(args)
    logger.debug('Delete test folder %s' % config.tests_dir)
    shutil.rmtree(config.tests_dir)
    os.makedirs(config.tests_dir)
    logger.debug('Done')


def clean_db(args):
    """
    Removes DB
    """
    config = dev_config.get_config(args)
    logger.debug("Clean remote db '%s'" % config.mongo_uri)
    conn = mapping.ensure_connection().rw
    db_name = urlparse.urlparse(conn.uri).path.strip("/")
    try:
        conn.connection.drop_database(db_name)
    except mapping.ConnectionFailure as er:
        logger.error("Failed to drop database: %s", er)
    logger.debug("Done")


def _remove_venvs():
    home_dir = os.path.expanduser("~/")
    for name in os.listdir(home_dir):
        if name.startswith("venv.") and name.split(".", 1)[1].isdigit():
            path = os.path.join(home_dir, name)
            logger.debug("Removing %s", path)
            common.fs.chmod_for_path(path, "0777", True)
            shutil.rmtree(path)


def clean_data(args):
    """
    Removes tasks directory
    """
    config = dev_config.get_config(args)
    logger.debug("Deleting folder %s", config.tasks_dir)
    common.fs.chmod_for_path(config.tasks_dir, "0777", True)
    shutil.rmtree(os.path.realpath(config.tasks_dir))
    os.makedirs(config.tasks_dir)
    _remove_venvs()
    logger.debug("Done")


def clean_all(args):
    kill(args)
    clean_db(args)
    config = dev_config.get_config(args)
    logger.debug("Deleting folder %s", config.runtime_data_dir)
    common.fs.chmod_for_path(config.runtime_data_dir, "0777", True)
    shutil.rmtree(os.path.realpath(config.runtime_data_dir))
    dev_utils.register_sandbox_install(py.path.local(sys.argv[0]).dirpath().dirname, None)
    _remove_venvs()
    logger.debug("Done")


def kill(args):
    cz = common.console.AnsiColorizer()
    self_pid = os.getpid()

    black_list = [
        "[sandbox]",
        "tvmtool",
    ]

    if sys.executable:
        black_list.append(sys.executable)

    white_list = [
        "devbox/ctl_impl.py shell",
        "bin/shell.py",
    ]
    kill_list = []

    for process in psutil.process_iter():
        cmdline = " ".join(process.cmdline)

        if process.pid == self_pid or any(_ in cmdline for _ in white_list):
            continue

        if any(_ in cmdline for _ in black_list):
            kill_list.append((process.pid, cmdline))

    if not kill_list:
        print("Nothing to kill")
        return

    print("Killing:")
    for pid, cmdline in kill_list:
        print("{}\t{}".format(pid, cmdline), end=' ')
        try:
            os.kill(pid, signal.SIGKILL)
            print(cz.green("OK"))
        except Exception as exc:
            print(cz.red("FAIL"), repr(exc))

    print("Done")


def get_info(args):
    """
    Gets information about launched client, file server and Sandbox server.
    """
    config = dev_config.get_config(args)
    if is_server_launched(config):
        logger.info(
            "Sandbox server is launched\nURL: http://%s:%s\nLogs: %s",
            config.hostname, config.serviceapi_port, config.runtime_data_dir
        )
    else:
        logger.info("Sandbox server is stopped.")

    if is_client_launched(config):
        logger.info("Sandbox client is launched.")
        logger.info("Sandbox file-server is launched. URL: http://%s:%s", config.hostname, config.client_port)
    else:
        logger.info("Sandbox client is stopped.")


@common.utils.singleton
def remove_precompiled_files(path):
    logger.debug("Removing precompiled python files at %s ...", path)
    for root, _, files in os.walk(path):
        for name in files:
            full_path = os.path.join(root, name)
            if full_path.endswith(".pyc") or full_path.endswith(".pyo"):
                os.remove(full_path)


def remove_previous_tests_data(runtime_data_path):
    tests_common_path = os.path.join(runtime_data_path, "tests")
    if os.path.exists(tests_common_path):
        os.system("chmod u+w -fR {}/*".format(tests_common_path))
        for p in py.path.local(tests_common_path).listdir():
            logging.debug("Removing previous tests data %s", str(p))
            p.remove()


def start_sandbox(args):
    """
    Launch client and Sandbox server
    """
    start_server(args)
    start_client(args)


def start_server(args):
    import scripts.sandbox_ctl
    from sandbox.yasandbox import manager

    config = dev_config.get_config(args)
    remove_precompiled_files(config.sandbox_dir)
    remove_precompiled_files(config.sandbox_tasks_dir)
    manager.use_locally(detain_init=True)
    controller.Settings.initialize()

    launched = is_server_launched(config)
    if not launched:
        if not dev_utils.is_port_free(config.web_server_port):
            print_and_exit('ERROR: Port %s is busy. Cannot launch Sandbox server.' % config.serviceapi_port)
        logger.info('Run Sandbox server...')

        info = _mongo_installation_info(config)
        if info:
            logger.warning(info)

        local_config = config.local_settings_file
        if local_config:
            dev_config.SandboxConfig.add_config_file_path_to_environment(local_config)

        if controller.Settings.state != controller.Settings.DatabaseState.OK:
            import scripts.database_upgrade
            scripts.database_upgrade.main()
            model = controller.Settings.model()
            registered = set(db_upgrade.__all__)
            if len(registered) != len(model.updates.applied.main) or len(registered) != len(model.updates.applied.post):
                print('Fixing list of applied database upgrade scripts.')
                for attr in model.updates.applied:
                    setattr(model.updates.applied, attr, list(set(getattr(model.updates.applied, attr)) & registered))
            model.save()

        scripts.sandbox_ctl.as_script().rc_action('start', 'server')
        # wait while server initializing
        # projects initialization, etc
        time.sleep(3)

    logger.info("Sandbox server is %slaunched. URL: http://%s:%s", "already " if launched else "", config.hostname,
                config.serviceapi_port)


def start_client(args):
    import scripts.sandbox_ctl
    import yasandbox.manager

    config = dev_config.get_config(args)
    config.make_directories()
    remove_precompiled_files(config.sandbox_dir)
    remove_precompiled_files(config.sandbox_tasks_dir)
    yasandbox.manager.use_locally(detain_init=True)
    if not is_client_launched(config):
        check_stop = os.path.join(config.run_dir, "client_check_stop")
        if os.path.exists(check_stop):
            logger.debug('Removing %s ...', check_stop)
            os.remove(check_stop)
        logger.debug('Run Sandbox client...')
        local_config = config.local_settings_file
        if local_config:
            dev_config.SandboxConfig.add_config_file_path_to_environment(local_config)
        scripts.sandbox_ctl.as_script().rc_action('start', 'client')
    else:
        logger.debug('Sandbox client is already launched')


def stop_sandbox(args):
    stop_client(args)
    stop_server(args)


def stop_server(args):
    import scripts.sandbox_ctl
    config = dev_config.get_config(args)
    info = _mongo_installation_info(config)
    if info:
        logger.warning(info)
    dev_config.SandboxConfig.add_config_file_path_to_environment(config.local_settings_file)
    scripts.sandbox_ctl.as_script().rc_action('stop', 'server')


def stop_client(args):
    import scripts.sandbox_ctl
    config = dev_config.get_config(args)
    dev_config.SandboxConfig.add_config_file_path_to_environment(config.local_settings_file)
    scripts.sandbox_ctl.as_script().rc_action('stop', 'client')


def restart_sandbox(args):
    stop_sandbox(args)
    start_sandbox(args)


def restart_server(args):
    stop_server(args)
    start_server(args)


def restart_client(args):
    stop_client(args)
    start_client(args)


def test_sandbox(args, unparsed_args=None):
    import pytest
    mod_dir = py.path.local(sys.argv[0]).dirpath().dirname
    if args.clean:
        remove_precompiled_files(mod_dir)
        remove_previous_tests_data(dev_config.get_runtime_data_path(args))
    with common.fs.WorkDir(mod_dir):
        return pytest.main(dev_utils.make_pytest_opts(args, unparsed_args))


def test_all(args, unparsed_args=None):
    return test_sandbox(args, unparsed_args) or test_tasks(args)


def _get_pytest_args(args, unparsed_args, sandbox_dir, projects_dir):
    pytest_args = ["--cache-clear"] if args.clean else []
    pytest_args.extend([
        "-s{}".format("v" if args.verbose else ""),
        "--sandbox-dir={}".format(sandbox_dir),
        "--confcutdir={}".format(projects_dir),
        "--no-print-logs",
    ])
    if args.task_type:
        pytest_args.extend([
            "--tests={}".format(args.task_type),
            "-n0",
            "projects/tests/tasks.py",
        ])
    else:
        pytest_args.extend(["-n", str(args.numprocesses)])
        if args.tests_path:
            pytest_args.append(args.tests_path)
        elif not any(map(os.path.exists, unparsed_args or [])):  # there is no paths among arguments for pytest
            pytest_args.append("projects")
    if unparsed_args:
        pytest_args.extend(unparsed_args)
    logging.info("Tests command: %s", " ".join(pytest_args))
    return pytest_args


def test_tasks(args, unparsed_args=None):
    import pytest

    sandbox_dir = py.path.local(sys.argv[0]).dirpath().dirname
    projects_dir = os.path.join(sandbox_dir, "projects")
    if args.clean:
        remove_precompiled_files(sandbox_dir)
    sandbox_revision, tasks_revision = map(dev_utils.get_path_revision, (sandbox_dir, projects_dir))

    if not args.runtime:
        arc_repo = dev_config.get_arc_repo(sandbox_dir)
        runtime_dir = dev_config.make_runtime_dir(sandbox_dir, arc_repo)
    else:
        runtime_dir = os.path.abspath(args.runtime)

    try:
        dev_config.get_config(args)
    except SystemExit:
        # Sandbox is not configured. Create a temporary config file for task tests
        with tempfile.NamedTemporaryFile(delete=False) as conf:
            settings = {"client": {"dirs": {"data": runtime_dir}}}
            yaml.dump(settings, conf, default_flow_style=False)
        logging.info("Dumped temporary config file: %s", conf.name)
        os.environ[common.config.Registry.CONFIG_ENV_VAR] = conf.name
        common.config.Registry().reload("client")
        assert common.config.Registry().client.tasks.build_cache_dir.startswith(runtime_dir)

    if "" in (sandbox_revision, tasks_revision):
        logging.info("Failed to compare revisions of core sandbox and sandbox projects")
    else:
        sandbox_revision, tasks_revision = map(int, (sandbox_revision, tasks_revision))
        if sandbox_revision < tasks_revision:
            cz = common.console.AnsiColorizer()
            print(cz.yellow(
                "Sandbox core's revision ({}) is older than that of Sandbox tasks' ({}), consider an update.".format(
                    sandbox_revision, tasks_revision
                )
            ))
            logging.info(
                "Testing sandbox projects against older sandbox core\n"
                "\tsandbox revision: %d\n"
                "\tsandbox projects revision: %d",
                sandbox_revision, tasks_revision
            )

    with common.fs.WorkDir(sandbox_dir):
        return pytest.main(_get_pytest_args(args, unparsed_args, sandbox_dir, projects_dir))


def run_shell(args):
    config = dev_config.get_config(args)
    run_tests_script = os.path.join(config.sandbox_dir, 'bin', 'shell.py')
    if sys.dont_write_bytecode:
        args = [sys.executable, '-B']
    else:
        args = [sys.executable]
    p = subprocess.Popen(args + [run_tests_script])
    p.wait()


def run_s_top(args):
    config = dev_config.get_config(args)
    cmd = [sys.executable, os.path.join(config.sandbox_dir, "scripts", "stop", "main.py")]
    env = os.environ.copy()
    env['PYTHONPATH'] = ":".join([env.get("PYTHONPATH", ""), config.sandbox_dir])
    os.execve(sys.executable, cmd, env)


def create_docs(args):
    from projects.sandbox.build_sandbox_docs import builder

    working_dir = os.path.join(os.path.dirname(SANDBOX_DIR), "runtime_data", "sphinx")
    tasks_dir = SANDBOX_DIR if args.with_tasks else None
    media_path = os.path.join(SANDBOX_DIR, "web", "media")
    builder.DocsBuilder(
        project_name="Sandbox",
        copyright="{} LLC Yandex".format(dt.date.today().year),
        author="sandbox-dev@",
        version="local",
        release="local",
        working_dir=working_dir,
        output_dir=media_path,
        sandbox_dir=SANDBOX_DIR,
        tasks_dir=tasks_dir
    ).build()


def _get_task_type_hosts(task_type):
    common.projects_handler.load_project_types(reuse=True)
    from sandbox.yasandbox.proxy import task as proxy_task
    cls = proxy_task.getTaskClass(task_type)
    sandbox = SandboxAPI.rest_client()
    clients_list = sandbox.client.read({"limit": MAX_CLIENTS_NUMBER})["items"]
    return filter(lambda _: ctc.Tag.LXC in _["tags"], clients_list) if cls.privileged else clients_list


def test_task_type_hosts(options):
    """
    Check specified task type hosts.

    :param options: arguments provided by user via command line
    :param options.task_type: the type of tasks
    """

    return_code = 0
    task_type = getattr(options, 'task_type', None)
    if task_type:
        dev_utils.print_task_type_hosts(task_type, _get_task_type_hosts(task_type))
    else:
        logger.info('Set --task_type param')
        return_code = 1
    return return_code


def upload_binary(args, *cmd_args, **cmd_kwargs):

    from sandbox.yasandbox import manager
    from sandbox.yasandbox.proxy import resource as resource_proxy

    logging.Logger.manager.emittedNoHandlerWarning = True
    manager.use_locally()
    config = dev_config.get_config(args)

    cz = common.console.AnsiColorizer()

    base_url = "http://{}:{}".format(config.hostname, config.serviceapi_port)
    rest = SandboxAPI.rest_client(base_url, total_wait=5)

    with common.console.LongOperation("Check input file") as op:

        if not os.path.isfile(args.binary):
            op.intermediate(cz.red("No such file: {}".format(args.binary)))
            raise op.Fail

        if os.path.islink(args.binary):
            args.binary = os.readlink(args.binary)

        file_name = os.path.basename(args.binary)

    with common.console.LongOperation("Prepare resource attributes") as op:

        try:
            exec_args = [args.binary, "content", "--self-info"]
            logger.debug("Running: `%s`", " ".join(exec_args))
            out = subprocess.check_output(exec_args, stderr=subprocess.STDOUT)

        except subprocess.CalledProcessError as exc:
            op.intermediate(cz.red("Got an error (return code: {})".format(exc.returncode)))
            op.intermediate(cz.red(exc.output))
            raise op.Fail

        except OSError as exc:
            op.intermediate(cz.red("Error: {}".format(exc)))
            raise op.Fail

        try:
            binary_info = json.loads(out)
        except Exception as exc:
            op.intermediate(cz.red("Could not read JSON output: {}".format(exc)))
            raise op.Fail

        attrs = {"ttl": 3}
        attrs.update(binary_info["resource_attributes"])
        if args.enable_taskbox:
            attrs[ctr.BinaryAttributes.TASKBOX_ENABLED] = True

        extra_attrs = dict(attr.split("=", 1) for attr in args.attr) if args.attr else {}
        attrs.update(extra_attrs)
        logger.debug("Resource attributes: %s", json.dumps(attrs))

    with common.console.LongOperation("Create parent task") as op:
        try:
            resp = rest.task(
                type="HTTP_UPLOAD_2",
                owner=controller.Group.anonymous.name,
                author=controller.User.anonymous.login,
                hidden=True,
            )
        except common.rest.Client.TimeoutExceeded:
            op.intermediate(cz.red("Could not reach local Sandbox API: {}.".format(base_url)))
            op.intermediate(cz.red("Active local Sandbox is required."))
            raise op.Fail
        except common.rest.Client.HTTPError as exc:
            op.intermediate(cz.red("Sandbox API Error: HTTP {}, {}.".format(exc.status, exc.response.text)))
            raise op.Fail

        task = controller.TaskWrapper(controller.Task.get(resp["id"]))
        task.set_status(ctt.Status.SUCCESS, force=True)
        logger.debug("Parent task ID: %d", task.id)

    with common.console.LongOperation("Create resource"):
        common.fs.create_task_dir(task.id)
        file_size = 0
        hasher = hashlib.md5()
        with open(args.binary, "rb") as f:
            for chunk in iter(lambda: f.read(4096), b""):
                file_size += len(chunk)
                hasher.update(chunk)
        file_md5 = hasher.hexdigest()

        resource = resource_proxy.Resource(
            0,
            name="Tasks binary: {}".format(file_name),
            file_name=file_name,
            file_md5=file_md5,
            resource_type=sdk2.service_resources.SandboxTasksBinary.name,
            task_id=task.id,
            owner=controller.Group.anonymous.name,
            arch=ctm.OSFamily.ANY,
            attrs=attrs,
            state=ctr.State.READY,
            timestamp=time.time(),
            size=file_size >> 10,
        )
        manager.resource_manager.create(resource)
        manager.resource_manager.add_host(resource.id, config.node_id)
        logger.debug("Resource ID: %d", resource.id)

    with common.console.LongOperation("Copy file content"):
        resource_path = pathlib2.Path(resource.abs_path())
        resource_path.parent.mkdir(mode=0o755, exist_ok=True)
        shutil.copy2(args.binary, str(resource_path))

    print("Binary tasks resource:", cz.green("{}/resource/{}".format(base_url, resource.id)))


def add_task(options):
    """
    Adds production task
    :param options.id: an iterable with task identifiers
    :return: `-1` if options.id not specified, else None
    """

    if not options.id:
        logger.info("Use --id option to specify identifier(s) of task(s) to import")
        return -1

    sandbox_config = dev_config.get_config(options)
    dev_config.SandboxConfig.add_config_file_path_to_environment(sandbox_config.local_settings_file)
    sandbox = SandboxAPI.rest_client(options.source)

    success = True
    for tid in options.id:
        success &= bool(_add_task(options, tid, sandbox, sandbox_config.node_id, force=options.force))
    if not success:
        return -1


def _register_group(group_name):
    if not group_name:
        logger.warning("Group name has not been setup '%s', use '%s' instead", group_name, ctu.OTHERS_GROUP.name)
        return ctu.OTHERS_GROUP.name

    local_name = mapping.Group.objects(name=group_name).scalar("name").first()
    if local_name:
        logger.debug("Group '%s' is already registered", local_name)
    else:
        new_group = mapping.Group(name=group_name, users=[ctu.ANONYMOUS_LOGIN]).save()
        logger.info("New group created with name '%s'", new_group.name)
    return group_name


def _import_task_resources(options, src_task_id, dst_task_id, sandbox_client, node_id, sync, force):
    import sandbox.yasandbox.manager as manager
    import sandbox.yasandbox.proxy.resource as proxy_resource

    # Pre-create the destination task's working directory while handling restrictive permissions on any of subpaths
    # (for ex., when {runtime_data}/tasks/b/a exists with 0555 access rights, so that it's impossible to create
    # intermediate directories)
    common.fs.create_task_dir(dst_task_id)

    task_resources = sandbox_client.resource.read({"task_id": src_task_id, "limit": 1000})["items"]
    for resource in task_resources:
        group_name = _register_group(resource.get("owner"))
        if (
            not getattr(options, "with_service", False) and
            issubclass(sdk2.Resource[resource["type"]], sdk2.ServiceResource)
        ):
            logger.debug("Service resource #%s skipped", resource["id"])
            continue

        mapping.ensure_connection()
        local_ids = list(dev_mapping.ImportedResource.objects(original_id=resource["id"]).scalar("local_id"))
        res = mapping.Resource.objects(
            id__in=local_ids, state__ne=ctr.State.DELETED
        ).order_by("-id").first()
        if res:
            logger.debug("Updating local copy of %s.", resource)
            new_resource = existing_resource = manager.resource_manager.load(res.id)
            existing_resource.attributes = resource["attributes"]
            existing_resource.owner = group_name
            manager.resource_manager.update(existing_resource)
        else:
            logger.debug("Copy %s", resource)
            new_resource = proxy_resource.Resource(
                0,
                name=resource["description"],
                file_name=resource["file_name"],
                file_md5=resource["md5"],
                resource_type=resource["type"],
                task_id=dst_task_id,
                owner=group_name,
                arch=resource["arch"],
                attrs=resource["attributes"],
                state=resource["state"],
                timestamp=time.mktime(dateutil.parser.parse(resource["time"]["created"]).timetuple()),
                skynet_id=resource["skynet_id"],
                size=resource["size"] >> 10,
            )
            if getattr(options, "preserve_id", False):
                new_resource._update(mapping.Resource(id=resource["id"], time=mapping.Resource.Time()))

            manager.resource_manager.create(new_resource)
            dev_mapping.ImportedResource.objects(original_id=resource["id"]).update(
                set__local_id=new_resource.id, upsert=True
            )

        if resource["state"] == ctr.State.READY and (not resource["skynet_id"] or force) and sync:
            resource_id = resource["id"]
            rsync = sandbox_client.resource[resource_id].data.rsync[:]
            rsync_src = next(iter(rsync), {}).get("url")
            if not rsync_src:
                logger.warning("Cannot sync resource #%s, it has no rsync links", resource_id)
                continue
            rsync_dst = "{}/".format(os.path.dirname(new_resource.abs_path()))
            logs_dir = dev_config.get_config(options).logs_dir
            logger.info("Begin rsync resource #%s from '%s' to '%s'.", resource["id"], rsync_src, rsync_dst)
            if not os.path.exists(rsync_dst):
                os.makedirs(rsync_dst, 0755)
            try:
                sdk1.copy.RemoteCopy(rsync_src, rsync_dst, log_dir=logs_dir)()
                logger.info("Resource #%s synced via rsync.", resource["id"])
                manager.resource_manager.add_host(new_resource.id, node_id)
            except Exception as ex:
                logger.exception(ex)


def _add_task(options, task_id, sandbox, node_id, sync=True, force=False, draft=False):
    from sandbox.yasandbox import manager

    manager.use_locally()
    common.projects_handler.load_project_types(reuse=True)

    rest = SandboxAPI.rest_client(options.source)
    task = rest.task[task_id].read()

    return _add_task_rec(options, task, sandbox, node_id, rest, sync=sync, force=force, draft=draft)


def _add_task_children(options, task, local_task, sandbox, node_id, rest, **kwargs):
    child_tasks = rest.task[task["id"]].children.read()["items"]
    for child_task in reversed(child_tasks):
        child_task = _add_task_rec(options, child_task, sandbox, node_id, rest, **kwargs)
        controller.Task.Model(id=child_task.id).update(parent_id=local_task.id)


def _add_task_rec(options, task, sandbox, node_id, rest, sync=True, force=False, draft=False):
    from sandbox.yasandbox import manager
    import sandbox.yasandbox.api.json.task as api_task

    local_id = dev_mapping.ImportedTask.objects(original_id=task["id"]).scalar("local_id").first()
    if local_id:
        local_task = controller.TaskWrapper(controller.Task.get(local_id))
        if local_task.status != ctt.Status.DELETED:
            logger.info(
                "Skip importing of already imported task #%d (local copy: #%d), but check its resources anyway",
                task["id"], local_id
            )
            _import_task_resources(options, task["id"], local_id, sandbox, node_id, sync, force)
            if getattr(options, "recursive", None):
                _add_task_children(
                    options, task, local_task, sandbox, node_id, rest, sync=sync, force=force, draft=draft
                )
            return local_task

    try:
        custom_fields = rest.task[task["id"]].custom.fields.read()
        ctx = rest.task[task["id"]].context.read()
    except common.rest.Client.HTTPError:
        custom_fields, ctx = {}, {}

    group_name = _register_group(task.get("owner"))
    logger.info("Download Sandbox task #%(id)s, type %(type)s, created by %(author)s.", task)
    model = controller.Task.Model(
        type=task["type"],
        execution=controller.Task.Model.Execution(status=task["status"] if not draft else None),
        description=task["description"],
        owner=group_name,
        author=controller.User.anonymous.login,
        priority=ctt.Priority.make(task["priority"]),
        kill_timeout=task.get("kill_timeout", 0),
        tags=task["tags"],
    )

    if task.get("release", {}).get("type"):
        release = task["release"]
        model.release = mapping.Task.Release(
            creation_time=dt.datetime.utcnow(),
            author=controller.User.anonymous.login,
            status=release["type"],
            message=mapping.Task.Release.Message(
                subject=release.get("template", {}).get("subject") or "",
                body=release.get("template", {}).get("message", "")
            )
        )
    if getattr(options, "preserve_id", False):
        model.id = task["id"]

    new_task = controller.TaskWrapper(model).create()

    dev_mapping.ImportedTask.objects(original_id=task["id"]).update(
        set__local_id=new_task.id, upsert=True
    )
    input_fields, output_fields = [], []
    for field in custom_fields:
        (output_fields if field.get('output') else input_fields).append(field)

    api_task.Task.update_and_validate_custom_fields(None, new_task, input_data=input_fields, output_data=output_fields)

    if new_task.model.type in sdk2.Task and new_task.status != ctt.Status.DRAFT:
        # normally this is done in on_save but if new_task is not in DRAFT state,
        # it could end up with unencoded parameters
        new_task._task(reset=True)
        new_task.update_model()

    if draft:
        new_task.set_info(
            'Cloned from production Sandbox. Original task is <a href={}>#{}</a>.'.format(task['url'], task['id']),
            do_escape=False
        )
    else:
        new_task.update_context(ctx)
        new_task.info = task.get('results', {}).get('info', {})

        release = sandbox.release.list.read({"task_id": task["id"], "limit": 10})
        if release:
            release = release[0]
            manager.release_manager.add_release_to_db(
                task_id=new_task.id, author=release["author"], status=release["status"],
                message_subject=release["subject"], creation_time=dt.datetime.fromtimestamp(release["timestamp"])
            )
        logger.info("Copy task #%s resources.", task["id"])
        _import_task_resources(options, task["id"], new_task.id, sandbox, node_id, sync, force)

    new_task.save()
    logger.info('Task #%s was successfully copied to local Sandbox task #%s', task['id'], new_task.id)
    if getattr(options, "recursive", None):
        _add_task_children(options, task, new_task, sandbox, node_id, rest, sync=sync, force=force, draft=draft)
    return new_task


def clone_task(options):
    """
    Clone task(s) with their parameters and dependencies and prepare them for local execution (change status to DRAFT).
    Use productional or any other installation, if given its REST API URL

    :param options.id: an iterable with task identifiers
    :return: None if options.id is specified, -1 otherwise
    """

    import sandbox.yasandbox.api.json.task as api_task

    if not options.id:
        logger.info("Use --id option to specify identifier(s) of task(s) to clone")
        return -1

    sandbox_config = dev_config.get_config(options)
    dev_config.SandboxConfig.add_config_file_path_to_environment(sandbox_config.local_settings_file)
    sandbox = SandboxAPI.rest_client(options.source)

    success = True
    for tid in options.id:
        new_task = _add_task(options, tid, sandbox, sandbox_config.node_id, draft=True)
        if not new_task:
            success = False
            continue

        if new_task.status != ctt.Status.DRAFT:
            logger.error(
                "Task #%d is in status %s different to DRAFT, please remove it before cloning",
                new_task.id, new_task.status
            )
            success = False
            continue

        rest = SandboxAPI.rest_client(options.source)
        input_parameters = filter(
            lambda input_parameter: issubclass(input_parameter, sdk1.parameters.ResourceSelector),
            new_task.input_parameters
        )

        field = {_["name"]: _ for _ in rest.task[tid].custom.fields[:]}
        res_count = 0
        for parameter in input_parameters:
            if parameter.name not in field:
                continue
            value = field[parameter.name]["value"]
            logger.info("Processing resource parameter %r with value %r", parameter.name, value)
            local_ids = []
            for identifier in filter(None, common.utils.chain(value)):
                local_id = _add_resource(identifier, options)
                if local_id is None:
                    logger.error(
                        "Failed to add resource #%d, which is a dependency of task #%d; stopping cloning",
                        identifier, tid
                    )
                    success = False
                    continue

                local_ids.append(local_id)
            res_count += len(local_ids)

            if isinstance(value, (int, basestring)):
                local_ids = local_ids[0]
            elif value is None:
                local_ids = None

            api_task.Task.update_and_validate_custom_fields(
                None, new_task, input_data=[{"name": parameter.name, "value": local_ids}]
            )

        input_parameters = filter(
            lambda input_parameter: issubclass(input_parameter, sdk1.parameters.TaskSelector),
            new_task.input_parameters
        )
        tasks_count = 0
        for parameter in input_parameters:
            if parameter.name not in field:
                continue
            value = field[parameter.name]["value"]
            logger.info("Processing task parameter %r with value %r", parameter.name, value)
            local_tasks = []
            for identifier in filter(None, common.utils.chain(value)):
                dependant_task = _add_task(options, identifier, sandbox, sandbox_config.node_id)
                if dependant_task is None:
                    logger.error(
                        "Failed to add task #%d, which is a dependency of task #%d; stopping cloning",
                        identifier, tid
                    )
                    success = False
                    continue

                local_tasks.append(dependant_task.id)
            tasks_count += len(local_tasks)

            if isinstance(value, (int, basestring)):
                local_tasks = local_tasks[0]
            elif value is None:
                local_tasks = None

            api_task.Task.update_and_validate_custom_fields(
                None, new_task, input_data=[{"name": parameter.name, "value": local_tasks}]
            )

        new_task.save()
        logger.info(
            "Task #%s with %s resource and %s tasks dependencies was successfully "
            "cloned from Sandbox and saved as DRAFT with id #%s",
            tid, res_count, tasks_count, new_task.id
        )

    if not success:
        return -1


def _add_resource(resource_id, options):
    """
    Adds production resource

    :return: local resource identifier
    """

    sandbox_config = dev_config.get_config(options)
    dev_config.SandboxConfig.add_config_file_path_to_environment(sandbox_config.local_settings_file)
    sandbox = SandboxAPI.rest_client(options.source)

    resource = sandbox.resource[resource_id].read()
    if resource is None:
        print_and_exit("Resource #{} not found.".format(resource_id))

    mapping.ensure_connection()
    local_id = dev_mapping.ImportedResource.objects(original_id=resource_id).scalar("local_id").first()
    if local_id:
        if mapping.Resource.objects(id=local_id).scalar("state").first() != ctr.State.DELETED:
            logger.info("Resource #%d already exists in local Sandbox with id #%d", resource_id, local_id)
            return local_id

    logger.info("Download %s.", resource)
    _add_task(options, resource["task"]["id"], sandbox, sandbox_config.node_id, force=options.force)
    new_id = dev_mapping.ImportedResource.objects(original_id=resource_id).scalar("local_id").first()
    if new_id:
        logger.info("Resource #%s was successfully copied to local Sandbox resource #%s", resource["id"], new_id)
    else:
        logger.info("Resource #%s was not copied to local Sandbox.", resource["id"])
    return new_id


def add_resource(options):
    """
    Add resource(s) from productional or any other installation, if given its REST API URL

    :param options.id: an iterable with resource identifiers
    :return: None if options.id is specified, -1 otherwise
    """

    if not options.id:
        logger.info("Use --id option to specify identifier(s) of resource(s) to import")
        return -1

    success = True
    for rid in options.id:
        success &= bool(_add_resource(rid, options))
    if not success:
        return -1


def destroy_privileged(options):
    sandbox_config = dev_config.get_config(options)
    dev_config.SandboxConfig.add_config_file_path_to_environment(sandbox_config.local_settings_file)
    import client.system
    import client.platforms.lxc

    if common.os.User.has_root:
        subprocess.call([
            "/usr/bin/sudo", "-En", "PYTHONPATH=/skynet", "USER=" + getpass.getuser(), sys.executable
        ] + sys.argv)
    else:
        stop_client(options)
        client.system.UNPRIVILEGED_USER = common.os.User(os.environ["USER"])
        client.platforms.lxc.logger = dev_config.logger
        client.platforms.lxc.PrivilegedLXCPlatform.destroy_container()

    if not common.os.User.has_root or os.environ["USER"] == "root":
        start_client(options)


def encrypt_secrets(args, *cmd_args, **cmd_kwargs):
    cz = common.console.AnsiColorizer()
    keyfile_path = common.fs.to_path(common.config.Registry().server.encryption_key)

    if os.path.exists(keyfile_path):
        print("Your secrets are already encrypted!")
        print("Encryption key: " + cz.red(keyfile_path))
        return

    mapping.ensure_connection()

    print("We are going to create a new encryption key at {}".format(cz.red(keyfile_path)))
    print("All your local vaults will be automatically encrypted with it.")
    print("The key is also required to use Yandex Vault (yav.yandex-team.ru) in local Sandbox.\n")

    yav_secrets = list(mapping.YavToken.objects.fast_scalar("secret_uuid").all())

    if yav_secrets:
        print(cz.red("Warning: ") + "Local delegations will be lost for the following Yav secrets")
        for secret in yav_secrets:
            print("* {}".format(secret))

        answer = raw_input("\n" "Continue? (y/N) ").strip() or "no"
        if not distutils.util.strtobool(answer):
            print(cz.red("Cancelled"))
            return

    with common.console.LongOperation("Generate encryption key") as op:
        key = common.crypto.AES.generate_key(keyfile_path)
        op.intermediate(cz.red(keyfile_path))

    old_values = {}
    ok = False

    try:
        if not mapping.Vault.objects.count():
            ok = True
            return

        vaults = list(mapping.Vault.objects.all())
        old_values = {obj.id: str(obj.data) for obj in vaults}

        print("\n" "You have got the following vaults (values are censored):")
        for vault in vaults:
            data = common.format.obfuscate_token(str(vault.data))
            print("* {}:{}".format(vault.owner, vault.name))
            print("  {}".format(data))

        answer = raw_input("\n" "Do they look like valid non-encrypted plain-text? (y/N) ").strip() or "no"
        if not distutils.util.strtobool(answer):
            print(cz.red("Cancelled"))
            return

        with common.console.LongOperation("Encrypt existing vaults") as op:
            for vault in vaults:
                vault.data = common.crypto.AES(key).encrypt(vault.data, use_base64=False, use_salt=True)
                vault.encrypted = True
                vault.save()

                new_value = controller.Vault.get_by_id(None, vault.id).data
                if new_value != old_values[vault.id]:
                    print(cz.red("Failure: decrypted value for `{}` does not match the original".format(vault.name)))
                    return

            mapping.YavToken.objects.all().delete()

        ok = True
        print(cz.green("Your secrets are encrypted now!"))
        print("Please, restart Sandbox with `./sandbox kill && ./sandbox restart`")

    finally:
        if ok:
            return

        with common.console.LongOperation("Revert all changes") as op:
            op.intermediate("Remove encryption key...")
            os.unlink(keyfile_path)
            op.intermediate("Restore vault values...")
            for id_, value in old_values.items():
                mapping.Vault.objects.filter(id=id_).update_one(data=value)


class Option(object):
    def __init__(self, *args, **kwargs):
        self.args = args
        self.kwargs = kwargs


_OPTIONS = {
    'port': Option(
        '-p',
        '--port',
        type=int,
        action='store',
        default=None,
        help='Sandbox server port.'
    ),
    'mongo-uri': Option(
        '--mongo-uri',
        type=str,
        action='store',
        default=None,
        help='Mongo DB connection string.'
    ),
    'config': Option(
        '-c',
        '--config',
        type=str,
        action='store',
        default=None,
        help='Sandbox config path.'
    ),
    'task_type': Option(
        '-t', '-s',
        '--task_type',
        type=str,
        action='store',
        default=None,
        help='Specific task type.'
    ),
    'tests_path': Option(
        '-p',
        '--tests_path',
        type=str,
        action='store',
        default=None,
        help='A directory relative to Sandbox path in which pytest will look for tests.'
    ),
    'verbose': Option(
        '-v',
        '--verbose',
        action='store_true',
        help="Show verbose output."
    ),
    "tid": Option(
        "-i",
        "--id",
        type=int,
        nargs="+",
        default=None,
        help="Task identifier(s) to import"
    ),
    "rid": Option(
        "-i",
        "--id",
        type=int,
        nargs="+",
        default=None,
        help="Resource identifier(s) to import"
    ),
    'source': Option(
        '-S',
        '--source',
        type=str,
        action='store',
        default=None,
        help="URL to Sandbox API (defaults to sandbox.yandex-team.ru)."
    ),
    'force': Option(
        '-f',
        '--force',
        action='store_true',
        help="Force syncing resources via rsync if there is attribute 'skynet_id'."
    ),
    "porto": Option(
        "--porto",
        action="store_true",
        help="Try use porto as runtime for containers"
    ),
    'clean': Option(
        '--clean',
        action='store_true',
        help="Remove precompiled source code files (*.pyc) and previous tests data before start of testing"
    ),
    'with_tasks': Option(
        '--with-tasks',
        action='store_true',
        help='Build documentation for tasks as well. Warning: this takes a few hours to complete'
    ),
    "numprocesses": Option(
        "-n",
        "--numprocesses",
        type=str,
        default="3",
        help="Number of test processes. Parameter is transferred to pytest. 'auto' is available.",
    ),
    "runtime": Option(
        "-r",
        "--runtime",
        type=str,
        action="store",
        default=None,
        help="Path to Sandbox `runtime-data` folder.",
    ),
    "auth": Option(
        "--auth",
        action="store_true",
        help="Enable authentification while requesting remote Sandbox API.",
    ),
    "recursive": Option(
        "--recursive",
        action="store_true",
        help="Specify adding task with children.",
    ),
    "with_service": Option(
        "--with_service",
        action="store_true",
        help="Do not skip service resources (for example, task logs)",
    ),
    "preserve_id": Option(
        "--preserve-id",
        "--preserve-ids",
        action="store_true",
        help="Keep original task/resource identifier(s) without overwriting"
    ),
    "key": Option(
        "--key",
        action="store",
        help="Config option, period-separated"
    ),
    "value": Option(
        "--value",
        action="store",
        help="Desired value"
    ),
    "enable_taskbox": Option(
        "--enable-taskbox",
        action="store_true",
        help="Enable server-side hooks executing for tasks created using this binary."
    ),
    "attr": Option(
        "--attr",
        type=str,
        action="append",
        help="Resource attribute in <name>=<value> form. Multiple argument."
    ),
    "binary": Option(
        "binary",
        type=str,
        help="Binary tasks executable to upload",
    )
}

_SUBPARSERS = [
    (
        add_task, "add_task", "add task from production (supports multiple values)",
        ["tid", "config", "force", "source", "recursive", "with_service", "preserve_id"]
    ),
    (
        add_resource, "add_resource", "add resource from production (supports multiple values)",
        ["rid", "config", "force", "source", "with_service", "preserve_id"]
    ),
    (
        clone_task, "clone_task",
        (
            "clone task from production with all its dependencies (input resources) and "
            "prepare for local execution by changing status to DRAFT (supports multiple values)"
        ),
        ["tid", "config", "force", "source", "with_service", "preserve_id"]
    ),
    (
        upload_binary, "upload_binary", "upload binary task to local Sandbox",
        ["config", "enable_taskbox", "attr", "binary"]
    ),
    (
        encrypt_secrets, "encrypt_secrets", "enable encription for Vault and Yav",
        ["config"]
    ),
    (get_info, "info", "get info about local Sandbox", ["config"]),
    (setup_sandbox, "setup", "setup local sandbox", ["port", "mongo-uri", "force", "source", "porto"]),
    (setup_client, "setup_client", "setup client", ["port", "mongo-uri", "source"]),
    (check_resource_updates, "check_updates", "update environment resources", ["force", "source"]),
    (clean_all, "clean_all", "clean all: `runtime-data` folder and database'", ["config"]),
    (kill, "kill", "kill all processes related to Sandbox", ["config"]),
    (clean_db, "clean_db", "clean database", ["config"]),
    (clean_data, "clean_data", "clean `runtime-data` folder", ["config"]),
    (clean_logs, "clean_logs", "clean logs", ["config"]),
    (clean_tests, "clean_tests", "clean tests", ["config"]),
    (create_docs, "create_docs", "create docs", ["with_tasks"]),
    (start_sandbox, "start", "start sandbox", ["config"]),
    (start_server, "start_server", "start server", ["config"]),
    (start_client, "start_client", "start client", ["config"]),
    (stop_sandbox, "stop", "stop sandbox", ["config"]),
    (stop_server, "stop_server", "stop server", ["config"]),
    (stop_client, "stop_client", "stop client", ["config"]),
    (restart_sandbox, "restart", "restart sandbox", ["config"]),
    (restart_server, "restart_server", "restart server", ["config"]),
    (restart_client, "restart_client", "restart client", ["config"]),
    (
        test_all, "test", "run all tests",
        ["mongo-uri", "task_type", "tests_path", "clean", "numprocesses"]
    ),
    (test_sandbox, "test_sandbox", "run sandbox tests", ["mongo-uri", "clean"]),
    (
        test_tasks, "test_tasks", "run tasks tests",
        ["mongo-uri", "task_type", "tests_path", "clean", "numprocesses"]
    ),
    (test_task_type_hosts, "test_task_hosts", "run task hosts tests", ["task_type"]),
    (run_shell, "shell", "run sandbox shell", ["config"]),
    (dev_config.print_config, "print_config", "print sandbox config", ["config"]),
    (dev_config.set_config_option, "set", "set config option", ["config", "key", "value"]),
    (run_s_top, "top", "run sandbox-top", ["config"]),
    (destroy_privileged, "destroy_container", "destroy running privileged container", ["config"]),
]


def _get_arguments_parser():
    parser = argparse.ArgumentParser(
        formatter_class=argparse.RawTextHelpFormatter,
        description="Sandbox maintenance tool",
        epilog=textwrap.dedent("""
            Wiki:                 https://wiki.yandex-team.ru/Sandbox
            RTFM:                 https://sandbox.yandex-team.ru/docs/html/index.html
            Bugs and questions:   sandbox@yandex-team.ru
        """)
    )
    verbose_arg = _OPTIONS["verbose"]
    parser.add_argument(*verbose_arg.args, **verbose_arg.kwargs)
    subparsers = parser.add_subparsers(
        help="Command description",
        description="Available commands are:",
        metavar="<command>",
    )
    common_options = ["runtime", "auth", "verbose"]
    for cmd, cmd_name, description, options in _SUBPARSERS:
        subparser = subparsers.add_parser(cmd_name, help=description)
        subparser.set_defaults(func=cmd)
        for opt_name in common.utils.chain(options, common_options):
            option = _OPTIONS[opt_name]
            subparser.add_argument(*option.args, **option.kwargs)
    return parser


def main():
    parser = _get_arguments_parser()
    if len(sys.argv) == 1:
        parser.print_help()
        sys.exit(1)

    args, unknown_args = parser.parse_known_args()
    logger.setLevel(logging.DEBUG if args.verbose else logging.INFO)
    if args.verbose:
        logging.getLogger().addHandler(logging.StreamHandler())
        logging.getLogger().setLevel(logging.DEBUG)
    SandboxAPI.enable_auth = args.auth

    ret = args.func(args, unknown_args) if unknown_args else args.func(args)
    sys.exit(ret)


if __name__ == "__main__":
    main()
