# Бинарные задачи

Стандартный механизм написания задач, описанный в предыдущих разделах, имеет ряд недостатков:

* **Медленная выкатка изменений**. Для появления изменений в коде задачи в Sandbox необходимо добавить эти изменения в единый репозиторий. Добавление даже мелкого изменения может занимать 30-40 минут.
* **Ограниченное количество доступных библиотек**. В задачах можно использовать только [небольшой набор](environment.md#libraries) внешних библиотек, остальные приходится устанавливать через pip при каждом запуске задачи, в некоторых случаях это сделать нельзя из-за конфликтов с уже существующими в текущем venv пакетами. Нельзя использовать большинство библиотек из единого репозитория.

Эти недостатки решены в **бинарных задачах**. Основное отличие от стандартного механизма состоит в том, что код одной или нескольких задач собирается в **исполняемый файл** (binary executable file), который скачивается и исполняется на агенте. Исполняемый файл использует [статическую линковку](https://en.wikipedia.org/wiki/Static_library), поэтому все внешние библиотеки встроены в него. Это позволяет использовать в коде задач любые библиотеки, доступные в едином репозитории.

{% note warning %}

Код бинарной задачи может быть написан только с использованием [SDKv2](https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/sdk2).

{% endnote %}

## Как сделать задачу бинарной { #migration }

Для того чтобы задача стала бинарной, необходимо внести некоторые изменения.

1. **ya.make** файл должен выглядеть так:

    ```bash
    ~/arcadia/sandbox/projects/my_project/my_test_task$ cat ya.make
    SANDBOX_PY23_TASK(my-test-task) # В скобках указано имя исполняемого файла

    OWNER(
        your-login # Сюда ваш логин на Staff
    )

    PY_SRCS(__init__.py)

    PEERDIR( # Здесь указываются все зависимости задачи (можно добавит sandbox/common) относительно корня единого репозитория. Можно указать каталог другой задачи и тогда она попадёт в исполняемый файл.
        library/python/retry
        sandbox/common
        sandbox/projects/common/binary_task
    )

    RESOURCE( # Здесь указываются все произвольные файлы, доступ к которым необходим внутри задачи. Каждая запись состоит из пути до файла и ключа, по которому он будет доступен внутри программы.
        config.yml config.yml
        artifacts.txt artifacts.txt
    )

    END()
    ```
   Будьте внимательны, в бинарную сборку не попадают произвольные файлы, если их не указать явно.
   Для указания таких файлов используйте [команды ya.make](https://wiki.yandex-team.ru/yatool/howtowriteyamakefiles/#qkakmnedobavitvmojuprogrammudannyeresursyproizvolnyefajjly).

2. Класс задачи в файле `__init__.py` остается таким же. При этом зависимости, не входящие в [список доступных библиотек](environment.md#libraries), необходимо импортировать внутри методов.

    {% note warning %}

    Внешние библиотеки можно импортировать только в [обработчиках событий](task-life.md#client-side), исполняющихся на агентах (т.е. во всех, кроме `on_create`, `on_save` и `on_enqueue`).

    {% endnote %}


    ```python
    from sandbox import sdk2

    class MyTestTask(sdk2.Task):

        def on_execute(self):
            self._fetch_data()

        def _fetch_data(self):
            import library.python.retry  # External library import

            # Use import here
    ```

3. После внесения изменений можно собрать исполняемый файл задачи:

    ```bash
    ~/arcadia$ ya make sandbox/projects/my_project/my_test_task # Получили исполняемый файл my-test-task
    ```

### Задача для сборки бинарных задач { #deploy-binary-task }
Кроме ручной сборки исполняемого файла существует возможность собрать его автоматически.
Для этого нужно создать Sandbox задачу типа [DEPLOY_BINARY_TASK](https://sandbox.yandex-team.ru/tasks?children=false&type=DEPLOY_BINARY_TASK), указать путь до каталога с исходным кодом в едином репозитории и атрибуты ресурса:

![Параметры задачи DEPLOY_BINARY_TASK](img/binary-task-build-with-sandbox.png "Параметры задачи DEPLOY_BINARY_TASK")

Во время выполнения задачи будет создан ресурс, содержащий исполняемый файл с кодом задач.
Для того чтобы этот ресурс после релиза мог автоматически использоваться задачами наследованными от `sandbox.projects.common.binary_task.LastBinaryTaskRelease`, в параметре задачи `Binary task archive attributes (attrs)` нужно добавить `task_type: <ваш тип задачи>`

### CI/CD бинарных задач { #binary-task-ci }
Для упрощения настройки релизного процесса бинарных задач мы сделали [кубик в CI](https://a.yandex-team.ru/arc/trunk/arcadia/ci/registry/common/sandbox/deploy_binary_task.yaml), в котором уже задано большинство параметров. Также мы подготовили [шаблон](https://a.yandex-team.ru/arc/trunk/arcadia/ci/registry/common/sandbox/README.md) для создания релизного процесса, в котором представлен полный релизный цикл для сборки, тестирования и релиза ресурса с бинарной задачей/задачами покоммитно. Если вставить шаблон в a.yaml в собираемой директории и заменить в нем переменные в `<>` на ваши значения, то вы получите рабочий релизный процесс с покоммитной сборкой и ручным релизом в TESTING/PRESTABLE/STABLE.

Мы сделали шаблоны, в которых для разных сценариев нужно сделать минимальное количество простых изменений.
* Если какой-то из типов релиза вам не нужен, его достаточно удалить из `releases` и `jobs`
* Если нужен автоматический запуск релиза, то достаточно переключить `manual:false`
* Можно добавить свои атрибуты к собираемому ресурсу в секции `attrs`
* Если вам не нужен пробный запуск задачи, то можно удалить секции `integrational_check` и `integrational_check_payload`.
* Если нужно изменить зависимости релизов (например, вы хотите, чтобы между ними выполнялись ваши тестовые задачи), достаточно указать их в зависимостях needs

Для примера, мы сделали сборку и релиз бинарной задачи [BUILD_SANDBOX_TASKS](https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/sandbox/build_sandbox_tasks/a.yaml), используя новый CI/CD.

### Пробрасывание бинаря в дочерние задачи { #sticky-resource}

Дочерние задачи можно выполнять с тем же бинарем, на котором выполняется родительская задача.

Для этого нужно:
* Подключить дочернюю задачу к сборке и создавать её с помощью sdk2
* В параметрах задачи указать `push_tasks_resource = True`, либо при создании задачи через [API](https://sandbox.yandex-team.ru/media/swagger-ui/index.html#/task/task_list_post) указать `push_tasks_resource: true`

В результате бинарь будет рекурсивно передаваться во все создаваемые подзадачи. Чтобы остановить передачу бинаря в подзадачу, при ее создании нужно выставить `push_tasks_resource=False` или `__requirements__: {"tasks_resource": resource_id}`, где `resource_id` может быть `None`.

Другой способ запускать подзадачи на нужном бинаре – указать при создании задачи `id` бинаря в требованиях. Например, так:
```python
subtask = TEST_TASK_2(__requirements__={"tasks_resource": resource_id})
```

### Бинарник как бандл с задачами { #as-a-bundle }

Бинарник может содержать сразу нескольких разных [типов задач](../tasks.md#type) и исполнять любую из них на выбор.
Вывести список типов задач, доступных в исполняемом файле, можно командой:

```bash
$ sandbox/projects/devtools/my_test_task/my-test-task content --list-types
{"types": ["MY_TEST_TASK"]}
```

### Автоматический поиск ресурса с бинарем { #executor-discovery }

По умолчанию при создании бинарной задачи из UI или при помощи планировщиков код будет запущен не бинарно.
Это можно исправить, реализовав поиск нужного ресурса и указание его в качестве ресурса с кодом задачи.

Для того чтобы **закрепить версию кода бинарной задачи**, достаточно указать ID нужного ресурса  в поле `tasks_resource`.
Искать нужный ресурс можно при помощи любых [атрибутов](../resources.md#attributes), но наиболее распространённым является применение
атрибута `released` (`stable`, `testing`, `unstable`). Этот атрибут означает тип релиза ресурса, т.е. стабильность кода задачи.

Также рекомендуем всегда добавлять в фильтр группу (`owner`), так как именно через группы в Sandbox можно контролировать доступ:
создать ресурс определённого типа может кто угодно, но создать его от имени заданной группы сможет только член этой группы.

Для того чтобы различать ресурсы различных бинарных задач, также удобно использовать дополнительные атрибуты: например, если бинарник отвечает только за один тип задачи, то `task_type = MY_TEST_TASK`, а если за много, то `tasks_bundle = MY_BUNDLE>`.

После того как критерий поиска ресурса будет определен, останется реализовать автоматическое выставление нового ресурса при запуске задачи (метод `on_create`):

```python
from sandbox import common
from sandbox import sdk2
from sandbox.sdk2 import service_resources

class MyTestTask(sdk2.Task):

    def on_create(self):
        self.Requirements.tasks_resource = service_resources.SandboxTasksBinary.find(
            owner="MY_GROUP",
            attrs={"task_type": "MY_TEST_TASK"},
            # или attrs={"tasks_bundle": "<MY_BUNDLE_NAME>"},
         ).first()
```

#### LastBinaryTaskRelease { #last-binary-task-release }
Часто при запуске бинарной задачи хочется использовать последнюю стабильную её версию, загруженную в виде ресурса в Sandbox.
Для того чтобы не писать один и тот же код в каждой задаче, создан специальный mixin-класс `LastBinaryTaskRelease` ([код](https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/common/binary_task/__init__.py)).
Подмешивание этого класса в код задачи добавляет следующую логику: в серверном обработчике `on_save` задачи проставляется Requirement `tasks_resource`, который (пере-)определяет, какой бинарник будет использоваться в задаче. 

При использовании значений `stable`/`prestable`/`testing`/`unstable` будет использован самый верхний ресурс из [подобного фильтра](https://sandbox.yandex-team.ru/resources?any_attr=false&order=-created&state=READY&limit=20&attrs=%7B%22task_type%22%3A%22YABS_DUTY_SCORES%22%2C%22released%22%3A%22stable%22%7D).
О том, как устроены `last_successful_stable`/`testing_or_upper`, можно посмотреть в коде миксина.

##### { #last-binary-task-release-common-mistake }
Частой ошибкой при использовании этого миксина является непонимание того, что исполняемый бинарник может подмениться уже **после** того, как вы запустили задачу.
Например, вы собрали бинарный файл с задачей и запускаете его так
```bash
./hello_world run --type HELLO_WORLD --enable-taskbox
```
но в задаче в значении "Task resource" проставлен не ваш бинарник, загруженный минуту назад, а прочерк `—` либо стабильный ресурс, загруженный роботом месяц назад.
В таком случае нужно запускать задачу, переопределяя параметр значением `custom`:
```bash
./hello_world run --type HELLO_WORLD --enable-taskbox '{"custom_fields": [ {"name":"binary_executor_release_type", "value":"custom"}]}'
```

При использовании этого миксина в UI в параметрах задачи появится вот такая секция:
![Выбор версии исполнителя кода задач](img/binary-task-executor-type.png "Выбор версии исполнителя кода задач" =320x190) 

### Утилита командной строки { #cli-mode }

Второй возможный способ использования бинарных задач - это запуск исполняемого файла **из командной строки**.
Любая бинарная задача умеет выполнять несколько команд:
* `run` - загрузить исполняемый файл в Sandbox как ресурс и выполнить на его основе задачу
* `upload` - загрузить исполняемый файл в Sandbox
* `content` - набор опций для получения различных данных из собранной бинарной задачи (список задач, дамп ресурсов и т.п.)
* `ipython` - запустить интерактивный интерпретатор Python из кода задачи
* `interpret` - выполнить переданный скрипт используя окружение бинарной задачи

Так, например, имея собранный исполняемый файл, можно выполнить:
```bash
$ sandbox/projects/my_project/my_test_task/my-test-task run --type MY_TEST_TASK --enable-taskbox
```
В этом случае исполняемый файл автоматически загружает себя в Sandbox ресурс
и нужным образом выставляет требование `tasks_resource`.
А затем запускает задачу, используя самого себя как исполнителя кода задачи.

Эта возможность позволяет **быстро отлаживать код задачи**,
собирая новый исполняемый файл после каждого внесения изменений.

Для команды `run` есть дополнительные флаги:
* `--create-only` создаст задачу, но не будет её запускать
* `--type` позволяет выбрать какую из задач, упакованных в исполняемый файл, нужно запустить.
  Если упакована только одна задача, то указывать этот флаг необязательно
* `--enable-taskbox` позволяет выполнять обработчики событий на сервере прямо из исполняемого файла
* позиционный параметр `payload` - JSON с дополнительными параметрами задачи, например: `'{"custom_fields": [ {"name": "foo",  "value": "bar"}]}'`

Если требуется только загрузить исполняемый файл, можно использовать команду:
```bash
$ sandbox/projects/devtools/my-test-task/my-test-task upload --attr author=$(whoami) --attr released=stable
```

Описание команд и доступных опций можно получить через флаг `-h` или `--help`.

#### Аутентификация для командной строки { #cli-auth }

Для запуска (`run`) или загрузки (`upload`) бинарной задачи из командной строки необходим [OAuth-токен](../api.md#auth-oauth)
для доступа к Sandbox.

Передать токен можно двумя способами:
* Автоматически, предоставив приватный SSH ключ в обмен на токен.
  По умолчанию бинарная задача пытается сама получить токен через SSH ключ.
  Для этого:
  * публичный ключ должен быть добавлен на стафф (см. раскладывание ключа при [настройке SSH-клиентов](https://wiki.yandex-team.ru/security/ssh/#instrukciiponastrojjkessh-klientov))
  * приватный ключ должен передаваться через [agent forwarding](https://wiki.yandex-team.ru/security/ssh/#chtotakoessh-agentiagentforwarding)
    либо лежать в одном из дефолтных мест (проверить список ключей можно так `ssh-add -l`)
* Вручную.
  * Для этого его нужно получить через браузер на странице [https://sandbox.yandex-team.ru/oauth](https://sandbox.yandex-team.ru/oauth).
  * Передать токен явно, используя один из следующих вариантов:
    * записать токен в переменную окружения `SANDBOX_TOKEN`
    * сохранить токен в файл и передать путь до файла через опцию `--token`
    * передать токен строкой в опцию `--token` (используется только если не получилось получить токен по SSH ключу).

Если автоматически ключ не передаётся с виртуальной машины (ошибка `Failed to get valid token...`), проверьте настройку ssh-agent-forwarding. Для этого убедитесь, что:
* вы подключаетесь по ssh с ключом `-A` либо в `~/.ssh/config` для целевого хоста прописано `ForwardAgent yes`
* на исходной машине корректно настроен ssh-agent: значение в `env | grep SSH_AUTH` совпадает с тем, что задано в `ssh-agent -s`

## Использование внешних библиотек { #dependencies }

Хотя это уже было показано выше, еще раз повторим каким образом нужно подключать внешние библиотеки к бинарным задачам. Для подключения внешних библиотек:

1. Необходимо добавить все эти библиотеки в секцию **PEERDIR** в файле **ya.make**:

    ```bash
    ~/arcadia/sandbox/projects/my_project/my-test-task$ cat ya.make
    SANDBOX_PY23_TASK(my-test-task)

    OWNER(
        your-login
    )

    PY_SRCS(__init__.py)

    PEERDIR(
        sandbox/common # Sandbox common classes
        library/python/retry # <== Добавили внешнюю библиотеку
    )

    END()
    ```

2. Часть методов класса Sandbox задачи исполняется на сервере Sandbox, где отсутствуют внешние библиотеки. Поэтому **нельзя импортировать внешние библиотеки на уровне класса - только внутри методов**:

    ```python
    import library.python.retry  # Wrong!
    from sandbox import sdk2

    class MyTestTask(sdk2.Task):

        def on_execute(self):
            self._fetch_data()

        def _fetch_data(self):
            import library.python.retry  # Correct
    ```

## Исполнение server-side методов из бинаря { #server-side }

Чтобы создать задачу, серверам Sandbox надо узнать мета-информацию о ней (её параметры, требования по умолчанию), а также исполнить ряд методов задачи, доступных для перегрузки (`on_create`, `on_save`, `on_enqueue`).
Для всего этого серверу нужен python-класс задачи, который по умолчанию приезжает в [бандле задач](basics.md#tasks-releases) `sandbox/projects`.

Чтобы исполнять server-side методы не из бандла, а из бинарника задачи, ресурсу с бинарником нужно выставить атрибут `taskbox_enabled=True`.
При запуске бинарной задачи из консоли достаточно передать аргумент `--enable-taskbox`, это проставит нужный атрибут бинарнику после его загрузки.

{% note warning %}

В режиме с server-side исполнением работают только бинарники, собранные для linux.
Попытка запустить в этом режиме программы, собранные на, например, macOS, приведут к ошибкам создания задачи:
`It's forbidden to run binaries built with OS other than LINUX and flag '--enable-taskbox'.`

{% endnote %}

## Бинарные задачи в произвольном месте аркадии { #everywhere }

Хранить код `taskbox_enabled`  бинарных задач можно вне директории `sandbox/projects`.
Для этого нужно, чтобы в директории с задачей находилась сборочная цель одного из типов: `SANDBOX_PY3_TASK` или `SANDBOX_PY23_TASK`.
При использовании этих макросов в `ya.make` в специальный реестр бинаря добавляется путь до директории, в которой лежит этот `ya.make` . При запуске бинаря будут проимпортированы все python-модули из `sandbox/projects` (для обратной совместимости), а также `__init__.py`-модули из директорий реестра.

Таким образом, если класс задачи располагается в `__init__.py`-модуле директории с одним из макросов, описанных выше, то тип задачи будет находиться в бинаре автоматически.

Если же директория с задачей объявлена, например, как `PY23_LIBRARY`, то в `ya.make` нужно явно указать, что в модуле `__init__.py` в директории лежит класс Sandbox-задачи.
Для этого есть специальный макрос `REGISTER_SANDBOX_IMPORT` с перечнем директорий/файлов, в которых лежат Sandbox-задачи.
Макрос `REGISTER_SANDBOX_IMPORT` работает через макрос `RESOURCE`, поэтому если одна и та же директория/файл будет задекларирован дважды, то сборка упадет с ошибкой дублирования путей.

{% note warning %}

Бинарные задачи, расположенные вне директории `sandbox/projects`, неизвестны нашим API-серверам и потому не могут быть созданы через UI по названию типа задачи.
Чтобы создать такую задачу, в форме создания задачи, планировщика или шаблона надо сначала указать ресурс с бинарником, который содержит в себе класс задачи.

{% endnote %}

### Пример описания задачи { #everywhere-example }

Как и в `sandbox/projects` , создайте `__init__.py` , в котором будет располагаться код задач. В `ya.make` в той же директории объявите мультимодуль `SANDBOX_PY3_TASK` или `SANDBOX_PY23_TASK`.

В качестве примера можно посмотреть [TEST_UNKNOWN_TASK](https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/tests/test_unknown_task).

## Бинарные задачи на Python 3 { #python3 }
### Отличия python3 задач от бинарных на python2 { #python3-difference }
1. Для бинарных задач на python 3 нужно использовать макрос **SANDBOX_PY3_TASK** вместо **SANDBOX_PY23_TASK** в ya.make файле. Он аналогичен макросу **SANDBOX_TASK** (который используется для задач на python2).
    ```bash
    ~/arcadia/sandbox/projects/sandbox/test_task_2/test_task_2_py3$ cat ya.make
    SANDBOX_PY3_TASK(test_task_2)  # макрос для маркировки python3 задачи с указанием именя исполняемого файла в скобках

    OWNER(g:sandbox-dev)

    PEERDIR(
        sandbox/projects/sandbox/test_task_2
    )

    END()
    ```
2. Цель **SANDBOX_PY3_TASK** нельзя и не нужно пытаться подключать по PEERDIR к библиотеке `projects/sandbox`,
   которая является **PY2_LIBRARY** и код которой попадает на сервера sandbox.
   По этой причине в коде на python 3 **можно делать импорты сторонних библиотек на глобальном уровне**.
3. Задача на Python 3 должна вызывать server-side методы с использованием бинарника (опция `--enable-taskbox`).
   Эта технология (**taskbox**) имеет недоработки: например, задачу можно создать только через вызов командной строке,
   создание задачи из UI возможно только при определённых условиях.
4. локальном sandbox при импорте будут игнорироваться директории под макросами **PY3_LIBRARY**, **PY3_PROGRAM**, **SANDBOX_PY3_TASK** и **PY3TEST**.
   То же самое будет происходить при сборке архива с кодом задач при помощи **BUILD_SANDBOX_TASKS** в режиме RAW (svn checkout из Аркадии).

### Python3 задачи с серверными python2 методами { #python3-with-python2-server-side }
Чтобы не использовать server-side методы из бинаря (taskbox), основной класс задачи должен быть совместим с Python 2. Для этого директория с классом задачи должна быть оформлена в виде **PY23_LIBRARY**  и подключена к `sandbox/projects`  (транзитивно), чтобы API-сервер мог узнать о существовании этого типа задачи.

Пример ya.make задачи [TEST_TASK_2](https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/sandbox/test_task_2):

```bash
PY23_LIBRARY()

OWNER(g:sandbox-dev)

PY_SRCS(
    __init__.py
)

PEERDIR(
    sandbox/projects/common/constants
    sandbox/projects/sandbox/resources
)

END()
```

Для сборки же самого бинарника на python 3 нужно создать отдельную директорию, содержащую только ya.make  файл с макросом **SANDBOX_PY3_TASK**. В зависимости этого ya.make нужно добавить все **PY3_LIBRARY** , которые будут использоваться внутри не server-side-методов (on_execute, on_prepare), а также PY23-библиотеку с объявлением основного класса задачи.

Для использования сторонних библиотек внутри серверных методов такая организация библиотек не подойдёт. В этом случае надо использовать taskbox.

Пример ya.make для поддиректории `test_task_2_py_3` задачи **TEST_TASK_2**:

```bash
SANDBOX_PY3_TASK()
OWNER(g:sandbox-dev)
PEERDIR(
    sandbox/projects/sandbox/test_task_2
)
END()
```

Если собрать эту цель, то полученный бинарь сможет сам себя залить в Sandbox и оформиться в виде ресурса точно так же, как это делают все бинарные задачи.
