# Ресурсы

**Ресурс** - это файл или директория вместе с набором некоторых метаданных, создаваемые задачей для оформления результатов своей работы. В одном ресурсе можно хранить один файл или целая директория, текстовые или двоичные данные. Все ресурсы хранятся в едином **хранилище ресурсов**. Каждая задача может **скачивать** из хранилища уже существующие ресурсы или **сохранять** в процессе работы новые данные в виде ресурса. Примеры того, что может лежать в ресурсах:

* Логи выполнения задачи
* Конфигурационный файл в формате JSON
* Образ виртуальной машины
* Debian пакет
* Отчет о тестировании

Каждому ресурсу при создании присваивается **уникальный числовой идентификатор** (например, 1749728533).

![Основные свойства ресурса](img/resource-properties.png "Основные свойства ресурса")

## Список ресурсов { #list }

Список ресурсов можно посмотреть на странице [Resources](https://sandbox.yandex-team.ru/resources). На странице можно фильтровать ресурсы по различным параметрам.

![Список ресурсов](img/resource-list.png "Список ресурсов")

## Тип ресурса { #type }

У каждого ресурса есть **тип**, например, **TASK_LOGS** или **SELENIUM_QEMU_IMAGE**.
Название типа задается таким, чтобы можно было легко понять, что хранится в таком ресурсе.
Пользователь может объявить собственные типы ресурсов в коде и использовать такие ресурсы в своих задачах.
Тип также позволяет фильтровать список ресурсов, отображая ресурсы, хранящие похожие данные.

{% note info %}

Имя типа ресурса всегда определяется в верхнем регистре.

{% endnote %}

![Фильтрация ресурсов по типу](img/resource-type.png "Фильтрация ресурсов по типу")

## Состояние ресурса { #status }

У каждого ресурса есть состояние. В таблице ниже приведены все возможные состояния ресурсов.

Состояние | Описание
:--- | :---
**BROKEN** | Ресурс был создан задачей, которая упала с ошибкой, либо у ресурса не осталось ни одного источника (например хост, на котором он бы создан, стал недоступен, а ресурс не был забэкаплен).
**DELETED** | Ресурс был удален по истечению ttl.
**NOT_READY** | Ресурс создан, но данные для него еще не готовы (создавшая его задача еще не завершила работу). Использовать эти данные пока нельзя.
**READY** | Ресурс создан, данные успешно созданы и готовы к использованию.

Ресурсы в состоянии **BROKEN** и **DELETED** могут быть просмотрены через браузер, но получить их данные (например, через HTTP-прокси) нельзя.
Это специальная предосторожность, чтобы внешние системы (в том числе системы деплоя) не могли использовать некорректные данные.
Если вы настойчиво хотите обойти данное ограничение, воспользуйтесь HTTP-заголовком `X-Ignore-State`.


## Атрибуты { #attributes }

Каждому ресурсу можно дополнительно назначить один или несколько атрибутов. **Атрибуты** - это простые пары ключ-значение, хранящие метаданные.
Атрибуты можно читать и использовать в задачах. Например, если в ресурсе хранится образ виртуальной машины,
в атрибутах можно сохранить требования к вычислительным ресурсам (ядрам, памяти и диску).
Также можно использовать атрибуты для точного поиска ресурса, предварительно сохранив туда, например, информацию о коммите или релизе.
У каждого атрибута есть тип (строка, целое и вещественное число, булево значение), который устанавливается при объявлении ресурса в коде и проверяется при сохранении значения.

Существует набор зарезервированных (**системных**) названий атрибутов:

Атрибут | Тип | Описание
:--- | :--- | :---
**any_arch** | `boolean` | Зависит ли ресурс от архитектуры (`True` - не зависит, `False` - зависит) <br />**default:** `True`
**auto_backup** | `boolean` | Создавать резервную копию ресурса в Sandbox storage сразу после создания <br />**default:** `False`, для ресурсов с `ttl=inf` автоматически выставляется в `True`
**backup_task** | `integer` | Идентификатор задачи, которая скопирует ресурс в Sandbox storage
**calc_md5** | `boolean` | Выполнять ли подсчет MD5 для ресурсов из одного файла <br />**default:** `True`
**deprecated** | `boolean` | Пометка о том, что тип ресурса устарел <br />**default:** `True`
**executable** | `boolean` | Пометка о том, что файл ресурс является исполняемым <br />**default:** `False`
**on_restart** | `enum` | Что делать с ресурсом в момент перезапуска задачи, которая его создала: удалить, переключить в состояние **NOT_READY** или не менять состояние. Допустимые значения перечислены в классе [sandbox.common.types.resource.RestartPolicy](https://a.yandex-team.ru/arc_vcs/sandbox/common/types/resource.py?rev=r7116596#L52) <br />**default:** `RESET`
**releasable** | `boolean` | Можно ли создавать релиз ресурса <br />**default:** `False`
**release** | `string` | Тип релиза. Рекомендуемые значения перечислены в классе [sandbox.common.types.task.ReleaseStatus](https://a.yandex-team.ru/arc_vcs/sandbox/common/types/task.py?rev=r8088426#L425) <br />**default:** `None`
**releasers** | `list<string>` | Список сотрудников или групп, которые имеют права на релиз ресурсов данного типа. <br />**default:** `None`
**release_subscribers** | `list<string>` | Список сотрудников или групп, которым придёт уведомление при релизе ресурса данного типа. <br />**default:** `None`
**shard** | `boolean` | Специальный атрибут для поисковых шардов <br />**default:** `False`
**share** | `boolean` | Нужно ли регистрировать ресурс в [Skynet Skybone](https://wiki.yandex-team.ru/skynet/services/skybone/) <br />**default:** `True`
**sync_upload_to_mds** | `boolean` | Включить или выключить синхронное копирование данных в [MDS/S3](https://wiki.yandex-team.ru/mds/s3-api/), `None` отключает загрузку в MDS для данного ресурса <br />**default:** `False`
**pack_tar** | `boolean` | [Автоматическая упаковка ресурса](#auto-pack) при его его завершении <br />**default:** `False`
**ttl** | `integer` | [Время жизни ресурса](#ttl) в днях (`inf` для бесконечного хранения) <br />**default:** `14`
**ttl_on_release** | `integer` | [Время жизни ресурса](#ttl) в днях после релиза этого ресурса (возможность переопределить поведение по умолчанию - `inf`) <br />**default:** `None`

## Политики для ресурсов при рестарте задачи

За политику, описывающую жизнь ресурса при рестарте задачи, отвечает атрибут ресурса `restart_policy` (он же `on_restart`). По умолчанию он имеет значение `RESET`. Может принимать значения:
1. `RESET` - ресурс при рестарте задачи переходит в статус `NOT_READY`.
2. `DELETE` - при рестарте задачи ресурс удаляется.
3. `IGNORE` - при рестарте задачи ресурс не трогается.

Нужно выбирать правильную политику, так как если задача создает после рестарта ресурс с тем же путем, что и зарегистрированный ею ранее ресурс, но с другой метаинформацией (с другими атрибутами или типом), то задача упадет в статус `EXCEPTION`.

## Время жизни ресурса { #ttl }

По умолчанию ресурсы хранятся **14 дней**, а затем автоматически удаляются.
Если требуется установить другое время хранения, следует указать **время в днях** системном атрибуте `ttl`, например:

```
ttl = 7  # Хранится 1 неделю
ttl = inf  # Хранится бесконечно
```

Значение `ttl = inf` приводит к тому, что ресурс хранится бесконечно.

Когда заканчивается время жизни ресурса, его данные помечаются как удаленные, а сам ресурс переходит в состояние **DELETED**.
Ресурсы удаляются (переходят в состоянии **DELETED**) через `ttl` дней с момента последнего использования:
- скачивания через [прокси](https://proxy.sandbox.yandex-team.ru/),
- выполнения метода [touch](https://sandbox.yandex-team.ru/media/swagger-ui/index.html#/resource/resource_touch_post) в API
- или скачивания ресурса задачами на агентах Sandbox средствами SDK.

Скачивание ресурса через skynet, rsync или напрямую по http-ссылке с клиента **не обновляют время последнего использования**.

Ресурсы, помеченные как **DELETED**, могут храниться на хостах ещё какое-то время. Однако, эти ресурсы являются
первыми кандидатами на удаление в случае, если Sandbox-агенту не будет хватать места для исполнения задач.
До момента физического удаления файлов с диска ресурсы возможно восстановить.
Для этого обратитесь в поддержку Sandbox.


## Аудит ресурсов { #audit }

Данные по изменению ресурсов отгружаются в [YT](https://yt.yandex-team.ru/hahn/navigation?path=//home/sandbox/production/logs/core/resources_audit).

Пример YQL запроса для аудита конкретного ресурса:
```sql
USE hahn;
SELECT
    `id`,
    `timestamp`,
    `state`,
    `attributes`,
    `author`,
    `message`
FROM hahn.RANGE(`home/sandbox/production/logs/core/resources_audit`, '2022-05-20', '2022-05-25')
WHERE id == 16247791
ORDER BY timestamp
LIMIT 100;
```

## Загрузка (upload) ресурсов { #upload }

Существует несколько способов загрузить ресурс в Sandbox:

1. С помощью команды `ya` {#ya-upload}

    ```bash
    $ ya upload /path/to/file -T MY_RESOURCE_TYPE --owner MY_GROUP_NAME --ttl 30 -d 'my resource description'
    ```

    Для авторизации `ya` по умолчанию использует временные токены, которые размениваются на ssh-ключ запускающего.
    Однако, в команду можно передать токен явно через аргумент `--token AQAD-XXXXXX`.
    Получить токен можно [по ссылке](https://sandbox.yandex-team.ru/oauth/).
    Если на машине, где выполняется команда, установлен [Skynet](https://wiki.yandex-team.ru/skynet/), то для загрузки большого количества данных рекомендуется использовать флаг `--skynet` – этот способ загрузки надёжнее. Утилита пишет логи в файл вида `~/.ya/logs/yyyy-MM-dd/HH24-mm-ss.fooooo.log`.

2. С помощью задачи [CREATE_TEXT_RESOURCE](https://sandbox.yandex-team.ru/tasks?type=CREATE_TEXT_RESOURCE). Подходит для загрузки небольших текстовых файлов, содержимое которых нужно вставить в соответствующий параметр задачи.

3. С помощью задачи [REMOTE_COPY_RESOURCE_2](https://sandbox.yandex-team.ru/tasks?type=REMOTE_COPY_RESOURCE_2). Позволяет создать ресурс из данных, доступных по [HTTP](https://en.wikipedia.org/wiki/Hypertext_Transfer_Protocol), [rcp](https://en.wikipedia.org/wiki/Berkeley_r-commands), [scp](https://en.wikipedia.org/wiki/SSH_(Secure_Shell)), [rsync](https://en.wikipedia.org/wiki/Rsync), [skynet](https://wiki.yandex-team.ru/skynet/) и так далее.

## Скачивание ресурсов { #download }

![Ссылки для скачивания ресурса](img/resource-download.png "Ссылки для скачивания ресурса")

### Скачивание при помощи Skybone { #download-via-skybone }
Для доставки ресурсов на хосты Sandbox использует [Skynet Skybone (он же Copier)](https://wiki.yandex-team.ru/skynet/services/skybone/) – сервис для обмена данными, работающий на протоколе, похожем на torrent.

Если вам требуется скачивать ресурс **в автоматизированных процессах**, рекомендуется пользоваться именно Skybone:
```bash
$ sky get -p rbtorrent:4e1f025a34ade0d459bd8ff22bbaf5daa5f496a9 # ID можно взять на странице ресурса
```

### Скачивание при помощи http-proxy { #download-via-http }

Если вам требуется **разово** скачать ресурс для использования за пределами Sandbox,
то для этого можно воспользоваться сервисом Sandbox Proxy.
Сервис позволяет находить и скачивать ресурсы разными способами в зависимости от содержимого.
Подробнее читайте [в описании сервиса](proxy.md#features).

## Хранение ресурсов { #storage }

При использовании задачами ресурсы синхронизируются на хосты и удаляются в случае, если свободного места не достаточно для исполнения новых задач.
Таким образом, на хостах всегда существует некоторый ресурсный кэш, который позволяет экономит время на скачивание часто используемых ресурсов.

### Бекап и репликация { #backup }

Данные ресурсов, у которых в свойствах указан атрибут `auto_backup=True`, хранятся с репликацией.

Исторически Sandbox для хранения использовал собственные выделенные сервера (хосты с именами `sandbox-storageXX`).
Однако, поддержание хранилища довольно затратно и потому Sandbox двигается в сторону использования [MDS](https://wiki.yandex-team.ru/mds/) для хранения данных.
Поэтому после перехода ресурса в состояние **READY** запускается процесс первичной репликации – ресурс уезжает в MDS.


### Бекап в MDS { #backup-in-mds }
Бекап в MDS возможен только для ресурсов, группа которых (owner) привязана к ABC-сервису и имеет корзину (bucket в терминологии S3). Если вы не уверены, есть ли у вашего сервиса корзина или хотите ее создать - вам нужно создать [тикет](https://st.yandex-team.ru/createTicket?queue=DEVTOOLSSUPPORT&_form=65090) в поддержку. В этом случае Sandbox в собственной квоте создаёт корзину в S3 MDS с именем `sandbox-<ABC-id>`, куда и будут загружаться ресурсы. Созданные таким образом корзины видны в виде ресурса `квота в хранилище` провайдера `Sandbox` на вкладке `Квоты` сервиса в ABC. Там же можно создать дополнительные корзины, но для того чтобы использовать, их нужно привязать к группе в Sandbox, пока что это можно сделать только через обращение в [поддержку Sandbox](https://st.yandex-team.ru/createTicket?queue=DEVTOOLSSUPPORT&_form=65090), но в скором времени в UI Sandbox появится возможность делать это самостоятельно.

Если загрузка в MDS выполняется без указания группы (например, так умеет `ya upload`) или от имени группы, которую не привязали к ABC (для новых групп это невозможно), то данные будут забекаплены **в общую квоту**.
Загружаемые в общую квоту ресурсы попадают в общую S3 корзину, для которой действует политика вытеснения. При исчерпании квоты из корзины будут удаляться данные для наиболее старых ресурсов.
Сами ресурсы при этом останутся целыми, так как их данные останутся на других источниках. Однако, надёжность хранения может ухудшиться.
По этой причине, для хранения в общей квоте не допускаются ресурсы с большим значением `ttl`.
При попытке загрузки ресурсов с, например, `ttl=inf` в общую корзину будет происходить ошибка.

Если же Sandbox-группа привязана к ABC-сервису и для неё заведена собственная корзина, то ограничения на `ttl` не будет.

### Форсированный бекап ресурсов { #sync-upload-to-mds }
Если на какие-то ресурсы завязаны важные процессы и необходимы обеспечить надёжное реплицирование данных, то первичный бекап в MDS можно выполнить синхронно.
Для этого есть атрибут ресурса `sync_upload_to_mds`. Он может принимать два значения:
- `sync_upload_to_mds=True`. В этом случае сначала происходит загрузка данных в MDS, а потом ресурс переключается в состояние **READY**.
  Этот процесс выполняется в рамках работы задачи, создающей ресурс, и потому удлиняет время её работы.
- `sync_upload_to_mds=False` В этом случае ресурс переводится в состояние **READY**, а задача корректно завершает свою работу до бекапа данных.
  Ресурс будет загружен в MDS, но на это может уйти некоторое время – обычно не более часа, но многое зависит от нагрузки на корзину S3 MDS.

При нехватке места в корзине:
- В случае синхронной загрузки ресурсов задачи будут завершаться с ошибкой.
- При асинхронной загрузке ресурсы будут переходить в статус **READY**, а задача будет штатно завершаться.
  Загрузка в S3 MDS произойдет только тогда, когда появится достаточное свободное место.

### Как следить за количеством свободного места { #storage-monitoring }
Нехватка места в корзине приводит к тому, что создаваемые задачами ресурсы не будут реплицированы.
Это ухудшает надёжность хранения, а в случае синхронного бекапа (`sync_upload_to_mds=True`) приводит к падению задач.
Поэтому важно не допускать нехватки места.

Каждая группа Sandbox привязывается к одной корзине в S3 MDS, имя которой соответствует шаблону `sandbox-<ABC service id>`.
Привязка осуществляется на основе ABC-сервиса, к которому привязана группа.
Например, для группы `BROWSER`, привязанной к сервису [browserinfra](https://abc.yandex-team.ru/services/browserinfra) с `id=1769`,
имя корзины будет `sandbox-1769`.

Текущее состояние корзины можно увидеть на странице любой группы ([пример для группы BROWSER](https://sandbox.yandex-team.ru/admin/groups/BROWSER)).
При этом группы, привязанные к одному и тому же ABC-сервису, будут использовать одну и ту же корзину.

Графики потребления стораджа можно также найти [на дашборде в Grafana](https://grafana.yandex-team.ru/d/3MdO34WMk/resources-statistics-production)
(не забудьте выбрать конкретную группу)
и [в Solomon](https://solomon.yandex-team.ru/?project=sandbox&cluster=sandbox_consumption&service=sandbox_consumption&l.sensor=mds_quota&l.mds_bucket_name=sandbox-1769&graph=auto)

Стоит отметить, что у каждой корзины в S3 MDS есть квота на использование API.
Поэтому при интенсивном скачивании или создании ресурсов возможны ответы 429.
Это может приводить к тому, что ресурсы будут медленнее реплицироваться в MDS или не будут скачиваться по ссылкам `proxy.sandbox.yandex-team.ru/`.
Поисследовать API-нагрузку на корзины в S3 можно в при помощи [специального сервиса](https://s3-int.chv.yandex-team.ru/?componentSettings=%7B%22projects%22:%7B%22query%22:%22sandbox-%22%7D%7D),
предоставляемого командой MDS ([wiki](https://wiki.yandex-team.ru/mds/s3-api/#clickhouseviewerdljazaprosovvs3)).

#### Настройка уведомлений { #storage-notifications }
Информация о состоянии корзин в S3 можно найти в Solomon по адресу:
```
project=sandbox&
cluster=sandbox_consumption&
service=sandbox_consumption&
sensor=mds_quota&
mds_bucket_name=sandbox-<ABC service id>
```

Соответственно, для этих графиков легко настроить алерты и нотификации [при помощи стандартных средств Solomon](https://solomon.yandex-team.ru/docs/concepts/alerting) ([пример настроенного алерта](https://solomon.yandex-team.ru/admin/projects/sandbox/alerts/sandbox-tmp-bucket-storage)).

#### Дефолтные уведомления от  Sandbox { #default-notifications }
В Sandbox предусмотрена нотификация по умолчанию на случай, если место в корзине заканчивается.
Если использовано 90% или более от ёмкости корзины, то отправляется e-mail уведомление ответственным за ресурсы и управляющим квотами (роли 742, 1553), либо, если их нет, то всем ответственным за сервис в ABC.
Такие уведомления можно отключить при помощи API-запроса (нужно подправить имя корзины в path запроса):

`curl -X PUT "https://sandbox.yandex-team.ru/api/v1.0/resource/buckets/sandbox-000" -H "Content-Type: application/json" -d "{ \"ignore_bucket_exhaust\": true}"`

### Что делать, если место закончилось { #lack-of-storage }

- [Найти и удалить ненужные ресурсы большого размера](#find-and-delete)
- [Увеличить размер хранилища](#increase-storage)

#### Как найти и удалить ресурсы { #find-and-delete }
Чтобы найти типы ресурсов, которые занимают больше всего места у конкретной группы, можно взглянуть на соответствующий график
([для группы SANDBOX](https://grafana.yandex-team.ru/d/3MdO34WMk/resources-statistics-production?viewPanel=10&orgId=1&var-owner=SANDBOX&var-type=All&var-count_deleted=no))
нашего основного дашборда ресурсов в графане
([для группы SANDBOX](https://grafana.yandex-team.ru/d/3MdO34WMk/resources-statistics-production?orgId=1&var-owner=SANDBOX&var-type=All&var-count_deleted=no)).

Если вы знаете, какие ресурсы надо удалить, то это можно легко сделать через UI [на основной странице ресурсов](https://sandbox.yandex-team.ru/resources).
Однако, если у ресурсов выставлен `ttl=inf`, то для удаления надо сначала снять этот атрибут.

В этом случае, а также если удаляемых ресурсов много, очистку удобнее делать с помощью REST API.
Для этого можно воспользоваться консолью в [sandbox/scripts/api](https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/scripts/api).

{% cut "Пример запуска кода" %}

Не забудьте отредактировать фильтр поиска ресурсов в функции `get_resources_chunk`.

```ipython
r-vetrov@rve:~/arc/arcadia/sandbox/scripts/api$ ya py
Creating temporary PY3_PROGRAM /home/r-vetrov/arc/arcadia/junk/r-vetrov/_ya_py
Baking shell with:
 sandbox/scripts/api/py3 (program)
Ok
Authenticated as r-vetrov
Python 3.9.5 (default, Jun 15 2021, 14:06:02)
Type 'copyright', 'credits' or 'license' for more information
IPython 7.24.1 -- An enhanced Interactive Python. Type '?' for help.

In [1]:

import datetime

def get_resources_chunk(chunk_size):
    now = datetime.datetime.utcnow()
    since = now - datetime.timedelta(days=10 * 365)
    till = now - datetime.timedelta(days=14)

    return api.resource.read(
        owner="SANDBOX",
        type="SANDBOX_TEST_RESULTS",
        attrs={"ttl": "inf"},
        state="READY",
        created="{}..{}".format(since.isoformat(), till.isoformat()),
        limit=chunk_size,
    )

def replace_attribute(rid, key, value):
    print("Set attribute {}={} for resource {}".format(key, value, rid))
    api.resource[rid].attribute[key] = {"value": value}

def remove_some_resources(limit):
    resp = get_resources_chunk(limit)
    print("Removing", len(resp["items"]), "resource out of", resp["total"])
    for res in resp["items"]:
        if res["attributes"].get("ttl") == "inf":
            # Drop infinite ttl to allow deleting of resource
            replace_attribute(res["id"], "ttl", "14")

    ids = [_["id"] for _ in resp["items"]]
    print("Removing", ids)
    api.batch.resources["delete"].update(
        id=ids,
        comment="Remove extra resources",
    )

In [2]: remove_some_resources(1)
Removing 1 resource out of 7
Removing [2211839871]

In [3]:
```

{% endcut %}

Учтите, что ресурсы удаляются из MDS не сразу, а через 1 день со времени последнего изменения.
Это сделано для избежания потери данных при случайных ошибках и других манипуляциях над ресурсами.

При необходимости, можно форсировать очистку ресурсов, используя REST API
  [/batch/bucket/cleanup](https://sandbox.yandex-team.ru/media/swagger-ui/index.html#/batch/buckets_op_put).
API принимает список бакетов (например, `[sandbox-1769]`) и проставляет дополнительные метки
  на ресурсы в состояниях `DELETED` и `BROKEN` из соответствующих корзин,
  после чего в течение нескольких минут помеченные ресурсы будут удалены.
Эту команду могут выполнить владельцы ABC-сервиса, к которому привязана корзина, а также менеджеры аппаратных ресурсов (ID роли в ABC: 742) сервиса.


#### Увеличить размер хранилища { #increse-storage }
Увеличения размера хранилища делается заказом соответствующего ресурса во время общего заказа ресурсов в компании (происходит раз в год).
Если хранилище надо увеличить быстрее, то можно передать квоту MDS сервису Sandbox. После этого мы увеличим размер вашей корзины на такой же объём.
В экстренных случаях возможно временное увеличение размера хранилища за счёт коммунальной квоты.

### Автоматическая упаковка ресурса { #auto-pack }

Ресурс может быть упакован в архив средствами sdk.
Для этого нужно выставить ему атрибут `pack_tar=1` в классе ресурса или передать такой же аргумент конструктора.
Если ресурс требуется также сжать, то надо выставить значение `pack_tar=2`.
Допустимые значения этого атрибута определены в перечислении [sandbox.common.mds.compression.base.CompressionType](https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/common/mds/compression/base.py?rev=r8249147#L11).

При наличии атрибута `pack_tar` ресурс **при переходе в статус READY** будет упакован в соответствующий архив.
При этом содержимое архива будет эффективно (быстро) открываться для просмотра по proxy-ссылкам.


## Удаление объектов ресурсов из базы данных { #purge-policy }

1. Ресурсы в состоянии `READY`, `NOT_READY` или `BROKEN` переводятся в состояние `DELETED` по истечении их [времени жизни](#ttl) (ttl)
    или **через 14 дней после последнего использования**, если время жизни не указано.
2. Ресурсы в состоянии `DELETED` полностью удаляются из базы данных **через 14 дней**.
