import os
import math
import time
import logging

from sandbox.agentr import client as agentr_client

from sandbox.executor.commands import base as base_commands

from sandbox.common import mds as common_mds
from sandbox.common import rest as common_rest
from sandbox.common import config as common_config
from sandbox.common import format as common_format
from sandbox.common import itertools as common_itertools
from sandbox.common import threading as common_threading

from sandbox.common.vcs import cache as vcs_cache

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc

import six

if six.PY2:
    import subprocess32 as sp
else:
    import subprocess as sp


logger = logging.getLogger("executor")


class ClearTasksDir(base_commands.ExecutableBase):
    SYSTEM_TMPDIR = "/tmp"
    SERVICE_TMPDIR = "/var/tmp"
    MIN_TMPFILE_AGE = 900

    def __init__(self, *args, **kwargs):
        super(ClearTasksDir, self).__init__()
        logger.debug("Clear client")

    @classmethod
    def __remove_path(cls, path):
        if not path:
            return
        try:
            with open(os.devnull, "wb") as devnull:
                sp.check_call(common_itertools.chain(("/bin/rm", "-rf"), path), stderr=devnull)
        except (sp.CalledProcessError, OSError) as ex:
            logger.error("Cannot remove %r: %s", path, ex)

    @classmethod
    def __empty_dir(cls, path):
        now = time.time()
        logger.debug("Emptying directory %r", path)
        trash = []
        for e in os.listdir(path):
            e = os.path.join(path, e)
            st = os.lstat(path)
            if st.st_mtime + cls.MIN_TMPFILE_AGE < now:
                trash.append(e)
        cls.__remove_path(trash)

    def execute(self):
        self.__empty_dir(self.SYSTEM_TMPDIR)
        kind = agentr_client.Service(logger).cleanup().kind
        if kind == ctc.RemovableResources.DELETED:
            return ctt.Status.SUCCESS

        self.__empty_dir(common_config.Registry().client.tasks.build_cache_dir)
        self.__empty_dir(self.SERVICE_TMPDIR)
        vcs_cache.VCSCache(logger=logger).clean()

        if not kind:
            logger.info("Nothing to clear.")
            return ctt.Status.EXCEPTION
        logger.debug('Clearing is done')
        return ctt.Status.SUCCESS


class ServiceOnIdle(base_commands.ExecutableBase):
    def __init__(self, *_, **kwargs):
        super(ServiceOnIdle, self).__init__()
        self.token = kwargs.get("mds_token")
        logger.debug("Service action on idle client")

    def execute(self):
        if not self.token:
            logger.info("No MDS token provided - nothing to do.")
            return ctt.Status.EXCEPTION

        if ctc.Tag.NEW_LAYOUT in common_config.Registry().client.tags:
            logger.info("Not supported on new layout hosts.")
            return ctt.Status.EXCEPTION

        proxy = common_rest.Client(component=ctm.Component.CLIENT)
        mds_settings = common_config.Registry().client.mds
        ids = self.rest_client.resource.backup.create(host=common_config.Registry().this.id, limit=3)
        if not ids:
            logger.info("Nothing to backup.")
            return ctt.Status.EXCEPTION

        kamikadze = common_threading.KamikadzeThread(proxy.DEFAULT_TIMEOUT * 2, logger)
        kamikadze.start()
        logger.debug("Resources for backup: %r", ids)
        for rid in ids:
            r = proxy.resource[rid][:]
            path = os.path.join(
                common_config.Registry().client.tasks.data_dir,
                *(common_itertools.chain(ctt.relpath(r["task"]["id"]), r["file_name"].rstrip(os.path.sep)))
            )
            basedir = os.path.dirname(path)
            logger.info("Uploading resource #%s (%s) located at %s", rid, r["type"], path)

            ttl = max(60, int(math.ceil(float(r["size"]) / mds_settings.up.min_speed)))
            kamikadze.ttl = ttl * 2  # Provide additional time for tarball creation.
            # MDS is very "modern" service - it does not accept requests without "Content-Length" header,
            # so I have to create temporary file first.
            tmpfile = os.path.join(mds_settings.tmpdir, "{}.tar.gz".format(rid))
            cmd = ["tar", "-C", basedir, "-zcf", tmpfile, os.path.basename(path)]
            logger.debug("Creating temporary archive at %s by %r", tmpfile, cmd)
            if os.path.exists(tmpfile):
                os.unlink(tmpfile)

            p = sp.Popen(cmd, stderr=sp.PIPE, stdout=sp.PIPE)
            stdout, stderr = p.communicate()

            if p.returncode:
                logger.error(
                    "Temporary archive creation failed with code %d.\nSTDERR:\n%s\nSTDOUT:\n%s",
                    p.returncode, stderr, stdout
                )
                continue

            size = os.stat(tmpfile).st_size
            logger.debug(
                "Uploading data of size %s (%s compressed)",
                common_format.size2str(r["size"]), common_format.size2str(size)
            )
            try:
                key = common_mds.MDS().upload(
                    tmpfile, "{}.tar.gz".format(rid), token=self.token, size=size, timeout=ttl, logger=logger
                )
            except common_mds.MDS.Exception as _:
                logger.error(str(_))
                continue
            finally:
                os.unlink(tmpfile)
            logger.info("Uploaded data key is %s. Adding appropriate resource attribute.", key)
            proxy.resource[rid].attribute(name="mds", value=key)

        kamikadze.stop().join()
        logger.debug("Service action finished.")
        return ctt.Status.SUCCESS


CMD2CLASS = {
    ctc.Command.CLEAR: ClearTasksDir,
    ctc.Command.IDLE: ServiceOnIdle,
}
