import logging
import itertools as it
import functools as ft

import six

from sandbox.common import log as common_log
from sandbox.common import data as common_data
from sandbox.common import urls as common_urls
from sandbox.common import config as common_config

import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr

from sandbox import sdk2

logger = logging.getLogger("executor")


class SDK2ResourceAdapter(object):
    def __init__(self, resource, task, register_in_agentr=True, for_parent=False):
        self.__resource = resource
        self.__abs_path = str(task.path(self.__resource.path))
        if register_in_agentr and resource.state == ctr.State.NOT_READY:
            cls = type(resource)
            task.agentr.resource_register_meta(
                self.__abs_path, resource.__meta__,
                share=cls.share, service=issubclass(cls, sdk2.ServiceResource)
            )

    @property
    def type(self):
        return type(self.__resource)

    def abs_path(self):
        return self.__abs_path


class SDK2TaskAdapter(object):
    taskLogger = None

    def __init__(self, task, parent_id, cmd):
        self.__task = task
        self.__cmd = cmd
        self.__parent_id = parent_id
        self.ctx = task.Context.__values__
        self.platform = None
        self._subproc_list = []

    def __getattr__(self, item):
        if item.startswith("__"):
            return super(SDK2TaskAdapter, self).__getattribute__(item)
        return getattr(self.__task, item)

    def __repr__(self):
        return repr(self.__task)

    def abs_path(self, *args):
        return str(self.__task.path(*args))

    def set_cmd(self, cmd):
        self.__cmd = cmd

    @property
    def task(self):
        return self.__task

    @property
    def type(self):
        return self.__task.type.name

    @property
    def ramdrive(self):
        return self.__task.ramdrive

    @ramdrive.setter
    def ramdrive(self, value):
        if value and value.path:
            value = value._replace(path=sdk2.Path(value.path))
        self.__task.ramdrive = value

    @property
    def container(self):
        return self.__task.container

    @container.setter
    def container(self, value):
        self.__task.container = value

    @property
    def agentr(self):
        return self.__task.agentr

    @agentr.setter
    def agentr(self, value):
        self.__task.agentr = value

    @property
    def _log_resource(self):
        return self.__task.log_resource

    @property
    def log_resource(self):
        return self.__task.log_resource

    @log_resource.setter
    def log_resource(self, value):
        self.__task.log_resource = value

    @property
    def status(self):
        return self.__task.status

    @status.setter
    def status(self, value):
        self.__task._Task__status = value

    @staticmethod
    def postprocess():
        sdk2.helpers.ProcessRegistry.finish()

    def log_path(self, *args):
        return str(self.__task.log_path(*args))

    def initLogger(self):
        """ Set root logger """
        if self.taskLogger is None and self.id:
            self.taskLogger = logging.getLogger()
            self.taskLogger.propagate = False
            handler_common = logging.FileHandler(self.log_path(ctt.LogName.COMMON))
            handler_common.setFormatter(logging.Formatter(ctt.TASK_LOG_FORMAT))
            handler_common.addFilter(common_log.TimeDeltaMeasurer())
            handler_common.setLevel(logging.INFO)
            self.taskLogger.addHandler(handler_common)
            self.taskLogger.setLevel(getattr(logging, common_config.Registry().client.executor.log.level))

    @property
    def environment(self):
        return self.__task.Requirements.environments

    @property
    def tcpdump_args(self):
        return self.__task.Parameters.tcpdump_args

    def _sync_resource_via_synchrophazotron(self, resource):
        pass  # TODO

    def list_resources(self, register_in_agentr=True):
        itmap = it.imap if six.PY2 else map
        query = sdk2.Resource.find(task=self.__task)
        return itmap(
            ft.partial(SDK2ResourceAdapter, task=self.__task, register_in_agentr=register_in_agentr),
            query.limit(query.MAX_LIMIT)
        )

    def list_parent_resources(self, register_in_agentr=True):
        if self.__parent_id is None:
            return []

        logger.debug("registering parent resources")
        parent_resources = []
        for cls in self.__task.type.Parameters:
            if issubclass(cls, sdk2.parameters.ParentResource):
                res = getattr(self.__task.Parameters, cls.name, None)
                if res is None:
                    continue
                if cls._get_contexted_attribute("multiple", "multiple"):
                    parent_resources.extend(r for r in res if r.task_id == self.__parent_id)
                elif res.task_id == self.__parent_id:
                    parent_resources.append(res)

        logger.debug("Parent #%d resources: %s", self.__parent_id, parent_resources)
        return [
            SDK2ResourceAdapter(
                resource, self.__task,
                register_in_agentr=register_in_agentr, for_parent=True
            ) for resource in parent_resources
        ]

    def cleanup(self):
        self.agentr.cleanup_work_directory()

    def _mark_resources(self):
        self.agentr.finished(drop_session=False, mark_as_ready=None)

    @staticmethod
    def get_vault_data(owner_or_name, name=None):
        return sdk2.Vault.data(owner_or_name, name)

    @staticmethod
    def current_action(action):
        return sdk2.helpers.ProgressMeter(action)

    def sync_resource(self, resource_id):
        """
            Download resource to current client if it was not already downloaded
            (and touch it in both cases).

            :param resource_id: resource identifier or resource object
            :return: local path to synchronized resource;
                raise SandboxTaskUnknownError exception when sync failed.
        """
        return self.agentr.resource_sync(common_data.obj_to_int(resource_id))

    def create_resource(self, description, resource_path, resource_type,
                        arch=None, attributes=None, owner=None):
        cls = sdk2.Resource[resource_type]
        logger.info('Create resource with type: %s, path: %s,  arch: %s', resource_type, resource_path, arch)
        resource = cls(self.__task, description, resource_path, arch=arch, **(attributes or {}))
        logger.info('task._create_resource result: {0}'.format(resource))
        return resource

    def mark_resource_ready(self, resource_id):
        self.agentr.resource_complete(resource_id)

    def http_url(self):
        return common_urls.get_task_link(self.id)

    def on_finish(self):
        self.__task.on_finish(self.__cmd.prev_status, self.__cmd.status)

    def on_failure(self):
        self.__task.on_failure(self.__cmd.prev_status)

    def on_success(self):
        self.__task.on_success(self.__cmd.prev_status)

    def on_break(self):
        self.__task.on_break(self.__cmd.prev_status, self.__cmd.status)

    def on_timeout(self):
        self.__task.on_timeout(self.__cmd.prev_status)

    def on_wait(self):
        self.__task.on_wait(self.__cmd.prev_status, self.__cmd.status)

    def on_before_timeout(self, seconds):
        self.__task.on_before_timeout(seconds)

    def timeout_checkpoints(self):
        return self.__task.timeout_checkpoints()
