import os
import traceback

import six

from sandbox.common import config as common_config
from sandbox.common import encoding as common_encoding

import sandbox.common.types.misc as ctm


if six.PY2:
    import pathlib2 as pathlib
else:
    import pathlib


_LINK_TEMPLATE = "<a href=\"https://a.yandex-team.ru/arc/trunk/{rel_path}?#L{line}\">{abs_path}</a>"
_ARCADIA_DIRS = ["client", "tasks"]


def _link_file(file, line):
    file = pathlib.Path(file)
    if not file.is_absolute():
        # binary tasks
        rel_path = "arcadia/" + file.as_posix()
    elif common_config.Registry().common.installation != ctm.Installation.LOCAL:
        # ['/', 'home', 'zomb-sandbox', 'client|tasks|...', 'relative', 'path']
        if len(file.parts) < 5 or file.parts[3] not in _ARCADIA_DIRS:
            return file

        # tasks/relative/path or client/sandbox/relative/path
        rel_path = "arcadia/sandbox/" + os.path.sep.join(file.parts[4 + (file.parts[3] == "client"):])
    else:
        # /home/user/.../arcadia/relative/path/from/root
        try:
            rel_path = os.path.sep.join(file.parts[file.parts.index("arcadia"):])
        except ValueError:
            return file

    return _LINK_TEMPLATE.format(rel_path=rel_path, abs_path=file, line=line)


if six.PY3:
    class ArcanumTB(traceback.TracebackException):
        def __init__(
            self, exc_type, exc_value, exc_traceback,
            limit=None, lookup_lines=True, capture_locals=False, _seen=None
        ):
            super(ArcanumTB, self).__init__(
                exc_type, exc_value, exc_traceback,
                limit=limit, lookup_lines=lookup_lines, capture_locals=capture_locals, _seen=_seen
            )
            for f in self.stack:
                f.filename = _link_file(f.filename, f.lineno)

            if self.__cause__:
                self.__cause__ = ArcanumTB(
                    type(exc_value.__cause__), exc_value.__cause__, exc_value.__cause__.__traceback__,
                    limit=limit, lookup_lines=False, capture_locals=capture_locals, _seen=_seen
                )
            if self.__context__:
                self.__context__ = ArcanumTB(
                    type(exc_value.__context__), exc_value.__context__, exc_value.__context__.__traceback__,
                    limit=limit, lookup_lines=False, capture_locals=capture_locals, _seen=_seen
                )

        def format_exception_only(self):
            return (common_encoding.escape(line) for line in (super(ArcanumTB, self).format_exception_only()))


def process_tb(exc_type, exc_val, exc_tb):
    # from traceback.format_exception
    if six.PY2:
        # python 2
        if exc_tb:
            message = (
                "Traceback (most recent call last):\n" +
                "".join(traceback.format_list([
                    (_link_file(frame[0], frame[1]),) + frame[1:]  # frame = (filename, lineno, ...)
                    for frame in traceback.extract_tb(exc_tb)
                ]))
            )
        else:
            message = ""

        message += common_encoding.escape("".join(traceback.format_exception_only(exc_type, exc_val)))
    else:
        # python 3
        message = "".join(ArcanumTB(exc_type, exc_val, exc_tb).format())

    return message
