import os
import six
import time
from sandbox.executor.commands.task import adapters

from sandbox.common import rest as common_rest
from sandbox.common import encoding as common_encoding

import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.sdk2 import legacy

if six.PY2:
    import subprocess32 as sp
else:
    import subprocess as sp


def update_agentr_meta(task):
    meta = task.agentr.meta
    meta["context"] = task.Context.__getstate__() if isinstance(task, adapters.SDK2TaskAdapter) else task.ctx
    meta["status"] = task.status
    task.agentr.meta = meta


def update_task(task, final=True):
    if isinstance(task, adapters.SDK2TaskAdapter):
        update_agentr_meta(task)
        if final:
            task.Context.save()
    else:
        from sandbox.yasandbox import manager
        if final:
            task.check_ctx(task.ctx, check_unpickle=True)
        manager.task_manager.update(task)


def task_set_info(task, info):
    if isinstance(task, adapters.SDK2TaskAdapter):
        task.set_info(info)
    else:
        rest_client = common_rest.Client(component=ctm.Component.EXECUTOR)
        rest_client.task.current.execution = {"description": task.set_info(info)}


def initialize_sdk(vault_key, task):
    common_encoding.setup_default_encoding()
    sdk2.Vault.vault_key = vault_key
    sdk2.yav.Yav.setup(vault_key)
    legacy.current_task = task


class DockerClient:
    DOCKER_EXECUTABLE = "docker"

    def __init__(self, task, config_dir):
        self.socket = self._find_docker_socket(task)
        self.config = config_dir

    def _check_docker_version(self, args=None):  # type: (list[str]) -> bool
        args = args or []
        try:
            with open(os.devnull, "w") as devnull:
                sp.check_call([self.DOCKER_EXECUTABLE] + args + ["version"], stderr=devnull)
            return True
        except sp.CalledProcessError:
            return False
        except Exception:
            raise

    def _find_docker_socket(self, task):
        if not self._check_docker_version():
            docker_socket_locations = [task.abs_path("docker_root/docker.sock")]
            if task.ramdrive:
                docker_socket_locations.append(os.path.join(task.ramdrive.path, "docker_root/docker.sock"))
            for docker_sock in docker_socket_locations:
                if os.path.exists(docker_sock) and self._check_docker_version(args=["-H", "unix://" + docker_sock]):
                    return docker_sock
            else:
                raise Exception("Failed to find docker socket")
        return None

    def _command(self, with_config=False):  # type: (bool) -> list[str]
        return (
            [self.DOCKER_EXECUTABLE] +
            (["-H", "unix://{}".format(self.socket)] if self.socket else []) +
            (["--config", self.config] if with_config and self.config else [])
        )

    def login_to_registry(self, docker_registry, docker_token, retries=3):  # type: (str, str, int) -> str
        for iteration in range(1, retries + 1):
            try:
                login_call = sp.Popen(
                    self._command(with_config=True) + [
                        "login", "-u", "zomb-sandbox", "--password-stdin", docker_registry
                    ],
                    stdin=sp.PIPE, stdout=sp.PIPE, stderr=sp.STDOUT, universal_newlines=True
                )
                out, _ = login_call.communicate(input=docker_token)
                if login_call.returncode != 0:
                    if iteration == retries:
                        return out
                    else:
                        time.sleep(3)
                        continue
                return ""
            except Exception as err:
                if iteration == retries:
                    return str(err)
                else:
                    time.sleep(3)
                    continue

    def find_external_docker_images(self, task_created_time):   # type: (int) -> (dict[str, str])
        docker_images = {}
        output = sp.check_output(
            self._command() + ["images", "--format", "{{.ID}},{{.Repository}},{{.Tag}},{{.CreatedAt}}"],
            stderr=sp.STDOUT, universal_newlines=True
        )
        for line in output.splitlines():
            if not line:
                continue
            image_id, repository, tag, created_at = line.split(",")
            if tag in ("latest", "build"):  # non-versioned updates
                continue
            created_at = created_at.split()
            del created_at[2]  # remove timezone indication (%z) as it's not supported in python2
            created_at = time.strptime(' '.join(created_at), "%Y-%m-%d %H:%M:%S %Z")
            if created_at > task_created_time:  # docker image was built during task
                continue
            if repository.startswith("registry.yandex.net") or repository.startswith("cr.yandex"):
                docker_images[image_id] = "yandex_internal"  # custom value for filtering out
            else:
                docker_images.setdefault(image_id, "{}:{}".format(repository, tag))
        return {
            image_id: image_name
            for image_id, image_name in docker_images.items()
            if image_name != "yandex_internal"
        }

    def tag_image(self, image_id, target_name):  # type: (str, str) -> str
        try:
            sp.check_call(self._command() + ["tag", image_id, target_name])
            return ""
        except Exception as err:
            return str(err)

    def push_image(self, image):  # type: (str) -> str
        try:
            sp.check_call(self._command(with_config=True) + ["push", image])
            return ""
        except Exception as err:
            return str(err)

    def get_disk_usage(self):  # type: () -> str
        return sp.check_output(self._command() + ["system", "df"], stderr=sp.STDOUT, universal_newlines=True).strip()

    def stop_containers(self):  # type: () -> (set[str], list[str])
        output = sp.check_output(
            self._command() + ["ps", "--format", "{{.ID}},{{.Image}}"],
            stderr=sp.STDOUT, universal_newlines=True
        )
        stopped_containers, failed_stopping = set(), list()
        for container in output.splitlines():
            if not container:
                continue
            container_id, image = container.split(",")
            try:
                sp.check_call(self._command() + ["stop", container_id], stderr=sp.STDOUT)
                stopped_containers.add(image)
            except Exception as err:
                failed_stopping.append(
                    "Failed to stop container {} ({}): {}".format(container_id, image, str(err))
                )
        return stopped_containers, failed_stopping

    def remove_unused_data(self):   # type: () -> str
        cleanup_arguments = ["system", "prune", "--force", "--volumes"]
        storage_driver = sp.check_output(
            self._command() + ["info", "-f", '{{.Driver}}'], stderr=sp.STDOUT, universal_newlines=True
        ).strip()
        if storage_driver == "vfs":
            cleanup_arguments.append("--all")
        return sp.check_output(
            self._command() + cleanup_arguments, stderr=sp.STDOUT, universal_newlines=True
        ).strip()
