import os
import sys
import json
import psutil
import logging

import six
from six.moves import cPickle as pickle


# Setup custom picklers explicitly (SANDBOX-7356)
# noinspection PyUnresolvedReferences
import kernel.util.pickle  # noqa

from sandbox.agentr import client as agentr_client

from sandbox.common import log as common_log
from sandbox.common import config as common_config
from sandbox.common import platform as common_platform
from sandbox.common import itertools as common_itertools

import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc


def parse_ipc_args(remove_args_file=True):
    argv = sys.argv[1:]
    if len(argv) != 1:
        raise Exception("Wrong executor args: {!r}".format(argv))
    args_filename = argv[0]
    if not common_itertools.progressive_waiter(0, 1, 5, lambda: os.path.exists(args_filename))[0]:
        raise Exception("{!r} does not exist".format(args_filename))
    with open(args_filename, "rb") as _:
        args = pickle.loads(_.read())
    if remove_args_file:
        os.unlink(args_filename)
    return args


# noinspection PyProtectedMember, PyBroadException
def save_result(result_filename, executor_ctx, logger=None):
    if logger is None:
        logger = logging.getLogger("result_saving")
    try:
        result_filename_tmp = result_filename + "~"
        logger.debug("Saving result to %r", result_filename)
        with open(result_filename_tmp, "w") as f:
            json.dump(executor_ctx, f)
        os.chmod(result_filename_tmp, 0o644)
        os.rename(result_filename_tmp, result_filename)
        logger.debug("Result saved")
    except Exception:
        logger.exception("Cannot write executor result to %r", result_filename)
        os._exit(1)
    # We use dirty os._exit() because executor runs potentially unsafe user code,
    # which we can't be sure will shutdown correctly (e.g. user can spawn non-daemon threads and not join them).
    # As a result all threads are terminated abruptly, global destructors (and __del__ methods) are not called.
    # This can lead to leaving spawned processes alive, but they should be cleaned up by sandbox client.
    os._exit(0)


def container_info(ipc_args):
    # SANDBOX-5358: Forward compatibility method to connect newer agent (client) version with newer executors.
    cnt = ipc_args.get("container", {})
    # noinspection PyProtectedMember
    return ctc.Container(*map(cnt.get, ctc.Container._fields)) if cnt else None


def configure_logger(task_id, iteration, session_token):
    logger = logging.getLogger()
    logger.propagate = False
    agentr = agentr_client.Session(session_token, iteration + 1, 0, logger)
    if task_id and iteration is not None:
        logdir = six.ensure_str(agentr.logdir)
        logfile = os.path.join(logdir, ctt.LogName.DEBUG)
        handler = logging.FileHandler(logfile)
    else:
        logdir = six.ensure_str(common_config.Registry().client.executor.log.root)
        logfile = os.path.join(logdir, common_config.Registry().client.executor.log.name)
        handler = common_log.GZipTimedRotatingFileHandler(logfile)

    handler.setFormatter(logging.Formatter(ctt.TASK_LOG_FORMAT))
    handler.vault_filter = common_log.VaultFilter()
    handler.addFilter(handler.vault_filter)
    handler.addFilter(common_log.TimeDeltaMeasurer())
    handler.setLevel(logging.DEBUG)
    logger.addHandler(handler)
    logger.addHandler(common_log.ExceptionSignalsSenderHandler())
    logger.setLevel(logging.DEBUG)
    agentr.logger = logger
    return agentr, logger, logdir


def get_executor_pid():
    """
    Pass current PID to AgentR for Windows, as it's impossible to retrieve it via socket inside WSL.
    For Linux and OSX it will be retrieved via unix socket.
    """
    return psutil.Process().pid if common_platform.on_windows() else None
