# *-* coding: utf-8 *-*
# This file is part of butterfly
#
# butterfly Copyright (C) 2015  Florian Mounier
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import fcntl
import io
import os
import pty
import random
import signal
import string
import struct
import termios
import tornado.ioloop
import tornado.options
import tornado.process
import tornado.web
import tornado.websocket
from logging import getLogger
from sandbox.fileserver.butterfly import utils, __version__

# Sandbox specific imports
import sandbox.common.fs
import sandbox.common.os
import sandbox.common.config

import sandbox.agentr.client as aclient

import kernel.util.sys.user

log = getLogger('butterfly')
ioloop = tornado.ioloop.IOLoop.instance()
server = utils.User()


# Python 2 backward compatibility
try:
    input = raw_input
except NameError:
    pass


class Terminal(object):
    agentr = aclient.Service(log)

    @classmethod
    def shell_available(cls, task_id):
        return bool(cls.agentr.fileserver_meta[task_id].shell_command)

    def __init__(self, user, path, session, socket, host, ws_handler, task_id=None):
        self.host = host
        self.session = session
        self.fd = None
        self.closed = False
        self.socket = socket
        self.path = path
        self.user = user if user else None
        self.caller = self.callee = None

        self.ws_handler = ws_handler
        self.task_id = task_id

        log.info("Terminal opening with session: %s and socket %r", self.session, self.socket)

        meta = self.agentr.fileserver_meta[self.task_id]
        self.cgroup = meta.cgroup
        self.shell_command = meta.shell_command

        # If local we have the user connecting
        if self.socket.local and self.socket.user is not None:
            self.caller = self.socket.user

        if self.user:
            try:
                self.callee = utils.User(name=self.user)
            except LookupError:
                log.debug(
                    "Can't switch to user %s" % self.user, exc_info=True)
                self.callee = None

        # If no user where given and we are local, keep the same
        # user as the one who opened the socket ie: the one
        # openning a terminal in browser
        if not self.callee and not self.user and self.socket.local:
            self.user = self.callee = self.caller

        motd = self.ws_handler.render_string(
            "motd",
            butterfly=self,
            version=__version__,
            opts=tornado.options.options,
            esc="\x1b",
            csi="\x9b",
            colors=utils.ansi_colors
        ).decode("utf-8").replace("\r", "").replace("\n", "\r\n")
        self.ws_handler.write_message("S" + motd)

        log.info("Forking pty for user %r", self.user)

    def pty(self):
        # Make a "unique" id in 4 bytes
        self.uid = ''.join(
            random.choice(
                string.ascii_lowercase + string.ascii_uppercase +
                string.digits)
            for _ in range(4))

        self.pid, self.fd = pty.fork()
        if self.pid == 0:
            self.determine_user()
            log.debug('Pty forked for user %r caller %r callee %r' % (
                self.user, self.caller, self.callee))
            with kernel.util.sys.user.UserPrivileges():
                self.shell()
        else:
            self.communicate()

    def determine_user(self):
        # if login is not required, we will use the same user as
        # butterfly is executed
        self.callee = self.callee or utils.User()

    def shell(self):
        try:
            os.chdir(self.path or self.callee.dir)
        except Exception:
            log.debug(
                "Can't chdir to %s" % (self.path or self.callee.dir),
                exc_info=True)

        # If local and local user is the same as login user
        # We set the env of the user from the browser
        # Usefull when running as root
        if self.caller == self.callee:
            env = os.environ
            env.update(self.socket.env)
        else:
            # May need more?
            env = {
                "PATH": os.environ["PATH"],
            }
        env["TERM"] = "xterm-256color"
        env["COLORTERM"] = "butterfly"
        env["HOME"] = self.callee.dir
        sandbox_settings = sandbox.common.config.Registry()
        fs_settings = sandbox_settings.client.fileserver

        env["LOCATION"] = (
            "{}://{}/shell/{}/".format(fs_settings.proxy.scheme.http, fs_settings.proxy.host, self.task_id)
            if fs_settings.proxy.host else
            "http://{}:{}/shell/{}/".format(sandbox_settings.this.fqdn, fs_settings.port, self.task_id)
        )
        env["TASK_DIR"] = sandbox.common.fs.get_task_dir(self.task_id)

        try:
            tty = os.ttyname(0).replace("/dev/", "")
        except Exception:
            log.debug("Can't get ttyname", exc_info=True)
            tty = ""
        if self.caller != self.callee:
            try:
                os.chown(os.ttyname(0), self.callee.uid, -1)
            except Exception:
                log.debug("Can't chown ttyname", exc_info=True)

        utils.add_user_info(
            self.uid,
            tty, os.getpid(),
            self.callee.name, self.host)

        if not self.shell_command:
            return
        args = self.shell_command.split(" ")
        # In some cases some shells don't export SHELL var
        # env['SHELL'] = args[0]
        cg = sandbox.common.os.CGroup(self.cgroup)
        if cg and cg.freezer:
            cg.freezer.set_current()
        os.execvpe(args[0], args, env)
        # This process has been replaced

    def communicate(self):
        fcntl.fcntl(self.fd, fcntl.F_SETFL, os.O_NONBLOCK)

        def utf8_error(e):
            log.error(e)

        self.reader = io.open(
            self.fd,
            'rb',
            buffering=0,
            closefd=False
        )
        self.writer = io.open(
            self.fd,
            'wt',
            encoding='utf-8',
            closefd=False
        )
        ioloop.add_handler(
            self.fd, self.shell_handler, ioloop.READ | ioloop.ERROR)

    def write(self, message):
        if not hasattr(self, 'writer'):
            self.close()

        if message[0] == 'R':
            cols, rows = map(int, message[1:].split(','))
            s = struct.pack("HHHH", rows, cols, 0, 0)
            fcntl.ioctl(self.fd, termios.TIOCSWINSZ, s)
            log.info('SIZE (%d, %d)' % (cols, rows))

        elif message[0] == 'S':
            log.debug('WRIT<%r' % message)
            self.writer.write(message[1:])
            self.writer.flush()

    def shell_handler(self, fd, events):
        if not self.shell_command:
            log.info('Closing terminal')
            self.ws_handler.close()  # Close all
            return self.close()
        if events & ioloop.READ:
            try:
                read = self.reader.read()
            except IOError:
                read = ''

            log.debug('READ>%r' % read)
            if read and len(read) != 0:
                self.ws_handler.write_message('S' + read.decode('utf-8', 'replace'))
            else:
                events = ioloop.ERROR

        if events & ioloop.ERROR:
            log.info('Error on fd %d, closing' % fd)
            # Terminated
            self.ws_handler.close()  # Close all
            self.close()

    def close(self):
        if self.closed:
            return
        self.closed = True
        if self.fd is not None:
            log.info('Closing fd %d' % self.fd)

        if getattr(self, 'pid', 0) == 0:
            log.info('pid is 0')
            return

        utils.rm_user_info(self.uid, self.pid)

        try:
            ioloop.remove_handler(self.fd)
        except Exception:
            log.error('handler removal fail', exc_info=True)

        try:
            os.close(self.fd)
        except Exception:
            log.debug('closing fd fail', exc_info=True)

        try:
            os.kill(self.pid, signal.SIGHUP)
            os.kill(self.pid, signal.SIGCONT)
            os.waitpid(self.pid, 0)
        except Exception:
            log.debug('waitpid fail', exc_info=True)
