#!/skynet/python/bin/python

import httplib
import textwrap

import requests
import simplejson.decoder

import utils


SERVICE_NAME = "coredumps"

CHECK_TIME_SEC = 10 * 60

COREDUMPS_COUNT_REQUEST = textwrap.dedent("""
    SELECT count() as cnt
    FROM sandbox.coredumpsd
    WHERE timestamp >= now() - {} and has(client_tags, 'SERVER')
    FORMAT JSON
""".format(CHECK_TIME_SEC))


SERVER_COREDUMPS_WARN_COUNT = 15
SERVER_COREDUMPS_CRIT_COUNT = 30


def main():
    data = None
    try:
        data = utils.query_clickhouse(COREDUMPS_COUNT_REQUEST)
    except (requests.RequestException, httplib.IncompleteRead) as exc:
        utils.say(SERVICE_NAME, utils.STATUS_WARN, "Data source is unreachable: {}".format(exc), die=True)
    except simplejson.decoder.JSONDecodeError:
        utils.say(SERVICE_NAME, utils.STATUS_WARN, "Clickhouse replied with: {}".format(data.text), die=True)
    if not data or not data["data"]:
        utils.say(SERVICE_NAME, utils.STATUS_WARN, "No data (empty response received from ClickHouse)", die=True)

    coredumps_count = int(data["data"][0]["cnt"])
    message = "There are {} coredumps count on servers in {} minutes".format(coredumps_count, CHECK_TIME_SEC / 60)

    if coredumps_count < SERVER_COREDUMPS_WARN_COUNT:
        utils.say(SERVICE_NAME, utils.STATUS_OK, message, die=True)
    elif coredumps_count < SERVER_COREDUMPS_CRIT_COUNT:
        utils.say(SERVICE_NAME, utils.STATUS_WARN, message, die=True)
    else:
        utils.say(SERVICE_NAME, utils.STATUS_CRIT, message, die=True)


if __name__ == "__main__":
    main()
