#!/skynet/python/bin/python

import textwrap
import calendar
import datetime as dt

import requests

import utils


SERVICE_NAME = "sandbox_mds_cleaner_delay"
TIMESTEP = 5 * 60
MAX_CLEANER_DURATION = 60 * 60 * 24 * 2

QUERY_TEMPLATE = textwrap.dedent("""
    SELECT
      max(delay) AS max_delay
    FROM sandbox.resourcesremovedfrommdsd
    WHERE
        timestamp BETWEEN toDateTime({utcnow} - {timestep}) AND toDateTime({utcnow})
    FORMAT JSON
""")


def main():
    utcnow = calendar.timegm(dt.datetime.utcnow().timetuple())
    query = QUERY_TEMPLATE.format(
        utcnow=utcnow,
        timestep=TIMESTEP,
    )

    try:
        data = utils.query_clickhouse(query)
    except requests.RequestException as exc:
        utils.say(SERVICE_NAME, utils.STATUS_WARN, "Data source is unreachable: {}".format(exc))
    else:
        if not data or not data["data"]:
            utils.say(SERVICE_NAME, utils.STATUS_WARN, "No data (empty response received from ClickHouse)", die=True)

        max_delay = int(data["data"][0]["max_delay"])
        status = utils.STATUS_CRIT if max_delay > MAX_CLEANER_DURATION else utils.STATUS_OK
        utils.say(
            SERVICE_NAME, status, "Mds cleaner delayed by more than 2 days. Current delay in hours: {}".format(
                max_delay / 60 / 60
            )
        )


if __name__ == "__main__":
    main()
