#!/skynet/python/bin/python

import collections

import pymongo
import pymongo.errors
import requests

import utils


SERVICE_NAME = "sandbox_mongocfg"
CONDUCTOR_TAG = "sandbox_mongocfg"
CONFIG_SERVER_PORT = 27019


def gethosts():
    """Get list of other mongocfg servers"""
    resp = requests.get("https://c.yandex-team.ru/api-cached/tag2hosts/{}?format=json".format(CONDUCTOR_TAG)).json()
    return resp


def get_hashes(host):
    """Obtain `config` database collections' hashes"""
    try:
        client = pymongo.MongoClient(host, CONFIG_SERVER_PORT)
        db = client["config"]
        collections_hash = db.command("dbhash")["collections"]
        for unique_collection in ("mongos", "lockpings"):
            collections_hash.pop(unique_collection)
        return ";".join((
            "{}={}".format(k, v)
            for k, v in sorted(collections_hash.items())
        ))
    except pymongo.errors.PyMongoError as exc:
        return exc


def main():
    hosts = gethosts()
    hashes = {host: get_hashes(host) for host in hosts}
    errors = " | ".join((
        "host {}: error {}".format(k, v)
        for k, v in hashes.items() if
        isinstance(v, pymongo.errors.PyMongoError)
    ))
    if errors:
        utils.say(SERVICE_NAME, utils.STATUS_CRIT, "Failed to read some replicas' hashes: {}".format(errors), die=True)

    counter = collections.Counter(hashes.values())
    if len(counter) == 1:
        utils.say(SERVICE_NAME, utils.STATUS_OK, "OK", die=True)
    elif len(counter) == len(hosts):
        utils.say(SERVICE_NAME, utils.STATUS_CRIT, "All config servers' collections have different hashes", die=True)
    else:
        majority_hash = counter.most_common(1)[0]
        for k, v in hashes.items():
            if v != majority_hash:
                utils.say(
                    SERVICE_NAME,
                    utils.STATUS_CRIT,
                    "Config server {} has config db hash that's different from others".format(k),
                    die=True
                )


if __name__ == "__main__":
    main()
