#!/skynet/python/bin/python

import textwrap
import calendar
import datetime as dt

import requests

import utils


SERVICE_NAME = "sandbox_queue_autocheck_precommits"
TIMESTEP = 10 * 60
MAX_AMOUNT = 100
QUERY_TEMPLATE = textwrap.dedent("""
    SELECT
      sum(amount) AS amount
    FROM (
      SELECT
        min(amount - wait_mutex_amount) AS amount
      FROM sandbox.queuestatisticsd
      WHERE
        timestamp BETWEEN toDateTime({utcnow} - {timestep}) AND toDateTime({utcnow}) AND
        owner = 'AUTOCHECK_TRUNK_PRECOMMIT' AND (
          task_type = 'AUTOCHECK_BUILD_YA_2' OR
          task_type = 'AUTOCHECK_BUILD_PARENT_2'
        )
      GROUP BY task_type
    )
    FORMAT JSON
""")
MESSAGE_TEMPLATE = (
    "{}/{} precommit check(s) in queue during last {} minute(s)"
)


def main():
    utcnow = calendar.timegm(dt.datetime.utcnow().timetuple())
    query = QUERY_TEMPLATE.format(
        utcnow=utcnow,
        timestep=TIMESTEP,
    )

    try:
        data = utils.query_clickhouse(query)
    except requests.RequestException as exc:
        utils.say(SERVICE_NAME, utils.STATUS_WARN, "Data source is unreachable: {}".format(exc))
    else:
        if not data or not data["data"]:
            utils.say(SERVICE_NAME, utils.STATUS_WARN, "No data (empty response received from ClickHouse)", die=True)

        amount = int(data["data"][0]["amount"])
        code = 2 if amount > MAX_AMOUNT else 0
        utils.say(SERVICE_NAME, code, MESSAGE_TEMPLATE.format(amount, MAX_AMOUNT, TIMESTEP / 60))


if __name__ == "__main__":
    main()
