#!/skynet/python/bin/python

import httplib
import textwrap
import datetime as dt

import requests
import simplejson.decoder

import utils


SERVICE_NAME = "sandbox_rejected_tasks"

CHECK_TIME_SEC = 10 * 60

REJECTED_TASKS_REQUEST = textwrap.dedent("""
    SELECT *
    FROM sandbox.taskrejectiond
    WHERE timestamp >= now() - {}
    FORMAT JSON
""".format(CHECK_TIME_SEC))

CLIENT_ALERT_AMOUNT = 30
TASKS_ALERT_AMOUNT = 250
NIGHT_ALERT_RATIO = 1.5


def check_rejected_tasks(rejections):
    rejected_clients = set()
    for rejection in rejections:
        rejected_clients.add(rejection["client_id"])

    daytime = utils.DailyAlertSpan.contains(dt.datetime.now())
    client_alert_amount = int(CLIENT_ALERT_AMOUNT * (1 if daytime else NIGHT_ALERT_RATIO))
    tasks_alert_amount = int(TASKS_ALERT_AMOUNT * (1 if daytime else NIGHT_ALERT_RATIO))

    if (
        len(rejected_clients) > client_alert_amount or
        len(rejections) > tasks_alert_amount
    ):
        return "In last {}s {} clients rejected {} tasks. See {}".format(
            CHECK_TIME_SEC,
            len(rejected_clients),
            len(rejections),
            "https://nda.ya.ru/t/mKjoo8Q14hEB9a"
        )
    return None


def main():
    data = None
    try:
        data = utils.query_clickhouse(REJECTED_TASKS_REQUEST)
    except (requests.RequestException, httplib.IncompleteRead) as exc:
        utils.say(SERVICE_NAME, utils.STATUS_WARN, "Data source is unreachable: {}".format(exc), die=True)
    except simplejson.decoder.JSONDecodeError:
        utils.say(SERVICE_NAME, utils.STATUS_WARN, "Clickhouse replied with: {}".format(data.text), die=True)
    if not data or not data["data"]:
        utils.say(SERVICE_NAME, utils.STATUS_WARN, "No data (empty response received from ClickHouse)", die=True)

    rejections = data["data"]
    error_string = check_rejected_tasks(rejections)

    if not error_string:
        utils.say(SERVICE_NAME, utils.STATUS_OK, "All good", die=True)
    utils.say(SERVICE_NAME, utils.STATUS_CRIT, error_string)


if __name__ == "__main__":
    main()
