#!/skynet/python/bin/python

import textwrap
import calendar
import datetime as dt

import requests

import utils


SERVICE_NAME = "sandbox_serviceq_activity"
TIMESTEP = 10 * 60
MIN_TASKS_GIVEN_TO_CLIENTS = 3000
QUERY_TEMPLATE = textwrap.dedent("""
    SELECT
      SUM(wants) AS wants,
      SUM(got) AS got
    FROM sandbox.serviceqtaskd
    WHERE
      timestamp BETWEEN toDateTime({utcnow} - {timestep}) AND toDateTime({utcnow})
    FORMAT JSON
""")


def main():
    utcnow = calendar.timegm(dt.datetime.utcnow().timetuple())
    query = QUERY_TEMPLATE.format(
        utcnow=utcnow,
        timestep=TIMESTEP,
    )

    try:
        data = utils.query_clickhouse(query)
    except requests.RequestException as exc:
        utils.say(SERVICE_NAME, utils.STATUS_WARN, "Data source is unreachable: {}".format(exc))
    else:
        if not data or not data["data"]:
            utils.say(SERVICE_NAME, utils.STATUS_WARN, "No data (empty response received from ClickHouse)", die=True)

        tasks_given = int(data["data"][0]["got"])
        if tasks_given < MIN_TASKS_GIVEN_TO_CLIENTS:
            utils.say(
                SERVICE_NAME,
                utils.STATUS_CRIT,
                "{} tasks given to clients during last {} minutes (threshold: {})".format(
                    tasks_given, TIMESTEP / 60, MIN_TASKS_GIVEN_TO_CLIENTS
                )
            )

        else:
            utils.say(
                SERVICE_NAME, utils.STATUS_OK, "{} tasks given to clients during last {} minutes (utcnow: {})".format(
                    tasks_given, TIMESTEP / 60, utcnow
                )
            )


if __name__ == "__main__":
    main()
