#!/skynet/python/bin/python

import textwrap
import calendar
import datetime as dt

import requests

import utils


SERVICE_NAME = "sandbox_serviceq_cpu"
TIMESTEP = 5 * 60
MAX_CPU_CONSUMPTION = 90

QUERY_TEMPLATE = textwrap.dedent("""
    SELECT
      AVG(user_cpu) AS user_cpu
    FROM sandbox.serviceqsystemresourcesd
    WHERE
      role = 'PRIMARY' AND timestamp BETWEEN toDateTime({utcnow} - {timestep}) AND toDateTime({utcnow})
    FORMAT JSON
""")


def main():
    utcnow = calendar.timegm(dt.datetime.utcnow().timetuple())
    query = QUERY_TEMPLATE.format(
        utcnow=utcnow,
        timestep=TIMESTEP,
    )

    try:
        data = utils.query_clickhouse(query)
    except requests.RequestException as exc:
        utils.say(SERVICE_NAME, utils.STATUS_WARN, "Data source is unreachable: {}".format(exc))
    else:
        if not data or not data["data"] or data["data"][0]["user_cpu"] is None:
            utils.say(SERVICE_NAME, utils.STATUS_WARN, "No data (empty response received from ClickHouse)", die=True)

        cpu_consumption = int(data["data"][0]["user_cpu"])
        status = utils.STATUS_CRIT if cpu_consumption > MAX_CPU_CONSUMPTION else utils.STATUS_OK
        utils.say(
            SERVICE_NAME, status, "primary has consumed {}% CPU during last {} minutes (threshold: {})".format(
                cpu_consumption, TIMESTEP / 60, MAX_CPU_CONSUMPTION
            )
        )


if __name__ == "__main__":
    main()
