#!/skynet/python/bin/python

import textwrap
import calendar
import datetime as dt

import requests

import utils


SERVICE_NAME = "sandbox_xmlrpc_exceptions"
TIMESTEP = 10 * 60
THRESHOLD = 30
QUERY_TEMPLATE = textwrap.dedent("""
    SELECT
      count() as errors
    FROM sandbox.taskexceptionstatisticsd
    WHERE
      timestamp BETWEEN toDateTime({utcnow} - {timestep}) AND toDateTime({utcnow}) AND
      toString(exc_type) == 'Fault'
    FORMAT JSON
""")


def main():
    utcnow = calendar.timegm(dt.datetime.utcnow().timetuple())
    query = QUERY_TEMPLATE.format(
        utcnow=utcnow,
        timestep=TIMESTEP,
    )

    try:
        data = utils.query_clickhouse(query)
    except requests.RequestException as exc:
        utils.say(SERVICE_NAME, utils.STATUS_WARN, "Data source is unreachable: {}".format(exc))
    else:
        if not data or not data["data"]:
            utils.say(SERVICE_NAME, utils.STATUS_WARN, "No data (empty response received from ClickHouse)", die=True)

        error_count = int(data["data"][0]["errors"])
        utils.say(
            SERVICE_NAME, utils.STATUS_OK if error_count < THRESHOLD else utils.STATUS_CRIT,
            "{} error(s) in {} minutes".format(error_count, TIMESTEP / 60)
        )


if __name__ == "__main__":
    main()
