import sys
import smtplib

import pymongo
import requests

from enum import Enum


REPLICA_STATES = {
    0: "STARTUP",
    1: "PRIMARY",
    2: "SECONDARY",
    3: "RECOVERING",
    5: "STARTUP2",
    6: "UNKNOWN",
    7: "ARBITER",
    8: "DOWN",
    9: "ROLLBACK",
    10: "REMOVED",
}
OK_STATES = {1, 2, 7}
WARNING_STATES = {0, 3, 5, 9}

CLICKHOUSE_BALANCER_URL = "https://clickhouse-sandbox.n.yandex-team.ru"
CLICKHOUSE_BALANCER_PREPROD_URL = "https://clickhouse-sandbox1.n.yandex-team.ru"

STATUS_OK = 0
STATUS_WARN = 1
STATUS_CRIT = 2


class DailyAlertSpan(object):
    day_start_hour = 9
    day_end_hour = 21

    @classmethod
    def contains(cls, dt):
        return cls.day_start_hour <= dt.hour < cls.day_end_hour


class Environment(Enum):
    PRODUCTION = 0
    PRE_PRODUCTION = 1


def say(service, code, msg, die=False):
    print("PASSIVE-CHECK:{};{};{}".format(service, code, msg))
    if die:
        sys.exit(0)


def replica_health(state):
    """
    Translate MongoDB replica state code to Juggler check output code with OK/WARNING/CRITICAL semantics
    """
    if state in OK_STATES:
        return 0
    if state in WARNING_STATES:
        return 1
    return 2


def check_replica_state(host, port=27017):
    """
    Return replica's state in replica set.
    Possible states: see REPLICA_STATES and https://docs.mongodb.com/v3.6/reference/replica-states/

    :param host: target host to connect to
    :param port: target port
    """

    client = pymongo.MongoClient(host, port)
    db = client.admin
    return int(db.command("replSetGetStatus")["myState"])


def send_mail(subject, message, to="sandbox-errors@yandex-team.ru"):
    body = "\r\n".join((
        "From: monitor",
        "To: {}".format(to),
        "Subject: {}".format(subject),
        "",
        message,
    ))
    s = smtplib.SMTP("localhost")
    s.sendmail("monitor", to, body)
    s.quit()


def __query_clickhouse(query, balancer_url):
    return requests.get(balancer_url, params=dict(query=query), verify=False).json()


def query_clickhouse(query, environment=Environment.PRODUCTION):
    balancer_url = CLICKHOUSE_BALANCER_URL if environment is Environment.PRODUCTION else CLICKHOUSE_BALANCER_PREPROD_URL
    return __query_clickhouse(query, balancer_url)
