# -*- coding: utf-8 -*-
import copy
import logging
import os

import jinja2
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.common.types.client as ctc

import sandbox.sandboxsdk.task as sdk_task
from sandbox.sandboxsdk import environments
import sandbox.projects.release_machine.input_params as rm_params
from sandbox.projects.release_machine import rm_notify
from parameters import ReferenceShardmap, ReferenceBinary, TestShardmap, TestBinary, StartrekTask, LaunchType, PushData

from sandbox.projects.AddrsShardedBasesearchPerformance import AddrsShardedBasesearchPerformance
from sandbox.projects.AddrsShardedBasesearchPerformance import ShardMap
from sandbox.projects.common.geosearch import task as addrs_task
from sandbox.projects.common.geosearch.startrek import StartrekClient
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.geosearch.tools import gencfg
from sandbox.projects.geosearch.tools import stat

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


@rm_notify.notify2()
class AddrsBasePerformanceAcceptance(sdk_task.SandboxTask, object):
    """Task for performance testing addrs services"""

    type = 'ADDRS_BASE_PERFORMANCE_ACCEPTANCE'
    environment = [task_env.TaskRequirements.startrek_client,
                   environments.PipEnvironment('yandex-yt')]

    basesearch_params = addrs_task.AddrsBasesearchTask.basesearch_common_parameters
    database_param = [p for p in basesearch_params if 'shard_rbtorrent' in p.name][0]
    geobasesearch_param = [p for p in basesearch_params if 'geobasesearch_executable_resource_id' in p.name][0]
    external_parameters = [p for p in AddrsShardedBasesearchPerformance.input_parameters if p not in (database_param, geobasesearch_param, ShardMap)]
    input_parameters = [
        ReferenceShardmap,
        ReferenceBinary,
        TestShardmap,
        TestBinary,
        StartrekTask,
        LaunchType,
        PushData,
        rm_params.ReleaseNum,
        rm_params.ComponentName,
    ]
    input_parameters.extend(external_parameters)
    client_tags = task_env.TaskTags.startrek_client & ctc.Tag.INTEL_E5_2650

    @property
    def footer(self):
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path), extensions=['jinja2.ext.do'])
        if 'result' in self.ctx:
            data_to_render = self.ctx['result']
        else:
            data_to_render = {}
        return env.get_template("footer.html").render(data_to_render)

    def __wait_tasks(self):
        self.wait_tasks(
            self.list_subtasks(load=False),
            self.Status.Group.SUCCEED + self.Status.Group.SCHEDULER_FAILURE,
            wait_all=True,
            state='Waiting for subtasks to complete')

    def __check_tasks(self):
        for task in self.list_subtasks(load=True):
            if not task.is_finished():
                raise SandboxTaskFailureError('Subtask %s has failed (status=%s)' % (task.descr, repr(task.status)))

    def __create_subtask(self, task, input_params):
        return self.create_subtask(
            task_type=task.type,
            description='%s subtask for #%d (%s)' % (task.type,
                                                     self.id,
                                                     self.descr),
            input_parameters=input_params)

    def get_worst_stats(self, task_id):
        stats = {}
        task = channel.sandbox.get_task(task_id)
        task_stats = task.ctx.get('result')
        stats.update({'dumper.rps': sum([subtask.get('dumper.rps') for subtask in task_stats]) / len(task_stats)})
        stats.update({'latency_0.95': min([subtask.get('latency_0.95') for subtask in task_stats])})
        stats.update({'latency_0.99': min([subtask.get('latency_0.99') for subtask in task_stats])})
        stats.update({'memory_rss': max([subtask.get('memory_rss') for subtask in task_stats])})
        stats.update({'memory_vsz': max([subtask.get('memory_vsz') for subtask in task_stats])})
        return stats

    def compare_stats(self, reference_stats, test_stats):
        result = {}
        keys = ['dumper.rps',
                'latency_0.95',
                'latency_0.99',
                'memory_rss',
                'memory_vsz']
        for key in keys:
            result.update({key: {'reference': reference_stats.get(key),
                                 'test': test_stats.get(key),
                                 'difference': float(test_stats.get(key)) - float(reference_stats.get(key)),
                                 'percent': float(test_stats.get(key)) * 100 / float(reference_stats.get(key)) - 100}})
        return result

    def make_report(self):
        data = self.ctx.get('result')
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path), extensions=['jinja2.ext.do'])
        return env.get_template("startrek_template").render(data)

    def check_mem_limit(self):
        gencfg_limit = gencfg.get_limit('addrs_base_18', 'memory_limit')
        memory_vsz = self.result['memory_rss']['test'] * 1000
        limit_perc = float(memory_vsz) / float(gencfg_limit) * 100
        logging.info('GenCfg limit = %s' % limit_perc)
        if limit_perc > 98:
            return limit_perc

    def get_client(self):
        import yt.wrapper as yt
        yt_config = {'proxy': {'url': 'hahn.yt.yandex.net'},
                     'token': self.get_vault_data('GEOMETA-SEARCH', 'yt-token')}
        return yt.YtClient(config=yt_config)

    def push_to_yt(self, data):
        if self.ctx.get(PushData.name):
            client = self.get_client()
            stat._add(data, 'perf', client)

    def on_execute(self):
        if '_REFERENCE_TASK' not in self.ctx and '_TEST_TASK' not in self.ctx:
            reference_params = copy.deepcopy(self.ctx)
            reference_params['shardmap'] = self.ctx.get(ReferenceShardmap.name)
            reference_params['geobasesearch_executable_resource_id'] = self.ctx.get(ReferenceBinary.name)
            self.ctx['_REFERENCE_TASK'] = self.__create_subtask(AddrsShardedBasesearchPerformance, reference_params).id
            test_params = copy.deepcopy(self.ctx)
            test_params['shardmap'] = self.ctx.get(TestShardmap.name)
            test_params['geobasesearch_executable_resource_id'] = self.ctx.get(TestBinary.name)
            self.ctx['_TEST_TASK'] = self.__create_subtask(AddrsShardedBasesearchPerformance, test_params).id
            self.__wait_tasks()
        else:
            self.__check_tasks()
        reference_stats = self.get_worst_stats(self.ctx.get('_REFERENCE_TASK'))
        logging.info('Performance statistics for reference task: %s' % reference_stats)
        test_stats = self.get_worst_stats(self.ctx.get('_TEST_TASK'))
        self.push_to_yt(test_stats)
        logging.info('Performance statistics for test task: %s' % test_stats)
        self.result = self.compare_stats(reference_stats, test_stats)
        self.ctx['result'] = {'result': self.result}
        logging.info('Result = %s' % self.result)
        startrek_token = self.get_vault_data('robot-geosearch',
                                             'robot_geosearch_startrek_token')
        startrek_ticket = self.ctx.get(StartrekTask.name)
        stat_table = self.make_report()
        mem_limit_prc = self.check_mem_limit()
        if self.ctx.get(LaunchType.name) == 'RM':
            startrek_helper = STHelper(startrek_token)
            c_info = rmc.COMPONENTS[self.ctx[rm_params.ComponentName.name]]()
            relese_num = self.ctx.get(rm_params.ReleaseNum.name)
            if mem_limit_prc:
                memory_msg = (u'!!(red)Внимание! В тестах производительности '
                              u'использовано %d %% памяти от лимита в '
                              u'GenCfg!!') % mem_limit_prc
                startrek_helper.comment(relese_num, memory_msg, c_info)
            startrek_helper.write_grouped_comment('====Performance test',
                                                  'Launch performance test',
                                                  stat_table,
                                                  relese_num,
                                                  c_info)
        elif self.ctx.get(LaunchType.name) == 'DB':
            self.startrek = StartrekClient(startrek_token)
            if mem_limit_prc:
                memory_msg = (u'!!(red)Внимание! В тестах производительности '
                              u'использовано %d %% памяти от лимита в '
                              u'GenCfg!!') % mem_limit_prc
                self.startrek.add_comment(startrek_ticket, memory_msg)
            self.startrek.add_comment(startrek_ticket, '<{%s}>' % stat_table)
