# -*- coding: utf-8 -*-
import logging
from sandbox.sandboxsdk import task as sandbox_task

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from parameters import (
    Action,
    ActivateAfterChanging,
    ForceNewSnapshot,
    Stand,
    Service,
    SandboxResource,
    DestinationService,
    Topology,
    TopologyAction,
)

from sandbox.projects import resource_types
from sandbox.projects.geosearch.tools.nanny import Nanny
from sandbox.projects.geosearch.ReleaseAddrsShardmap import ReleaseAddrsShardmap as shardmap_release_task
from sandbox.projects.release_machine import rm_notify


@rm_notify.notify2()
class AddrsDeploy(sandbox_task.SandboxTask):
    '''Deploy addrs services'''

    type = 'ADDRS_DEPLOY'

    cores = 1

    input_parameters = [Action,
                        ActivateAfterChanging,
                        ForceNewSnapshot,
                        Stand,
                        Service,
                        DestinationService,
                        SandboxResource,
                        TopologyAction,
                        Topology]

    def get_input_parameters(self):
        self.act = self.ctx.get(Action.name)
        if self.act == 'make_production':
            self.stand = self.ctx.get(Stand.name)
            self.activate = self.ctx.get(ActivateAfterChanging.name)
            self.force_snapshot = self.ctx.get(ForceNewSnapshot.name)
        elif self.act == 'copy_service':
            self.src_service = self.ctx.get(Service.name)
            self.dst_service = self.ctx.get(DestinationService.name)
            self.sandbox_resource = ''
            self.activate = self.ctx.get(ActivateAfterChanging.name)
            self.force_snapshot = self.ctx.get(ForceNewSnapshot.name)
        elif self.act == 'sandbox_resource':
            self.src_service = ''
            self.dst_service = self.ctx.get(DestinationService.name)
            self.sandbox_resource = [str(resource) for resource in self.ctx.get(SandboxResource.name)]
            self.activate = self.ctx.get(ActivateAfterChanging.name)
        elif self.act == 'change_topology':
            self.dst_service = self.ctx.get(DestinationService.name)
            self.topology = self.ctx.get(Topology.name)
            self.topology_action = self.ctx.get(TopologyAction.name)
            logging.info(self.topology)
            logging.info(self.topology_action)

    def __wait_tasks(self):
        self.wait_tasks(
            self.list_subtasks(load=False),
            self.Status.Group.SUCCEED + self.Status.Group.SCHEDULER_FAILURE,
            wait_all=True,
            state='Waiting for subtasks to complete')

    def __check_tasks(self):
        for task in self.list_subtasks(load=True):
            if not task.is_finished():
                raise SandboxTaskFailureError('Subtask %s has failed (status=%s)' % (task.descr, repr(task.status)))

    def __create_subtask(self, task, input_params, execution_space=None):
        return self.create_subtask(
            task_type=task.type,
            description='%s subtask for #%d (%s)' % (task.type,
                                                     self.id,
                                                     self.descr),
            input_parameters=input_params,
            execution_space=execution_space
        )

    def on_execute(self):
        token = self.get_vault_data('robot-geosearch', 'ADDRS')
        nanny = Nanny(token)
        self.get_input_parameters()
        if self.act == 'make_production':
            self.all_services = nanny.expand_stand(self.stand)
            logging.info('Applying changes to %s' % self.stand)
            changing_list = nanny.compare_stands('/rcss/addrs/prod/',
                                                 self.stand)
            basesearches = [item for item in changing_list if 'base' in item[1]]
            logging.info('Services to be copied from prod: %s' % changing_list)
            nanny.loadstand_to_prod(changing_list,
                                    self.stand,
                                    self.author,
                                    self.activate,
                                    self.force_snapshot)
            for basesearch in basesearches:
                nanny_data = nanny.get_sandbox_files(basesearch[0])
                shardmap = nanny_data['sandbox_bsc_shard']
                shardmap_task = channel.sandbox.get_task(shardmap['sandbox_shardmap']['task_id'])
                shardmap_id = channel.sandbox.list_resources(task_id=shardmap_task,
                                                             resource_type=resource_types.ADDRS_BUSINESS_SHARDMAP)[0].id
                logging.info('shardmap_id = %s' % shardmap_id)
                nanny.copy_service(basesearch[0],
                                   basesearch[1],
                                   self.author,
                                   self.force_snapshot)
                self.__check_tasks()
                with self.memoize_stage.release_shardmap:
                    self.__create_subtask(shardmap_release_task,
                                          {'service': basesearch[1],
                                           'shardmap': shardmap_id})
                    self.__wait_tasks()
        elif self.act == 'copy_service':
            logging.info('Changing %s configuration' % self.dst_service)
            nanny.copy_service(self.src_service,
                               self.dst_service,
                               self.author,
                               self.force_snapshot)
            comment = 'Copied runtime attrs from %s by %s' % (self.src_service, self.author)
            if self.activate:
                nanny.activate_service(self.dst_service, comment)
        elif self.act == 'sandbox_resource':
            logging.info('self.sandbox_resource = %s' % self.sandbox_resource)
            nanny.replace_sandbox_file(self.sandbox_resource, self.dst_service, self.author)
            comment = ('Added #%s resource (%s)' %
                       (','.join(self.sandbox_resource), self.author))
            if self.activate:
                nanny.activate_service(self.dst_service, comment)
        elif self.act == 'change_topology':
            instances = nanny.get_instances(self.dst_service)
            if self.topology_action == 'add_topology':
                logging.info('Adding %s topology to %s service' %
                             (self.topology, self.dst_service))
                new_instances = nanny.add_topology(instances, self.topology)
                nanny.put_instances(self.dst_service, new_instances)
                comment = ('Added %s  topology (%s)' %
                           (self.topology, self.author))
                nanny.activate_service(self.dst_service, comment)
            elif self.topology_action == 'delete_topology':
                logging.info('Deleting %s topology to %s service' %
                             (self.topology, self.dst_service))
                new_instances = nanny.delete_topology(instances, self.topology)
                nanny.put_instances(self.dst_service, new_instances)
                comment = ('Deleted %s topology (%s)' %
                           (self.topology, self.author))
                nanny.activate_service(self.dst_service, comment)
        logging.info('Action %s completed' % self.act)
