# -*- coding: utf-8 -*-
import logging
import gzip
import requests
import os
import time
from functools import wraps


def retry(tries=1, delay=1):
    def deco_retry(func):
        @wraps(func)
        def func_retry(*args, **kwargs):
            my_tries, my_delay = tries, delay
            while my_tries > -1:
                try:
                    return func(*args, **kwargs)
                except Exception as err:
                    msg = '%s - %s. retrying in %d seconds' % (func.func_name,
                                                               str(err),
                                                               my_delay)
                    my_tries -= 1
                    if my_tries == -1:
                        return False
                    logging.debug(msg)
                    time.sleep(my_delay)
        return func_retry
    return deco_retry


@retry(tries=3, delay=2)
def get_gencfg_groups(service):
    url = 'https://nanny.yandex-team.ru/v2/services/%s/runtime_attrs/instances/' % service
    data = requests.get(url).json()
    groups = data['content']['extended_gencfg_groups']['groups']
    return [group.get('name') for group in groups]


@retry(tries=3, delay=2)
def expand_gencfg_group(group):
    url = 'https://api.gencfg.yandex-team.ru/trunk/searcherlookup/groups/%s/instances' % group
    data = requests.get(url).json()
    data = data.get('instances')[0]
    host = data.get('hostname')
    port = data.get('port')
    return host, port


def get_dst_service(src_service, stand):
    if src_service:
        parts = src_service.split('_')
        load_num = stand[-2]
        return '%s_%s_l%s' % (parts[0], parts[1], load_num)


def gunzip_file(path):
    with gzip.open(path, 'rb') as src_file:
        content = src_file.read()
        file_name = path.split('/')[-1]
        unzipped_file_path = './%s' % file_name.strip('.gzip')
        dst_file = open(unzipped_file_path, 'w')
        dst_file.write(content)
        dst_file.close()
        return unzipped_file_path


def extract_tags(ammo_file_path):
    '''Returns tags from tank ammo file'''
    with open(ammo_file_path, 'r') as f:
        ammos = f.readlines()
    return {ammo.split()[1] for ammo in ammos}


def get_tank_tasks(path):
    '''Returns created tank tasks'''
    tasks = []
    log_dir = '%s/logs' % path
    for item in os.listdir(log_dir):
        if os.path.isdir('%s/%s' % (log_dir, item)):
            tasks.append(item)
    return tasks


def get_tank_urls(tasks):
    all_tasks = set()
    all_tasks.update(tasks.get('reference'))
    if tasks.get('test'):
        all_tasks.update(tasks.get('test'))
    comparison = ('https://lunapark.yandex-team.ru/compare/#jobs=%s'
                  '&tab=test_data'
                  '&mainjob=%s'
                  '&helper=all'
                  '&cases='
                  '&plotGroup=additional'
                  '&metricGroup='
                  '&target=') % (','.join(all_tasks), list(all_tasks)[0])
    short_comparison = comparison.split('&')[0]
    reference_shootings = ['https://lunapark.yandex-team.ru/%s' % task for task in tasks.get('reference')]
    test_shootings = ['https://lunapark.yandex-team.ru/%s' % task for task in tasks.get('test')]
    reference_url = ['<p><a href=%s>%s</a></p>' % (shooting, shooting) for shooting in reference_shootings]
    test_url = ['<p><a href=%s>%s</a></p>' % (shooting, shooting) for shooting in test_shootings]

    return {'<h3>Shootings</h3>': {'<h4>Reference</h4>': '\n'.join(reference_url),
                                   '<h4>Test</h4>': '\n'.join(test_url),
                                   '<h4>Comparison</h4>': '<p><a href=%s>%s</a></p>' % (comparison, short_comparison)}}


def clear_cache(services):
    cleaner = 'addrs_cleaner_l%s' % services[0][-1]
    cleaner_gencfg_group = get_gencfg_groups(cleaner)[0]
    cleaner_host, cleaner_port = expand_gencfg_group(cleaner_gencfg_group)

    for service in services:
        gencfg_group = get_gencfg_groups(service)[0]
        host, port = expand_gencfg_group(gencfg_group)
        if 'addrs_upper' in service:
            pass
        elif 'addrs_wizard' in service:
            action = 'resetcache'
            service = 'wizard'
        else:
            action = 'clearcache'
            service = 'admin'
        url = ('http://{cleaner_host}:{cleaner_port}/package/cgi/'
               'cache_cleaner.py?'
               'host={host}'
               '&port={port}'
               '&service={service}'
               '&action={action}').format(cleaner_host=cleaner_host,
                                          cleaner_port=cleaner_port,
                                          host=host,
                                          port=port,
                                          service=service,
                                          action=action)
        data = requests.get(url)
        logging.info('%s response code: %s' % (url, data.status_code))
