# -*- coding: utf-8 -*-
from datetime import datetime, timedelta, date
import json
import logging
import requests
import time

from sandbox.projects.Afisha.PushSender import queries
from sandbox.projects.common.yql import run_query, run_batch

import sandbox.sdk2 as sdk2
from sandbox.sandboxsdk.environments import PipEnvironment


class AfishaPushSender(sdk2.Task):
    """
        Расчет и рассылка пушей посещений театров афишы
    """
    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("Входные параметры"):
            export_dir = sdk2.parameters.String('директория посещений', default_value='//home/user_identification/orgvisits/prod/state/export')
            schedule_log = sdk2.parameters.String('Расписание афишы', default_value='//home/afisha/export/production/schedules')

        with sdk2.parameters.Group("Внутренние параметры"):
            tables_limit = sdk2.parameters.Integer("Количество хранимых в каждой директории таблиц", default_value=90)
            visit_dir = sdk2.parameters.String('директория посещений', default_value='//home/geosearch/gav1995/afisha/visits')
            schedule_dir = sdk2.parameters.String('Кэшированная таблица расписания афишы', default_value='//home/geosearch/gav1995/afisha/schedules')
            schedule_limit = sdk2.parameters.Integer("Количество хранимых таблиц расписаний", default_value=7)
            push_dir = sdk2.parameters.String('директория посещений', default_value='//home/geosearch/gav1995/afisha/pushes')

        flag_from_today = sdk2.parameters.Bool('таблица задается числом дней от сегодня?', default_value=True)

        with flag_from_today.value[True]:
            table_from_d = sdk2.parameters.Integer('Дата, с которой считать в формате дней от сегодня', default_value=-1)

        with flag_from_today.value[False]:
            table_from_s = sdk2.parameters.String('Дата, с которой считать в строковом формате ISO (YYYY-MM-DD)', default_value=(date.today()-timedelta(1)).isoformat())

        push_url = sdk2.parameters.String('push url ', default_value='https://sup.yandex.net/pushes/batch')
        with sdk2.parameters.Group("Токены"):
            yt_token_owner = sdk2.parameters.String('владелец токена YT в vault', default_value='gav1995')
            yt_token_name = sdk2.parameters.String('название токена YT в vault', default_value='YT_token')
            yql_token_owner = sdk2.parameters.String('владелец токена YQL в vault', default_value='gav1995')
            yql_token_name = sdk2.parameters.String('название токена YQL в vault', default_value='YQL_token')
            push_token_owner = sdk2.parameters.String('владелец токена push в vault', default_value='gav1995')
            push_token_name = sdk2.parameters.String('название токена push в vault', default_value='push-token')

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment('yandex-yt'),
            PipEnvironment('yql')
            ]

    def on_execute(self):
        top_visit = self.clear_directory(self.Parameters.visit_dir, self.Parameters.tables_limit)
        top_push = self.clear_directory(self.Parameters.push_dir, self.Parameters.tables_limit)
        self.clear_directory(self.Parameters.schedule_dir, self.Parameters.schedule_limit)
        top = max(top_push, top_visit)
        to_count = self.get_tables_to_count(top)

        import yt.wrapper as yt
        yt.config["proxy"]["url"] = "hahn"
        yt.config["token"] = sdk2.Vault.data(self.Parameters.yt_token_owner, self.Parameters.yt_token_name)

        tables = json.loads(yt.list(self.Parameters.schedule_dir, format='json'))
        tables.sort()
        schedule_table = tables[0] if len(tables) == 1 else tables[-2]

        batch = []
        for table in to_count:
            export_table = '{}/{}'.format(self.Parameters.export_dir, table)
            while not yt.exists(export_table):
                logging.info('wait {}'.format(export_table))
                time.sleep(60)
            batch.append(('{}/{}'.format(self.Parameters.visit_dir, table), queries.create_pushes.format(
                export_dir=self.Parameters.export_dir,
                schedule_dir=self.Parameters.schedule_dir,
                schedule_table=schedule_table,
                push_dir=self.Parameters.push_dir,
                visit_dir=self.Parameters.visit_dir,
                table=table
            )))
        logging.info('batch size = {}'.format(len(batch)))
        run_batch(batch, sdk2.Vault.data(self.Parameters.yql_token_owner, self.Parameters.yql_token_name), 'hahn')
        run_query(
            '{dir}/{table}'.format(dir=self.Parameters.schedule_dir, table=date.today().isoformat()),
            queries.save_schedule.format(
                table=date.today().isoformat(),
                schedule_dir=self.Parameters.schedule_dir,
                schedule_log=self.Parameters.schedule_log
            ),
            sdk2.Vault.data(self.Parameters.yql_token_owner, self.Parameters.yql_token_name),
            'hahn'
        )
        HEADERS = {}
        HEADERS['Content-Type'] = 'application/json;charset=UTF-8'
        HEADERS['Authorization'] = 'OAuth {}'.format(sdk2.Vault.data(self.Parameters.push_token_owner, self.Parameters.push_token_name))
        for table in to_count:
            logging.info('sending push for table {}'.format(table))
            data = {'path': '{push_dir}/{table}'.format(push_dir=self.Parameters.push_dir, table=table)}
            logging.info('url = {}'.format(self.Parameters.push_url))
            logging.info('HEADERS = {}'.format(HEADERS))
            logging.info('data = {}'.format(data))
            resp = requests.post(self.Parameters.push_url, headers=HEADERS, data=json.dumps(data), timeout=2)
            limit = 5
            while resp.status_code != requests.codes.ok and limit:
                resp = requests.post(self.Parameters.push_url, headers=HEADERS, data=json.dumps(data), timeout=2)
                logging.info('response = {}'.format(resp.text))
                limit -= 1
            if resp.status_code == requests.codes.ok:
                logging.info('sending successful finished')
            else:
                resp.raise_for_status()
        logging.info('task finished successful!!!')

    def clear_directory(self, yt_dir, limit=90):
        # оставляет в директории все таблицы за даты не старше today - limit, возвращает имя младщей таблицы в директории до очистки

        logging.info('cleaning directory {}'.format(yt_dir))

        import yt.wrapper as yt
        yt.config["proxy"]["url"] = "hahn"
        yt.config["token"] = sdk2.Vault.data(self.Parameters.yt_token_owner, self.Parameters.yt_token_name)

        tables = json.loads(yt.list(yt_dir, format='json'))
        logging.info("yt_dir {dir} consists of :{list}".format(dir=yt_dir, list=tables))

        tables.sort()

        logging.info("yt_dir {dir} consists of :{list}".format(dir=yt_dir, list=tables))

        top = tables[-1] if tables else '0001-01-01'
        to_del = [table for table in tables if table < (date.today() - timedelta(limit)).isoformat()]
        logging.info("removing tables :{list}".format(list=to_del))

        for table in to_del:
            path = "{}/{}".format(yt_dir, table)
            logging.info('removing ' + path)
            yt.remove(path)
        logging.info('cleaning directory {} is finished'.format(yt_dir))
        return top

    def get_tables_to_count(self, top_table):
        # определяет таблицы, которые надо посчитать (все из диапазона [from_table, -1] старше чем top_table)
        from_date = date.today() + timedelta(self.Parameters.table_from_d) if self.Parameters.flag_from_today else datetime.strptime(self.Parameters.table_from_s, '%Y-%m-%d').date()
        logging.info('from_date = {}, top_table = {}'.format(from_date, top_table))
        from_date = max(from_date, datetime.strptime(top_table, '%Y-%m-%d').date() + timedelta(1))
        count_days = (date.today() - from_date).days
        logging.info('count_days = {}'.format(count_days))
        to_count = [(from_date + timedelta(day)).isoformat() for day in range(count_days)]
        logging.info('to count :{}'.format(to_count))
        return to_count
