""" Sandbox helpers and constants """
import logging
import pytz
from datetime import datetime
import sandbox.sdk2 as sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.sdk2.service_resources import SandboxTasksBinary
from sandbox.projects.common import binary_task
from sandbox.projects.release_machine.helpers import arcanum_helper
from sandbox.projects.sandbox_ci.utils.github import GitHubApi


ARC_SVN_URL = "arcadia:/arc/trunk/arcadia"
DEFAULT_SUBTASK_WAIT_STATUS = ctt.Status.Group.FINISH | ctt.Status.Group.BREAK


class AfishaArcanumApi(arcanum_helper.ArcanumApi):

    def get_review_request(self, rr_id, **kwargs):
        return self._do_get("v1/review-requests/{}".format(rr_id), params=kwargs)

    def is_pr_closed(self, vcs_path, rr_id):
        _ = vcs_path  # For other clients
        params = {"fields": "id,state"}
        pr = self.get_review_request(rr_id, **params)
        if pr["data"]["state"] == "closed":
            return True
        return False

    def get_pr_summary(self, vcs_path, rr_id):
        _ = vcs_path  # For other clients
        params = {"fields": "id,summary"}
        pr = self.get_review_request(rr_id, **params)
        return pr["data"]["summary"]

    def get_pr_issues(self, vcs_path, rr_id):
        _ = vcs_path  # For other clients
        params = {"fields": "id,issues"}
        pr = self.get_review_request(rr_id, **params)
        return pr["data"]["issues"]

    def last_pr_update_days(self, vcs_path, rr_id):
        _ = vcs_path
        params = {"fields": "id,updated_at"}
        pr = self.get_review_request(rr_id, **params)
        pr_update_date = datetime.strptime(pr["data"]["updated_at"], '%Y-%m-%dT%H:%M:%S.%fZ').replace(tzinfo=pytz.UTC)
        pr_duration = datetime.now(pytz.UTC) - pr_update_date
        return pr_duration.days

    def comment(self, vcs_path, rr_id, message):
        _ = vcs_path
        self.comment_review(rr_id, message)


class AfishaGitHubApi(GitHubApi):

    @staticmethod
    def owner_repo_from_vcs_path(vcs_path):
        return vcs_path.split("/")

    def is_pr_closed(self, vcs_path, pr_number):
        owner, repo = self.owner_repo_from_vcs_path(vcs_path)
        pr_info = self.get_pr_info(owner, repo, pr_number)
        if pr_info["state"] == "closed":
            return True
        return False

    def last_pr_update_days(self, vcs_path, pr_number):
        owner, repo = self.owner_repo_from_vcs_path(vcs_path)
        pr = self.get_pr_info(owner, repo, pr_number)
        pr_update_date = datetime.strptime(pr["updated_at"], '%Y-%m-%dT%H:%M:%SZ').replace(tzinfo=pytz.UTC)
        pr_duration = datetime.now(pytz.UTC) - pr_update_date
        return pr_duration.days

    def comment(self, vcs_path, pr_number, message):
        owner, repo = self.owner_repo_from_vcs_path(vcs_path)
        path = "repos/{owner}/{repo}/issues/{pr_number}/comments".format(
            owner=owner, repo=repo, pr_number=pr_number)
        data = {"body": message}
        res = self._GitHubApi__http_send_request_post(path, data,
                                                      make_request=self._GitHubApi__post_request_with_retries)
        assert res.status_code == 201, {'status_code': res.status_code, 'text': res.text}
        return res


class AfishaSandboxMethods(sdk2.Task):

    def _init_staff(self, user):
        """
        provides self.staff object with staff api client
        :return: None
        """
        from afisha.infra.libs.staff import StaffApiClient
        vault = "{}.st-token".format(user)
        token = sdk2.Vault.data(user, vault).rstrip()
        self.staff = StaffApiClient(token=token)

    def _init_yav_staff(self, yav, key="staff"):
        from afisha.infra.libs.staff import StaffApiClient
        try:
            token = yav.data()[key]
        except KeyError as error:
            raise KeyError("Couldn't find staff token by key %s in secret. Details: %s" % (key, error))
        self.staff = StaffApiClient(token=token)

    def _init_st(self, user, useragent="AfishaSandbox"):
        """
            Provides self.st object with startrek client
        """
        import startrek_client
        vault = "{}.st-token".format(user)
        token = sdk2.Vault.data(user, vault).rstrip()
        self.st = startrek_client.Startrek(token=token, useragent=useragent)

    def _init_yav_st(self, yav, key="startrek", useragent="AfishaSandbox"):
        """
            Provides self.st object with startrek client using yav secret
        """
        import startrek_client
        try:
            token = yav.data()[key]
        except KeyError as error:
            raise KeyError("Couldn't find startrek token by key %s in secret. Details: %s" % (key, error))
        self.st = startrek_client.Startrek(token=token, useragent=useragent)

    def _init_qc(self, user):
        """
        Provides self.qc object with qloud client
        :param user: owner of secret
        :return: None
        """
        from afisha.infra.libs.platform import PlatformClient
        vault = "{}.qloud-token".format(user)
        token = sdk2.Vault.data(user, vault).rstrip()
        self.qc = PlatformClient(token=token)

    def _init_dc(self, user):
        """
        Provides self.dc object with deployer client
        :param user: owner of secret
        :return: None
        """
        from afisha.infra.libs.deployer import DeployerClient
        vault = "{}.qloud-token".format(user)
        token = sdk2.Vault.data(user, vault).rstrip()
        self.dc = DeployerClient(token=token)

    def _init_tc(self, user):
        from teamcity_client import TeamcityClient
        vault = "{}.teamcity-token".format(user)
        token = sdk2.Vault.data(self.user, vault).rstrip()
        self.tc = TeamcityClient(server_url="teamcity.yandex-team.ru", auth=token)

    def _init_nyan(self):
        """
        Provides self.nyan object with nyanbot client
        :return: None
        """
        from afisha.infra.libs.nyan import NyanClient
        self.nyan = NyanClient()

    def _init_u(self):
        """ Provides self.u object with U client """
        from afisha.infra.libs.u import UApiClient
        self.u = UApiClient()

    def _init_abc(self, user):
        """ Provides self.abc object with ABC client """
        from afisha.infra.libs.abc import AbcApiClient
        vault = "{}.abc-token".format(user)
        token = sdk2.Vault.data(user, vault).rstrip()
        self.abc = AbcApiClient(token=token)

    def _init_yav_abc(self, yav, key="abc"):
        from afisha.infra.libs.abc import AbcApiClient
        try:
            token = yav.data()[key]
        except KeyError as error:
            raise KeyError("Couldn't find abc token by key %s in secret. Details: %s" % (key, error))
        self.abc = AbcApiClient(token=token)

    def _init_infra(self, user):
        """ Provides self.infra object with Infra client """
        from afisha.infra.libs.infra import InfraApiClient
        vault = "{}.infra-token".format(user)
        token = sdk2.Vault.data(user, vault).rstrip()
        self.infra = InfraApiClient(token=token)

    def _init_rc(self):
        """ Provides self.rc object with ReleaseApi client """
        from afisha.infra.libs.releases import ReleasesApiClient
        self.rc = ReleasesApiClient()

    def _init_ac(self, user):
        """ Provides self.ac object with ArcanumApi client """
        vault = "{}.arc-token".format(user)
        token = sdk2.Vault.data(user, vault).rstrip()
        self.ac = AfishaArcanumApi(token=token)

    def _init_bbc(self, user):
        """ Provides self.bbc object with BitBucket api client """
        from afisha.infra.libs.bb import BitBucketClient
        bb_host = "https://bb.yandex-team.ru/"
        vault = "{}.bb-token".format(user)
        token = sdk2.Vault.data(user, vault).rstrip()
        self.bbc = BitBucketClient(bb_host, token=token)

    def _init_ghc(self, user):
        """ Provides self.ghc object with GitHub api client """
        vault = "{}.gh-token".format(user)
        token = sdk2.Vault.data(user, vault).rstrip()
        self.ghc = AfishaGitHubApi(token=token)

    @staticmethod
    def find_task(tasks_ids, key, value):
        logging.debug("Searching %s parameter key with %s value in % task ids", key, value, tasks_ids)
        for task_id in tasks_ids:
            task_ = sdk2.Task[task_id]
            if getattr(task_.Parameters, key, None) == value:
                logging.debug("Found in %s", task_id)
                return task_
            logging.debug("Not found in %s", task_id)
        logging.debug("Not found in %s tasks", tasks_ids)
        return None

    @property
    def subtasks(self):
        if self.Context.subtasks is ctm.NotExists:
            return {}
        return self.Context.subtasks

    def subtasks_run(self, tasks, statuses=DEFAULT_SUBTASK_WAIT_STATUS, wait_all=True, timeout=None):
        from afisha.infra.libs.base.funcs import to_sandbox_name
        if not self.subtasks:
            self.Context.subtasks = {}
        enqueued = []
        for task in tasks:
            task.enqueue()
            task_name = to_sandbox_name(type(task).__name__)
            if task_name not in self.subtasks:
                self.subtasks[task_name] = []
            if task.id not in self.subtasks[task_name]:
                self.subtasks[task_name].append(task.id)
            enqueued.append(task.id)
        if statuses:
            raise sdk2.WaitTask(enqueued, statuses, wait_all=wait_all, timeout=timeout)


class AfishaSandboxBaseTask(AfishaSandboxMethods):
    """ Base Afisha sandbox task """

    BINARY_TASK_ATTR_TARGET = None

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Task resource settings", collapse=True) as build_settings_block:
            UseLastBinary = sdk2.parameters.Bool("Use last binary archive", default=True)
            with UseLastBinary.value[True]:
                with sdk2.parameters.RadioGroup("Binary release type") as ReleaseType:
                    ReleaseType.values.stable = ReleaseType.Value("stable", default=True)
                    ReleaseType.values.test = ReleaseType.Value("test")
            with UseLastBinary.value[False]:
                custom_tasks_archive_resource = sdk2.parameters.Resource("task archive resource", default=None)

    def on_save(self):
        if self.BINARY_TASK_ATTR_TARGET is None:
            raise RuntimeError("AfishaSandboxBaseTask must define BINARY_TASK_ATTR_TARGET attribute")

        if self.Parameters.UseLastBinary:
            self.Requirements.tasks_resource = SandboxTasksBinary.find(
                attrs={"target": self.BINARY_TASK_ATTR_TARGET,
                       "release": self.Parameters.ReleaseType or "stable"}
            ).first().id
        else:
            self.Requirements.tasks_resource = self.Parameters.custom_tasks_archive_resource

    def task(self, task_type, **kwargs):
        kwargs["UseLastBinary"] = kwargs.get("UseLastBinary", self.Parameters.UseLastBinary)
        if kwargs["UseLastBinary"]:
            kwargs["ReleaseType"] = kwargs.get("ReleaseType", self.Parameters.ReleaseType)
        task = sdk2.Task[task_type](self, **kwargs)
        return task


class AfishaSandboxBaseTaskBinary(binary_task.LastBinaryTaskRelease, AfishaSandboxMethods):
    """ Base Afisha sandbox binary task """

    class Parameters(sdk2.Task.Parameters):
        binary_release = binary_task.binary_release_parameters(stable=True)

    def task(self, task_type, **kwargs):
        kwargs["binary_executor_release_type"] = kwargs.get("binary_executor_release_type", self.Parameters.binary_executor_release_type)
        task = sdk2.Task[task_type](self, **kwargs)
        return task
