# coding: utf-8
import logging
import sandbox.sdk2 as sdk2

from requests.exceptions import HTTPError, ConnectionError, Timeout
from sandbox import common
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask


class AfishaBalancerDeploy(AfishaSandboxBaseTask):

    BINARY_TASK_ATTR_TARGET = "Afisha/deploy/common/AfishaBalancerDeploy"

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(AfishaSandboxBaseTask.Parameters):
        kill_timeout = 120
        with sdk2.parameters.Group("Settings") as settings_block:
            qloud_environments = sdk2.parameters.String(
                "Path to qloud environments separated by ; "
                "(e.g.: afisha-tickets.worker.testing;afisha-tickets.workercritical.testing)",
                required=True)
            registry_url = sdk2.parameters.String(
                "Docker image url in registry",
                required=True)
            comment = sdk2.parameters.String(
                "Deploy comment",
                required=False)
            with sdk2.parameters.RadioGroup("Target state") as tgt_state:
                tgt_state.values["COMMITTED"] = tgt_state.Value("Committed")
                tgt_state.values["PREPARED"] = tgt_state.Value("Prepared")
                tgt_state.values["DEPLOYED"] = tgt_state.Value("Deployed",
                                                               default=True)
            deploy_with_robot = sdk2.parameters.Bool("Deploy with robot",
                                                     default=True)
            with deploy_with_robot.value[True]:
                deployer_username = sdk2.parameters.String("Robot username")

    @property
    def user(self):
        if not self.Parameters.deploy_with_robot:
            return self.author
        return self.Parameters.deployer_username

    @staticmethod
    def split_registry_tag(registry_tag_full):
        if ":" in registry_tag_full:
            registry_url, registry_tag = registry_tag_full.split(":")
        else:
            registry_url, registry_tag = registry_tag_full, "latest"
        if registry_url.startswith("registry.yandex.net/"):
            registry_url = registry_url[len("registry.yandex.net/"):]

        return registry_url, registry_tag

    def _run_deploy(self, environment, image_url):
        registry_url, registry_tag = self.split_registry_tag(image_url)
        comment = self.Parameters.comment if self.Parameters.comment else \
            "Deploying {} from sandbox".format(registry_tag)

        logging.info("Updating component with path %s, registry %s and tag %s", environment, registry_url, registry_tag)
        try:
            result = self.dc.deploy(environment, registry_tag, state=self.Parameters.tgt_state, comment=comment,
                                    balancer_only=True).split("\n")[0]
        except (HTTPError, ConnectionError, Timeout) as error:
            raise common.errors.TemporaryError("Not 2xx answer from deployer (%s), will try again later" % error)

        return result

    def on_execute(self):
        # 2 hours (1 retry == 60 sec)
        MAX_RETRIES = 120
        self._init_dc(self.user)
        result = []
        with self.memoize_stage.set_retry:
            self.Context.retry = 0
        for environment in self.Parameters.qloud_environments.split(";"):
            result.append(self._run_deploy(environment, self.Parameters.registry_url))
        if result.count("True") != len(self.Parameters.qloud_environments.split(";")):
            self.Context.retry += 1
            if self.Context.retry >= MAX_RETRIES:
                raise common.errors.TaskError("Retries exceeded (%s >= %s), balancer still not deployed" %
                                              (self.Context.retry, MAX_RETRIES))
            raise sdk2.WaitTime(60)
