# coding: utf-8
import logging
import sandbox.sdk2 as sdk2
from sandbox.common import errors
import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask, ARC_SVN_URL
from sandbox.projects.Afisha.resource_types import AfishaTicketsBackendBinary
from six.moves.urllib import parse as urlparse


class AfishaBuildDocker(AfishaSandboxBaseTask):

    BINARY_TASK_ATTR_TARGET = "Afisha/deploy/common/AfishaBuildDocker"
    DOCKER_REGISTRY = "registry.yandex.net"
    BUILD_TASK_NAME_OLD = "YA_PACKAGE"
    BUILD_TASK_NAME = "YA_PACKAGE_2"
    ARCADIA_ARC_SCHEME = "arcadia-arc"

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(AfishaSandboxBaseTask.Parameters):
        kill_timeout = 120
        with sdk2.parameters.Group("Settings") as settings_block:
            package_path = sdk2.parameters.String("Package path (relative to arc root)", required=True)
            arc_svn_url = sdk2.parameters.String("Svn url for arc with revision", default=ARC_SVN_URL, required=True)
            checkout_arcadia_from_url = sdk2.parameters.String("Svn url for arc with revision", default=ARC_SVN_URL, required=True)
            arcadia_patch = sdk2.parameters.String("Arcadia patch", default=None)
            pr = sdk2.parameters.Bool("PR build", default=False)
            component_name = sdk2.parameters.String("Component name", required=True)
            release = sdk2.parameters.Bool("Release build", default=False)
            use_new_ya_package = sdk2.parameters.Bool("Use YA_PACKAGE_2 task", default=True)
            with pr.value[True]:
                arcanum_review_id = sdk2.parameters.String("Review id", required=True)
            with release.value[True]:
                number = sdk2.parameters.String("Release number", required=True)

        with sdk2.parameters.Group("Specific settings", collapse=True) as specific_settings_block:
            docker_build_args = sdk2.parameters.Dict("Args for docker build in ya_package_2", required=False)
            ya_package_env_vars = sdk2.parameters.String("Env Vars for ya_package_2 (e.g. VAR1=val1 VAR2='val2'). May be used with Vault: $(vault:value:owner:name)", required=False)
            use_robot = sdk2.parameters.Bool("Run with robot", default=True)
            skip_on_empty_arcadia_patch = sdk2.parameters.Bool("Skip if arcadia_path param is empty/None", default=False)
            with sdk2.parameters.RadioGroup("Build type") as BuildSystem:
                BuildSystem.values.ya = BuildSystem.Value("ya", default=True)
                BuildSystem.values.distbuild = BuildSystem.Value("distbuild")
                BuildSystem.values.semidistbuild = BuildSystem.Value("semi_distbuild")

        with sdk2.parameters.Output:
            registry_url = sdk2.parameters.String("registry url with result image")
            tag = sdk2.parameters.String("Result image tag")

    def on_failure(self, prev_status):
        self.setup()
        if self.Parameters.release:
            self.release_log("build_failed")
        super(AfishaBuildDocker, self).on_failure(prev_status)

    def on_break(self, prev_status, status):
        self.setup()
        if self.Parameters.release:
            self.release_log("build_failed")
        super(AfishaBuildDocker, self).on_break(prev_status, status)

    def setup(self):
        self._init_rc()
        self._init_nyan()
        self._init_u()

    @property
    def rerun(self):
        if self.Context.rerun is ctm.NotExists:
            return False
        return self.Context.rerun

    @rerun.setter
    def rerun(self, value):
        self.Context.rerun = value

    def _get_release_issue(self):
        return self.st.issues.find("Queue: {} AND Tags: component_{} AND Tags: release_number_{}".format(
            self.component.st.release_queue, self.Parameters.component_name, self.Parameters.number))[0]

    @property
    def user(self):
        if not self.Parameters.use_robot:
            return self.author
        return self.component.robot.robot_name

    @property
    def release_issue(self):
        if self.Context.release_issue is ctm.NotExists:
            self._release_issue = self._get_release_issue()
            self.Context.release_issue = self._release_issue.key
        if not getattr(self, "_release_issue"):
            self._release_issue = self.st.issues[self.Context.release_issue]
        return self._release_issue

    @property
    def component(self):
        if getattr(self, "_component", None):
            return self._component
        if self.Context.component is not ctm.NotExists:
            self._component = self.rc.components.restore(**self.Context.component)
            return self._component
        self._component = self.rc.components.find(name=self.Parameters.component_name)[0]
        self.Context.component = self._component.to_dict()
        return self._component

    @property
    def release(self):
        if getattr(self, "_release", None):
            return self._release
        if self.Context.release is not ctm.NotExists:
            self._release = self.rc.releases.restore(**self.Context.release)
            return self._release
        try:
            self._release = self.component.releases(number=self.Parameters.number)[0]
        except IndexError:
            return None
        self.Context.release = self._release.to_dict()
        return self._release

    @release.setter
    def release(self, obj):
        self._release = obj
        self.Context.release = self._release.to_dict()

    def create_release(self):
        self._init_st(self.user, useragent="SandboxAfishaBuildDocker")

        if not self.Parameters.release or self.release:
            return
        logging.info("Creating release")
        self.release = self.rc.releases.new(self.component.link, self.Parameters.number, self.release_issue.key)
        self.release.save()

    def build_docker(self):
        logging.info("Building docker")
        build_task_params = {
            "owner": self.Parameters.owner,
            "priority": self.Parameters.priority,
            "description": "Building and publishing Docker image",
            "kill_timeout": 120*60,
            "checkout_arcadia_from_url": self.Parameters.checkout_arcadia_from_url,
            "arcadia_patch": self.Parameters.arcadia_patch,
            "checkout": False,
            "packages": self.Parameters.package_path,
            "package_type": "docker",
            "resource_type": AfishaTicketsBackendBinary.name,
            "docker_registry": self.DOCKER_REGISTRY,
            "docker_image_repository": self.component.build.registry_prefix,
            "docker_user": self.user,
            "docker_token_vault_name": "{}.docker-token".format(self.user),
            "docker_push_image": True,
            "docker_build_network": "host",
            "ya_yt_store": self.component.build.ya_yt_store,
            "use_aapi_fuse": True,
            "aapi_fallback": True,
            "ignore_recurses": True,
            "build_system": self.Parameters.BuildSystem,
        }
        if self.Parameters.docker_build_args:
            build_task_params["docker_build_arg"] = self.Parameters.docker_build_args
        if self.Parameters.ya_package_env_vars:
            build_task_params["env_vars"] = self.Parameters.ya_package_env_vars
        url = sdk2.svn.Arcadia.normalize_url(self.Parameters.checkout_arcadia_from_url)
        parsed_url = urlparse.urlparse(url)
        if parsed_url.scheme == self.ARCADIA_ARC_SCHEME:
            build_task_params["custom_version"] = self.Parameters.checkout_arcadia_from_url.split("/")[-1]
        if self.Parameters.pr:
            revision = sdk2.svn.Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url).revision or "unknown"
            build_task_params["custom_version"] = "pr-{}.{}".format(self.Parameters.arcanum_review_id, revision)
        if self.Parameters.use_new_ya_package:
            task = self.subtasks.get(self.BUILD_TASK_NAME, [None])[0]
        else:
            task = self.subtasks.get(self.BUILD_TASK_NAME_OLD, [None])[0]
        logging.debug("Already saved subtasks: %s", task)
        if task:
            task = sdk2.Task[task]
            logging.debug("Task status: %s", task.status)
            if task.status in ctt.Status.Group.SUCCEED:
                return
            else:
                if not self.rerun:
                    self.rerun = True
                    raise errors.TaskError("YA_PACKAGE not in SUCCESS status, you may try to rerun task if there was tmp error")
        else:
            logging.debug("There is no already saved subtasks, creating new one")
            if self.Parameters.use_new_ya_package:
                task = self.task(self.BUILD_TASK_NAME, **build_task_params)
            else:
                task = self.task(self.BUILD_TASK_NAME_OLD, **build_task_params)
        logging.debug("Enqueueing task: %s", task.id)
        self.rerun = False
        self.subtasks_run([task], timeout=7200)

    def save_output(self):
        logging.debug("Saving result parameters to output")
        if self.Parameters.use_new_ya_package:
            task = sdk2.Task[self.subtasks[self.BUILD_TASK_NAME][0]]
        else:
            task = sdk2.Task[self.subtasks[self.BUILD_TASK_NAME_OLD][0]]

        self.Parameters.registry_url = task.Context.output_resource_version
        self.Parameters.tag = ":".join(task.Context.output_resource_version.split(":")[1:])

    def release_log(self, status):
        if self.Parameters.release:
            if status == "built":
                self.release.log(status, version=self.Parameters.tag, build_task=str(self.id))
            else:
                self.release.log(status, build_task=str(self.id))

    def create_resource(self):
        # AfishaTicketsBackendBinary Why? BECAUSE
        logging.debug("Creating mocked resource")
        resource = AfishaTicketsBackendBinary(self, "Output file", "filename.txt",
                                              major_release_num=self.Parameters.tag.split(".")[0],
                                              minor_release_num=self.Parameters.tag.split(".")[1],
                                              component_name=self.Parameters.component_name)
        resource_data = sdk2.ResourceData(resource)
        resource_data.path.write_bytes("Mock for RM!")
        resource_data.ready()

    def on_execute(self):
        if self.Parameters.skip_on_empty_arcadia_patch and self.Parameters.arcadia_patch is None:
            # NOTE TESTENV-4013 DEVTOOLSSUPPORT-1193
            self.create_resource()
            return

        self.setup()
        with self.memoize_stage.create_release(commit_on_entrance=False):
            self.create_release()
        with self.memoize_stage.release_log_build(commit_on_entrance=False):
            self.release_log("build")
        self.build_docker()
        self.save_output()
        with self.memoize_stage.create_resource(commit_on_entrance=False):
            self.create_resource()
        with self.memoize_stage.release_log_built(commit_on_entrance=False):
            self.release_log("built")
