# coding: utf-8
import logging
from requests.exceptions import HTTPError, ConnectionError, Timeout

import sandbox.sdk2 as sdk2
from sandbox import common
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask


# 2 hours (1 retry == 60 sec)
MAX_RETRIES = 120


class AfishaCheckDeploy(AfishaSandboxBaseTask):

    BINARY_TASK_ATTR_TARGET = "Afisha/deploy/common/AfishaCheckDeploy"

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(AfishaSandboxBaseTask.Parameters):
        kill_timeout = 120
        with sdk2.parameters.Group("Settings") as settings_block:
            environment = sdk2.parameters.String("Environment id to check, e.g.: afisha.www.production", required=True)
            tag = sdk2.parameters.String("Tag to wait", required=True)
            wait = sdk2.parameters.Bool("Wait while status would be DEPLOYED", required=False, default=False)
            deploy_with_robot = sdk2.parameters.Bool("Deploy with robot", default=True)
            with deploy_with_robot.value[True]:
                deployer_username = sdk2.parameters.String("Robot username")

            with sdk2.parameters.Group("Callbacks") as callbacks:
                teamcity_callback = sdk2.parameters.Bool("teamcity", default=False)
                nyanbot_callback = sdk2.parameters.Bool("nyanbot", default=False)
                startrek_callback = sdk2.parameters.Bool("startrek", default=False)

                with teamcity_callback.value[True]:
                    teamcity_build_id = sdk2.parameters.String("Teamcity build ID", required=True)
                    teamcity_build_parameters = sdk2.parameters.Dict("Teamcity build parameters", required=False)

                with startrek_callback.value[True]:
                    startrek_ticket_id = sdk2.parameters.String("Release issue ID", required=True)
                    startrek_target_status = sdk2.parameters.String("Target issue status e.g.: Move to \'InProd\'", required=True)
                    startrek_comment_text = sdk2.parameters.String("Comment after execute transition", required=False)

                with nyanbot_callback.value[True]:
                    nyanbot_chatid = sdk2.parameters.String("Chat id", required=True)
                    with sdk2.parameters.RadioGroup("Nyan Parse mode") as nyanbot_parse_mode:
                        nyanbot_parse_mode.values[""] = nyanbot_parse_mode.Value("", default=True)
                        nyanbot_parse_mode.values["Markdown"] = nyanbot_parse_mode.Value("Markdown")
                        nyanbot_parse_mode.values["HTML"] = nyanbot_parse_mode.Value("HTML")
                    nyanbot_message_template = sdk2.parameters.String(
                        "Message template", default="{tag} успешно развернут на стенде [{app_id}](https://deploy.yandex-team.ru/stages/{stage}) \n"
                        "[График](https://yasm.yandex-team.ru/menu/media_infrastructure/plus/promo/{environment}/{app}/ работоспособности сервиса)")

    @property
    def user(self):
        if not self.Parameters.deploy_with_robot:
            return self.author
        return self.Parameters.deployer_username

    def setup(self):
        self._init_st(self.user, useragent="SandboxAfishaCheckDeploy")
        self._init_tc(self.user)
        self._init_nyan()

    def get_tags(self, env_id):
        try:
            versions = self.dc.versions(env_id)
        except (HTTPError, ConnectionError, Timeout) as error:
            if error.response and error.response.status_code == 404:
                raise common.errors.TaskFailure("Smthng totally wrong, no such env in clouds: %s" % error)
            raise common.errors.TemporaryError("Not 2xx answer from deployer (%s), will try again later" % error)
        else:
            self.Context.result = versions

    def _run_teamcity_callback(self):
        bt = self.tc.build_types[self.Parameters.teamcity_build_id].run_build(properties=self.Parameters.teamcity_build_parameters)
        self.log.info("Teamcity build started: %s (%s)", bt.id, bt.web_url)

    def _run_nyanbot_callback(self):
        message_content = {
            "tag": self.Parameters.tag,
            "app_id": self.Parameters.environment,
            "environment": self.Parameters.environment.split('.')[2],
            "app": self.Parameters.environment.split('.')[1],
            "stage": self.Parameters.environment.replace('.', "-")
        }
        message = self.Parameters.nyanbot_message_template.format(**message_content)
        nyan_mod = self.Parameters.nyanbot_parse_mode
        nyan_mod = None if len(nyan_mod) == 0 else nyan_mod
        self.nyan.send(self.Parameters.nyanbot_chatid, text=message, parse_mode=nyan_mod)

    def _run_startrek_callback(self):
        issue = self.st.issues[self.Parameters.startrek_ticket_id]
        transition = issue.transitions[self.Parameters.startrek_target_status]
        self.log.info("https://st.yandex-team.ru/%s was moved to \'%s\'", issue.key, self.Parameters.startrek_target_status)
        transition.execute(comment=self.Parameters.startrek_comment_text)

    def run_callbacks(self):
        callback_map = {
            "teamcity": self._run_teamcity_callback,
            "nyanbot": self._run_nyanbot_callback,
            "startrek": self._run_startrek_callback
        }
        for callback_type, callback_func in callback_map.items():
            if getattr(self.Parameters, "{}_callback".format(callback_type), False):
                self.log.info("%s callback is set to True, executing", callback_type)
                callback_func()

    def on_execute(self):
        self.log = logging.getLogger(__name__)
        self.setup()
        with self.memoize_stage.wait():
            self.log.info("Sleeping due to YD race")
            raise sdk2.WaitTime(60)
        self.log.info("Checking env %s", self.Parameters.environment)
        self._init_dc(self.user)
        success = False
        self.get_tags(self.Parameters.environment)
        self.log.info("Got tag: %s", self.Context.result)
        versions = [x["version"] for x in self.Context.result if x["version"] is not None]
        if versions.count(self.Parameters.tag) == len(versions):
            self.log.info("Versions synced, checking statuses")
            statuses = [x["state"] for x in self.Context.result]
            if statuses.count("Ready") + statuses.count("DEPLOYED") == len(statuses):
                self.log.info("Statuses OK, marking as deployed")
                success = True
        if not success:
            if not self.Parameters.wait:
                return
            with self.memoize_stage.set_retry:
                self.Context.retry = 0
            self.Context.retry += 1
            if self.Context.retry >= MAX_RETRIES:
                raise common.errors.TaskError("Retries exceeded (%s >= %s), environment still not in Deployed state (%s)" % (self.Context.retry, MAX_RETRIES, self.Context.result[0]["state"]))
            self.log.info("Still not deployed, waiting...")
            raise sdk2.WaitTime(60)
        with self.memoize_stage.run_callbacks():
            self.run_callbacks()
