# coding: utf-8
import logging
import re
from requests.exceptions import HTTPError

import sandbox.sdk2 as sdk2
import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
from sandbox.common import errors as sb_errors
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask

ANY_STRING_PATTERN = ".+"


class AfishaPrRelease(AfishaSandboxBaseTask):

    BINARY_TASK_ATTR_TARGET = "Afisha/deploy/common/AfishaPrRelease"
    BUILD_TASK = "AFISHA_BUILD_DOCKER"
    DEPLOY_TASK = "AFISHA_DEPLOY"
    CHECK_DEPLOY_TASK = "AFISHA_CHECK_DEPLOY"
    DEPLOY_BALANCER_TASK = "AFISHA_BALANCER_DEPLOY"

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(AfishaSandboxBaseTask.Parameters):
        kill_timeout = 120
        with sdk2.parameters.Group("Settings") as settings_block:
            components = sdk2.parameters.Dict(
                "Components to deploy. Value: list of applications to deploy separated by semicolon (;). "
                "If value empty: deploy all applications from components pr config", required=True)
            pr_id = sdk2.parameters.String("Pr ID", required=False, default=None)
            build = sdk2.parameters.Bool("Run build", default=True)
            with build.value[False]:
                version = sdk2.parameters.String("Registry tag", required=True)
            arcanum_review_id = sdk2.parameters.String("Review Id", required=False, default=None)
            iteration = sdk2.parameters.Integer("Review iteration, default: last", default=-1)
            env_prefix = sdk2.parameters.String("Env prefix for pr stages", required=False, default="pr-")
            domains_pattern = sdk2.parameters.String(
                "A regular expression for filtering domains in a successful deployment message",
                required=False,
                default=ANY_STRING_PATTERN
            )
            use_robot = sdk2.parameters.Bool("Run with robot", required=False, default=True)
            st_task = sdk2.parameters.String("Startrek tasks to write comments in them, e.g.: 'PLUSFRONT-12,ASD-34,AS-56'", default=None)
            docker_build_args = sdk2.parameters.Dict("Args for docker build in afisha_build_docker", required=False)
            ya_package_env_vars = sdk2.parameters.String("Env Vars for ya_package_2 (e.g. VAR1=val1 VAR2='val2'). May be used with Vault: $(vault:value:owner:name)", required=False)
        with sdk2.parameters.Group("Callbacks") as callbacks:
            teamcity_callback = sdk2.parameters.Bool("teamcity", default=False)
            with teamcity_callback.value[True]:
                teamcity_build_id = sdk2.parameters.String("Teamcity build ID", required=True)
                teamcity_build_parameters = sdk2.parameters.Dict("Teamcity build parameters", required=False)

    def on_enqueue(self):
        components = self.Parameters.components.items()
        components_pr_locks = []
        for component, applications in components:
            if not applications:
                self._init_rc()
                if not self.component.pr:
                    raise sb_errors.TaskFailure("Unable to find applications in infra-api and task parameters")
                applications = self.component.pr.applications
            else:
                applications = applications.split(";")
            for application in applications:
                components_pr_locks.append(ctt.Semaphores.Acquire(name="{}_{}_{}".format(component, application, self.pr_id), weight=1, capacity=1))
        release = (ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        self.Requirements.semaphores = ctt.Semaphores(acquires=components_pr_locks, release=release)

    @property
    def user(self):
        if not self.Parameters.use_robot:
            return self.author
        return self.component.robot.robot_name

    @property
    def pr_id(self):
        if self.Parameters.arcanum_review_id:
            return self.Parameters.arcanum_review_id
        return self.Parameters.pr_id

    @property
    def component(self):
        if getattr(self, "_components", None):
            return self._components[0]
        if self.Context.components is not ctm.NotExists:
            self._components = [self.rc.components.restore(**component) for component in self.Context.components]
            return self._components[0]
        self._components = [
            self.rc.components.find(name=component)[0] for component in self.Parameters.components.keys()]
        self.Context.components = [component.to_dict() for component in self._components]
        return self._components[0]

    def _get_components_deploy_config(self):
        result = {}
        applications = self.Parameters.components[self.component.name]
        if not applications:
            logging.debug("No applications defined, getting them from components pr config")
            if not self.component.pr:
                logging.warning("No applications defined, no pr config in component defined. Ingoring")
                raise sb_errors.TaskFailure("Unable to find applications in infra-api and task parameters")
            applications = self.component.pr.applications
        else:
            applications = applications.split(";")

        data = {"applications": applications, "env_prefix": self.component.pr.pr_env_prefix}
        if self.Parameters.build:
            if not self.component.build:
                logging.warning("No build config in component. Ignoring")
                raise sb_errors.TaskFailure("Unable to find build config in infra-api")
            data["build"] = self.component.build.to_dict()
        result[self.component.name] = data
        return result

    def _run_teamcity_callback(self):
        from teamcity_client import TeamcityClient
        vault = "{}.teamcity-token".format(self.user)
        token = sdk2.Vault.data(self.user, vault).rstrip()
        tca = TeamcityClient(server_url="teamcity.yandex-team.ru", auth=token)
        bt = tca.build_types[self.Parameters.teamcity_build_id].run_build(properties=self.Parameters.teamcity_build_parameters)
        logging.info("Teamcity build started: %s (%s)", bt.id, bt.web_url)

    @property
    def components_deploy_config(self):
        if self.Context.components_deploy_config is ctm.NotExists:
            self.Context.components_deploy_config = self._get_components_deploy_config()
        return self.Context.components_deploy_config

    def _get_pr_diff_info(self):
        params = {"fields": "diff_sets(zipatch(url,svn_base_revision))"}
        info = self.ac.get_review_request(self.Parameters.arcanum_review_id, **params)
        return info["data"]["diff_sets"][self.Parameters.iteration]["zipatch"]

    @property
    def arcadia_url(self):
        if self.Context.pr_diff_info is ctm.NotExists:
            self.Context.pr_diff_info = self._get_pr_diff_info()
        return "svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia@{}".format(self.Context.pr_diff_info["svn_base_revision"])

    @property
    def arcadia_patch(self):
        if self.Context.pr_diff_info is ctm.NotExists:
            self.Context.pr_diff_info = self._get_pr_diff_info()
        return "zipatch:{}".format(self.Context.pr_diff_info["url"])

    def setup(self):
        self._init_rc()
        self._init_ac(self.user)
        self._init_bbc(self.user)
        self._init_ghc(self.user)
        self._init_dc(self.user)
        self._init_st(self.user)

    def generate_env_id(self, app, env_prefix):
        return "{}.{}{}".format(app, env_prefix, self.pr_id)

    def register_pr(self):
        pr = self.rc.prs.new(self.component.link, self.pr_id)
        try:
            logging.debug("Register pr %s for component %s", self.pr_id, self.component.name)
            pr.save()
        except HTTPError as error:
            if error.response.status_code != 409:
                raise
            logging.debug("Already registered")

    def run_build(self, component_name, component_config, arcadia_url, arcadia_patch):
        subtasks = self.subtasks.get(self.BUILD_TASK, [])
        package_path = component_config["build"]["arc_package"]
        task = self.find_task(subtasks, "package_path", package_path)
        if task:
            if sdk2.Task[task].status in ctt.Status.Group.SUCCEED:
                logging.debug("Found success build %s in %s", component_name, task.id)
                return
            logging.debug("Not success build for %s in %s, restarting", component_name, task.id)
        else:
            task = self.task(
                self.BUILD_TASK,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description="Building {} from {} pr".format(component_name, self.Parameters.arcanum_review_id),
                kill_timeout=120,
                package_path=package_path,
                checkout_arcadia_from_url=arcadia_url,
                arcadia_patch=arcadia_patch,
                pr=True,
                arcanum_review_id=self.Parameters.arcanum_review_id,
                component_name=component_name,
                docker_build_args=self.Parameters.docker_build_args,
                ya_package_env_vars=self.Parameters.ya_package_env_vars,
            )
        return task

    def run_builds(self):
        if not self.Parameters.build:
            logging.debug("Build disabled via parameter, setting version to %s", self.Parameters.version)
            for component_config in self.components_deploy_config.values():
                component_config["version"] = self.Parameters.version
            return
        result_tasks = []
        for component_name, component_config in self.components_deploy_config.items():
            task = self.run_build(component_name, component_config, self.arcadia_url, self.arcadia_patch)
            if task:
                result_tasks.append(task)
        if result_tasks:
            self.subtasks_run(result_tasks, statuses=ctt.Status.Group.SUCCEED, timeout=300)
        for task_id in self.subtasks.get(self.BUILD_TASK, []):
            task_ = sdk2.Task[task_id]
            tag = task_.Parameters.tag
            package = task_.Parameters.package_path
            for component_config in self.components_deploy_config.values():
                if component_config["build"]["arc_package"] == package:
                    component_config["version"] = tag

    def deploy_stage(self, env_id, version):
        logging.debug("Starting deploy for %s", env_id)
        subtasks = self.subtasks.get(self.DEPLOY_TASK, [])
        logging.debug("Got deploy subtasks: %s", subtasks)
        task = self.find_task(subtasks, "qloud_environments", env_id)
        if task:
            if sdk2.Task[task].status in ctt.Status.Group.SUCCEED:
                logging.debug("Found success deploy %s in %s", env_id, task.id)
                return
            logging.debug("Not success deploy for %s in %s, restarting", env_id, task.id)
        else:
            logging.debug("No deploy task for %s, creating new", env_id)
            task = self.task(
                self.DEPLOY_TASK,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description="Deploying {}".format(env_id),
                kill_timeout=120,
                qloud_environments=env_id,
                registry_url=":{}".format(version),
                tgt_state="DEPLOYED",
                comment="Deploy for {} pr".format(self.pr_id),
                deploy_with_robot=self.Parameters.use_robot,
                deployer_username=self.user)
        return task

    def wait_stage(self, env_id, version):
        from afisha.infra.libs.deploy import DeployEnvInfo
        d_env = DeployEnvInfo.from_envid(env_id)
        subtasks = self.subtasks.get(self.CHECK_DEPLOY_TASK, [])
        task = self.find_task(subtasks, "environment", env_id)
        if task:
            if sdk2.Task[task].status in ctt.Status.Group.SUCCEED:
                return
            if sdk2.Task[task].status in ctt.Status.FAILURE:
                self.comment_pr("FAILURE")
                raise sb_errors.TaskFailure("%s failed, failing all process" % task)
        else:
            task = self.task(
                self.CHECK_DEPLOY_TASK,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description="Waiting while {} get deployed".format(d_env.url),
                kill_timeout=120,
                environment=env_id,
                tag=version,
                wait=True,
                deploy_with_robot=self.Parameters.use_robot,
                deployer_username=self.user)
        return task

    def deploy_balancer(self, env_id, version):
        logging.debug("Starting balancer deploy for %s", env_id)
        subtasks = self.subtasks.get(self.DEPLOY_BALANCER_TASK, [])
        logging.debug("Got deploy subtasks: %s", subtasks)
        task = self.find_task(subtasks, "qloud_environments", env_id)
        if task:
            if sdk2.Task[task].status in ctt.Status.Group.SUCCEED:
                logging.debug("Found success deploy %s in %s", env_id, task.id)
                return
            logging.debug("Not success deploy for %s in %s, restarting", env_id, task.id)
        else:
            logging.debug("No deploy task for %s, creating new", env_id)
            task = self.task(
                self.DEPLOY_BALANCER_TASK,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description="Deploying balancer for {}".format(env_id),
                kill_timeout=120,
                qloud_environments=env_id,
                registry_url=":{}".format(version),
                tgt_state="DEPLOYED",
                comment="Deploying for pr: {}".format(self.pr_id),
                deploy_with_robot=self.Parameters.use_robot,
                deployer_username=self.user)
        return task

    def run_callbacks(self):
        callback_map = {
            "teamcity": self._run_teamcity_callback,
        }
        for callback_type, callback_func in callback_map.items():
            if getattr(self.Parameters, "{}_callback".format(callback_type), False):
                logging.info("%s callback is set to True, executing", callback_type)
                callback_func()

    def run_deploy_tasks(self, *args):
        result_tasks = []
        for component_config in self.components_deploy_config.values():
            for application in component_config["applications"]:
                env_id = self.generate_env_id(application, component_config["env_prefix"])
                for func in args:
                    task = func(env_id, component_config["version"])
                    if task:
                        result_tasks.append(task)
        if result_tasks:
            logging.debug("Enqueue tasks from %s method: %s", func.__name__, result_tasks)
            self.subtasks_run(result_tasks, timeout=300)

    def get_deploying_app_env(self, env_id):
        from afisha.infra.libs.deployer import DeployerCloudEnvInfo
        subtasks = self.subtasks.get(self.DEPLOY_TASK, [])
        task = self.find_task(subtasks, "qloud_environments", env_id)
        return [DeployerCloudEnvInfo.from_url(data["url"]) for data in task.Context.result_envs]

    def comment_pr(self, status="SUCCEED"):
        from startrek_client import exceptions as st_exceptions
        vcs_clients_map = {
            "bitbucket": self.bbc,
            "arcadia": self.ac,
            "github": self.ghc
        }
        vcs_type = self.component.vcs.vcs_type
        vcs_path = self.component.vcs.vcs_path
        if status == "FAILURE":
            pr_message = "PR deploy was failed, please check the sandbox task: https://sandbox.yandex-team.ru/task/{}".format(self.id)
        else:
            pr_message = st_message = "PR задеплоился:\n"
            if self.component.name in self.components_deploy_config:
                for application in self.components_deploy_config[self.component.name]["applications"]:
                    env_id = self.generate_env_id(application, self.components_deploy_config[self.component.name]["env_prefix"])
                    envs = self.get_deploying_app_env(env_id)
                    for env in envs:
                        pr_message += "* [{}]({})\n".format(env.id, env.url)
                        st_message += "* (({} {}))\n".format(env.url, env.id)
                        for domain in self.dc.domains(env_id):
                            if not re.search(self.Parameters.domains_pattern, domain):
                                continue
                            pr_message += "  * [{domain}](https://{domain})\n".format(domain=domain)
                            st_message += "  * ((https://{domain} {domain}))\n".format(domain=domain)

        if pr_message:
            vcs_clients_map[vcs_type].comment(vcs_path, self.pr_id, pr_message)

        if self.Parameters.st_task and st_message:
            st_tasks = self.Parameters.st_task.split(',')
            for st_task in st_tasks:
                try:
                    self.st.issues[st_task].comments.create(text=st_message)
                except st_exceptions.Forbidden:
                    logging.warning("No access to %s, cannot comment. Skipping", st_task)
                except st_exceptions.NotFound:
                    logging.warning("Not found %s, skipping", st_task)
                except Exception as error:
                    logging.warning("Unknown exception while trying to comment %s: %s, skipping", st_task, error)

    def on_execute(self):
        self.setup()
        with self.memoize_stage.register_pr(commit_on_entrance=False):
            self.register_pr()
        self.run_builds()
        self.run_deploy_tasks(self.deploy_stage)
        self.run_deploy_tasks(self.wait_stage, self.deploy_balancer)
        with self.memoize_stage.comment_pr(commit_on_entrance=False):
            self.comment_pr()
        with self.memoize_stage.run_callbacks():
            self.run_callbacks()
