# coding: utf-8

import logging
import time
import sandbox.sdk2 as sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.projects.Afisha.base import AfishaSandboxBaseTaskBinary


class AfishaPreRelease(AfishaSandboxBaseTaskBinary):
    BINARY_TASK_ATTR_TARGET = "Afisha/deploy/common/AfishaPreRelease"
    DOCKER_REGISTRY = "registry.yandex.net"
    MERGE_TASK = "PLUS_TEST_BRANCH"
    DEPLOY_TASK = "AFISHA_DEPLOY"
    CHECK_DEPLOY_TASK = "AFISHA_CHECK_DEPLOY"
    BUILD_DOCKER_TASK = "YA_PACKAGE_2"

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(AfishaSandboxBaseTaskBinary.Parameters):
        kill_timeout = 300
        with sdk2.parameters.Group("Common settings") as common_settings_block:
            component_name = sdk2.parameters.String("Component name", required=True)
        with sdk2.parameters.Group("Specific settings", collapse=True) as specific_settings_block:
            use_robot = sdk2.parameters.Bool("Run with robot", default=True)
            correct_statuses = sdk2.parameters.List("Статусы при переходе в которые предрелиз пересобирается",
                                                        default=["Решен", "Можно тестировать", "Тестируется", "Протестировано", "Закрыт"],
                                                        required=False)
            manual_run_mode = sdk2.parameters.Bool("Режим работы для запуска задачи руками", default=True)

    def on_enqueue(self):
        component_flow_lock = ctt.Semaphores.Acquire(
            name="{}_{}".format(self.Parameters.component_name, 'prerelease'),
            weight=1, capacity=1)
        release = (ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        self.Requirements.semaphores = ctt.Semaphores(acquires=[component_flow_lock], release=release)

    @property
    def stopped(self):
        if self.Context.stopped is ctm.NotExists or not self.Context.stopped:
            return False
        return True

    def stop(self):
        if not self.stopped:
            self.Context.stopped = True

    def on_failure(self, prev_status):
        self.setup()
        self.stop()
        super(AfishaPreRelease, self).on_failure(prev_status)

    def on_break(self, prev_status, status):
        self.setup()
        if status in ctt.Status.STOPPED:
            self.stop()
        super(AfishaPreRelease, self).on_break(prev_status, status)

    @property
    def user(self):
        if not self.Parameters.use_robot:
            return self.author
        return self.component.robot.robot_name

    def setup(self):
        self.Context.prs_info = dict()
        self._init_rc()
        self._init_qc(self.user)
        self._init_st(self.user, useragent="SandboxAfishaPreRelease")
        self._init_u()
        self._init_abc(self.user)
        self._init_ac(self.user)
        self._init_infra(self.user)

    @property
    def component(self):
        if getattr(self, "_component", None):
            return self._component
        if self.Context.component is not ctm.NotExists:
            self._component = self.rc.components.restore(**self.Context.component)
            return self._component
        self._component = self.rc.components.find(name=self.Parameters.component_name)[0]
        self.Context.component = self._component.to_dict()
        return self._component

    def _get_applications(self):
        try:
            for flow in self.component.flows:
                if flow.name == 'testing':
                    return flow.applications
        except KeyError:
            pass
        raise RuntimeError("Cant find applications for %s flow in component config" % self.Parameters.flow)

    @property
    def applications(self):
        if self.Context.applications is ctm.NotExists:
            self.Context.applications = self._get_applications()
        return self.Context.applications

    def _get_duty_data_from_u(self, duty_id):
        return {
            "members": list(self.u.team(duty_id).keys()),
            "on_duty": self.u.onduty(duty_id)
        }

    def _get_duty_data_from_abc(self, duty_id):
        abc_slug, abc_duty_slug = duty_id.split(":")
        return {
            "members": self.abc.find_duty_members(abc_slug, abc_duty_slug),
            "on_duty": self.abc.find_duty(abc_slug, abc_duty_slug)
        }

    def _get_duty_data(self, duty_type, duty_id):
        if duty_type == "u":
            duty = self._get_duty_data_from_u(duty_id)
        elif duty_type == "abc":
            duty = self._get_duty_data_from_abc(duty_id)
        else:
            raise NotImplementedError()
        return duty

    @property
    def duties(self):
        if self.Context.duties is ctm.NotExists:
            if self.component.duties:
                result = {}
                for duty in self.component.duties:
                    result[duty.duty_name] = self._get_duty_data(duty.duty_type, duty.duty_id)
                self.Context.duties = result
            elif self.component.u_team:
                self.Context.duties = {"dev": self._get_duty_data("u", self.component.u_team)}
            else:
                self.Context.duties = {}
        return self.Context.duties

    def merge_test_branches(self):
        result_tasks = []
        subtasks = self.subtasks.get(self.MERGE_TASK, [])
        task = self.find_task(subtasks, "component", self.Parameters.component_name)
        if task:
            if sdk2.Task[task].status in ctt.Status.Group.SUCCEED:
                self.Context.branch = task.Parameters.branch
                self.Context.prs_info = task.Context.prs_info
                self.Context.merge_conflicts = False
                return
            elif sdk2.Task[task].status in ctt.Status.Group.BREAK:
                if task.Context.merge_conflicts:
                    issue = self.st.issues[self.Context.issue_key]
                    if not self.Context.merge_conflicts:
                        branch_pattern = "((https://a.yandex-team.ru/arcadia/?rev={0} {0}))"
                        first_branch = branch_pattern.format(task.Context.merge_condidates[0])
                        second_branch = branch_pattern.format(task.Context.merge_condidates[1])
                        comment = "Возникли конфликты при намерживании в предрелизную ветку между ветками {} и {}\n\n".format(first_branch, second_branch)
                        comment += "Подробности в логах ((https://sandbox.yandex-team.ru/task/{}/view задачи))".format(task.id)
                        issue.comments.create(text=comment, summonees=task.Context.conflicts_authors)
                    self.Context.merge_conflicts = True
                else:
                    self.Context.merge_conflicts = False
                result_tasks.append(task)
            else:
                result_tasks.append(task)
        else:
            task = self.task(
                self.MERGE_TASK,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description="Merging branches for prerelease",
                component=self.Parameters.component_name,
                prerelease_num=self.Context.prerelease_number,
                use_robot=self.Parameters.use_robot,
                robot_name=self.user,
                kill_timeout=120
            )
            result_tasks.append(task)
        if result_tasks:
            self.subtasks_run(result_tasks, statuses=ctt.Status.Group.SUCCEED, timeout=120)

    def start_deploy(self):
        result_tasks = []
        for application in self.applications:
            subtasks = self.subtasks.get(self.DEPLOY_TASK, [])
            task = self.find_task(subtasks, "qloud_environments", application)
            if task:
                if sdk2.Task[task].status in ctt.Status.Group.SUCCEED:
                    continue
            else:
                task = self.task(
                    self.DEPLOY_TASK,
                    owner=self.Parameters.owner,
                    priority=self.Parameters.priority,
                    description="Deploying {}".format(application),
                    kill_timeout=120,
                    qloud_environments=application,
                    registry_url=self.Context.registry_url,
                    tgt_state="DEPLOYED",
                    comment="https://st.yandex-team.ru/{}".format(self.Context.issue_key),
                    deploy_with_robot=self.Parameters.use_robot,
                    deployer_username=self.user
                )
                result_tasks.append(task)
        if result_tasks:
            self.subtasks_run(result_tasks, statuses=ctt.Status.Group.SUCCEED, timeout=300)

    def wait_deploy(self):
        from afisha.infra.libs.deploy import DeployEnvInfo
        result_tasks = []
        for application in self.applications:
            subtasks = self.subtasks.get(self.CHECK_DEPLOY_TASK, [])
            task = self.find_task(subtasks, "environment", application)
            d_env = DeployEnvInfo.from_envid(application)
            if task:
                if sdk2.Task[task].status in ctt.Status.Group.SUCCEED:
                    continue
            else:
                task = self.task(
                    self.CHECK_DEPLOY_TASK,
                    owner=self.Parameters.owner,
                    priority=self.Parameters.priority,
                    description="Waiting while {} get deployed".format(d_env.url),
                    kill_timeout=120,
                    environment=application,
                    tag=self.Context.tag,
                    wait=True,
                    deploy_with_robot=self.Parameters.use_robot,
                    deployer_username=self.user
                )
                result_tasks.append(task)
        if result_tasks:
            self.subtasks_run(result_tasks, statuses=ctt.Status.Group.SUCCEED, timeout=300)

    def build_docker(self):
        logging.info("Building docker")
        arcadia_build_url = "arcadia-arc:/#" + self.Context.branch
        tag = arcadia_build_url.split("/")[-1]
        build_task_params = {
            "owner": self.Parameters.owner,
            "priority": self.Parameters.priority,
            "description": "Building and publishing Docker image",
            "kill_timeout": 90*60,
            "checkout_arcadia_from_url": arcadia_build_url,
            "checkout": False,
            "packages": self.component.build.arc_package,
            "package_type": "docker",
            "docker_registry": self.DOCKER_REGISTRY,
            "docker_image_repository": self.component.build.registry_prefix,
            "docker_user": self.user,
            "docker_token_vault_name": "{}.docker-token".format(self.user),
            "docker_push_image": True,
            "docker_build_network": "host",
            "ya_yt_store": self.component.build.ya_yt_store,
            "use_aapi_fuse": True,
            "aapi_fallback": True,
            "ignore_recurses": True,
            "build_system": "ya",
            "custom_version": tag,
        }
        result_tasks = []
        subtasks = self.subtasks.get(self.BUILD_DOCKER_TASK, [])
        task = self.find_task(subtasks, "checkout_arcadia_from_url", arcadia_build_url)
        if task:
            if sdk2.Task[task].status in ctt.Status.Group.SUCCEED:
                self.Context.registry_url = task.Context.output_resource_version
                self.Context.tag = tag
                return
            else:
                result_tasks.append(task)
        else:
            task = self.task(
                self.BUILD_DOCKER_TASK,
                **build_task_params
            )
            result_tasks.append(task)
        if result_tasks:
            self.subtasks_run(result_tasks, statuses=ctt.Status.Group.SUCCEED, timeout=300)

    def issue_transition(self, issue, transition, **kwargs):
        from startrek_client import exceptions as st_exceptions

        try:
            issue.transitions[transition].execute(**kwargs)
            issue.update()
        except st_exceptions.NotFound:
            logging.warning("Not found transition %s for issue: %s", transition, issue.key)

    def create_issue(self):
        issues_old = self.st.issues.find(
            query='Queue: {} and status: !closed and tags: {}'.format(
                self.component.st.release_queue,
                'prerelease_{}'.format(self.Parameters.component_name)
            ),
            order=['updated']
        )
        if issues_old:
            logging.info("Open prerelease for %s already exists. Skipping ticket creation" % self.Parameters.component_name)
            issue_old_first = list(issues_old)[0]
            self.Context.issue_key = issue_old_first.key
            prerelease_number = issue_old_first.summary.split(' ')[-1]
            if prerelease_number.isdigit():
                self.Context.prerelease_number = int(prerelease_number)
            else:
                self.Context.prerelease_number = 1
            self.issue_transition(issue_old_first, "open")
            return
        closed_prereleases = self.st.issues.find(
            query='Queue: {} and status: closed and tags: {}'.format(
                self.component.st.release_queue,
                'prerelease_{}'.format(self.Parameters.component_name)
            ),
            order=['updated']
        )
        closed_prereleases_summary_nums = [closed_prerelease.summary.split(' ')[-1] for closed_prerelease in closed_prereleases]
        previous_prereleases = [int(num) for num in closed_prereleases_summary_nums if num.isdigit()]
        if previous_prereleases:
            previous_prerelease_number = sorted(previous_prereleases)[-1]
            prerelease_number = previous_prerelease_number + 1
        else:
            prerelease_number = 1
        component_title = " ".join(map(lambda x: x.title(), self.Parameters.component_name.split("_")))
        issue = self.st.issues.create(
            queue=self.component.st.release_queue,
            summary=u"Пререлиз {} {}".format(component_title, str(prerelease_number)),
            type={'name': u'Релиз'},
            tags=['prerelease_{}'.format(self.Parameters.component_name)]
        )
        self.Context.prerelease_number = prerelease_number
        self.Context.issue_key = issue.key

    def get_state_comment(self, issue):
        import datetime
        comments = list(issue.comments.get_all())
        min_create_time = None
        min_time_comment = None
        for comment in comments:
            create_time = datetime.datetime.strptime(comment.createdAt[:-5], "%Y-%m-%dT%H:%M:%S.%f")
            if comment.createdBy.id == self.user and (min_time_comment is None or create_time < min_create_time):
                min_create_time = create_time
                min_time_comment = comment
        return min_time_comment

    def update_issue(self):
        issue = self.st.issues[self.Context.issue_key]
        description = u"**Предрелизная ветка:** ((https://a.yandex-team.ru/arcadia/?rev={0} {0}))\n\n".format(self.Context.branch)
        description += u"===+Вошедшие пуллреквесты:\n"
        for pr in self.Context.prs_info:
            description += u"- {}\n".format(pr["summary"])
            description += u"<["
            for linked_issue in pr["issues"]:
                description += linked_issue + "\n"
            description += u"PR: ((https://a.yandex-team.ru/review/{0}/details {0}))]>\n".format(pr["id"])
        issue.update(
            description=description
        )
        time.sleep(15)
        for issue_linked in list(issue.links):
            logging.info("deleting %s linked issue", issue_linked.object.key)
            if issue_linked.object.key not in self.Context.current_issues:
                try:
                    issue_linked.delete()
                except:
                    logging.warning("failed to unlink %s. Check permissions for %s", issue_linked.object.key, self.component.robot.robot_name)

    def check_state(self):
        if self.Parameters.manual_run_mode:
            linked_issues = set()
            for pr in self.Context.prs_info:
                for pr_issue in pr["issues"]:
                    linked_issues.add(pr_issue)
            self.Context.new_issues = list(linked_issues)
            self.Context.current_issues = list(linked_issues)
            return

        issue = self.st.issues[self.Context.issue_key]
        state_comment = self.get_state_comment(issue)
        correct_statuses = self.Parameters.correct_statuses
        self.Context.stopped = True
        linked_issues = set()
        for pr in self.Context.prs_info:
            for pr_issue in pr["issues"]:
                linked_issues.add(pr_issue)
        self.Context.new_issues = []
        comment_linked_state = u""
        for linked_issue in linked_issues:
            comment_linked_state += linked_issue + u":"
            comment_linked_state += self.st.issues[linked_issue].status.name + u"\n"
        self.Context.current_issues = list(linked_issues)
        new_in_work = False
        if not state_comment:
            logging.info("State comment not found")
            issue.comments.create(text=comment_linked_state)
            self.Context.new_issues = list(linked_issues)
            self.Context.stopped = False
            time.sleep(15)
        else:
            state = state_comment.text.split("\n")[:-1]
            statuses = dict(map(lambda x: x.split(":"), state))

            for linked_key in linked_issues:
                linked_issue = self.st.issues[linked_key]
                if linked_key not in statuses.keys():
                    self.Context.new_issues.append(linked_key)
                    self.Context.stopped = False
                elif statuses[linked_key] not in correct_statuses \
                        and linked_issue.status.name in correct_statuses:
                    self.Context.new_issues.append(linked_key)
                    self.Context.stopped = False
                elif statuses[linked_key] in correct_statuses \
                        and linked_issue.status.name not in correct_statuses:
                    new_in_work = True
            removed_issues = set(statuses.keys()) - linked_issues

            if not self.stopped or new_in_work or removed_issues:
                state_comment.update(text=comment_linked_state)
                time.sleep(15)

    def summon_qa(self):
        issue = self.st.issues[self.Context.issue_key]

        qa_duty = self.duties.get("qa", {}).get("on_duty", None)
        dev_duty = self.duties.get("dev", {}).get("on_duty", None)
        if qa_duty:
            issue = self.st.issues[self.Context.issue_key]
            issue.update(
                assignee=qa_duty
            )
        elif dev_duty:
            issue = self.st.issues[self.Context.issue_key]
            issue.update(
                assignee=dev_duty
            )

        self.issue_transition(issue, "readyForTest")

        comment = u"Эта задача вошла в предрелиз: {}".format(self.Context.issue_key)
        for new_issue_key in self.Context.new_issues:
            new_issue = self.st.issues[new_issue_key]
            if qa_duty:
                new_issue.comments.create(text=comment, summonees=[qa_duty])
            else:
                new_issue.comments.create(text=comment, summonees=[new_issue.assignee.id])

            self.issue_transition(new_issue, "readyForTest")

    def notify_deployed(self):
        issue = self.st.issues[self.Context.issue_key]
        comment = u"Выложена версия в **TESTING**: ((https://sandbox.yandex-team.ru/task/{}/view {}))".format(self.id, self.Context.tag)
        issue.comments.create(text=comment)
        vcs_path = self.component.vcs.vcs_path
        for pr in self.Context.prs_info:
            pr_message = "PR вошел в предрелиз: https://st.yandex-team.ru/{}\n\n".format(self.Context.issue_key)
            self.ac.comment(vcs_path, pr["id"], pr_message)

    def on_execute(self):
        super(AfishaPreRelease, self).on_execute()
        self.setup()

        with self.memoize_stage.create_issue_stage(commit_on_entrance=False):
            self.create_issue()

        self.merge_test_branches()

        with self.memoize_stage.check_state_stage(commit_on_entrance=False):
            self.check_state()

        if self.stopped and not self.Parameters.manual_run_mode:
            return

        with self.memoize_stage.update_issue_stage(commit_on_entrance=False):
            self.update_issue()

        self.build_docker()
        self.start_deploy()
        self.wait_deploy()

        with self.memoize_stage.notify_deployed_stage(commit_on_entrance=False):
            self.notify_deployed()

        with self.memoize_stage.summon_qa_stage(commit_on_entrance=False):
            self.summon_qa()
