# coding: utf-8
import logging
import sandbox.sdk2 as sdk2
from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask
from sandbox.projects.release_machine.core.const import ReleaseStatus


ACTION_IN_PROD_COMPONENT = "ProdAction"
RELATED_STATUS_WHITE_LIST = {
    ReleaseStatus.prestable: ["testingInProd", "closed", "tested"],
    ReleaseStatus.stable: ["testingInProd", "closed", "tested"],
    ReleaseStatus.testing: ["testingInProd", "closed", "tested", "readyForTest", "testing"]
}
DEPLOY_TASK = "AFISHA_DEPLOY"
CHECK_DEPLOY_TASK = "AFISHA_CHECK_DEPLOY"
DEFAULT_DEPLOY_USER = "robot-afisha-deploy"


class AfishaRelease(AfishaSandboxBaseTask):

    BINARY_TASK_ATTR_TARGET = "Afisha/deploy/common/AfishaRelease"

    class Parameters(AfishaSandboxBaseTask.Parameters):
        """
        applications: list of applications to release
        mode: [auto, manual]. auto - use robot, manual - use user creds
        flow: [stable, testing] release flow
        auto flow params:
            build_task_id: task id with url to registry with builded image
            release_number: release number
            startrek_queue: startrek queue slug for release issue
            component_name: RM component name
            registry_url: registry url with image to deploy
        manual flow params:
            st: url to release issue

        target_state: [COMMITTED, DEPLOYED] - qloud target state
        """

        applications = sdk2.parameters.List("Applications", required=True)

        with sdk2.parameters.RadioGroup("Deployment mode") as mode:
            mode.values["auto"] = mode.Value("Auto mode (Only for RM)")
            mode.values["manual"] = mode.Value("Manual mode", default=True)

        with sdk2.parameters.RadioGroup("Release flow") as flow:
            flow.values[ReleaseStatus.stable] = flow.Value(ReleaseStatus.stable, default=True)
            flow.values[ReleaseStatus.prestable] = flow.Value(ReleaseStatus.prestable)
            with mode.value["auto"]:
                flow.values[ReleaseStatus.testing] = flow.Value(ReleaseStatus.testing)

        with mode.value["auto"]:
            build_task_id = sdk2.parameters.Integer("Build task number", required=True)
            release_number = sdk2.parameters.Integer("Release number", required=True)
            startrek_queue = sdk2.parameters.String("Startrek queue for release issue",
                                                    required=True)
            component_name = sdk2.parameters.String("RM component name")
            registry_url = sdk2.parameters.String("Registry url with image to deploy", required=True)

        with mode.value["manual"]:
            st = sdk2.parameters.Url("Release issue", required=True)

        with sdk2.parameters.RadioGroup("Target state") as target_state:
            target_state.values["COMMITTED"] = target_state.Value("COMMITTED", default=True)
            target_state.values["DEPLOYED"] = target_state.Value("DEPLOYED")

    @property
    def skip_release_tag(self):
        try:
            postfix = "".join([c if c.isalnum() else "_" for c in self.release_issue.summary.lower()])
            return "skip_release_{}".format(postfix)
        except AttributeError:
            self.logger.warning("Couldnt calc skip_release_tag, no release_issue attribute")
            return None

    def bad_status(self, issue, release_to):
        from startrek_client import exceptions as st_exceptions
        try:
            if issue.status.key in RELATED_STATUS_WHITE_LIST.get(release_to, []) or self.skip_release_tag in issue.tags:
                return False
            return True
        except st_exceptions.Forbidden:
            self.logger.warning("No access to %s, skipping bad_status check", issue.key)
            return False

    def __get_release_issue(self):
        """
            Provides
                self.release_issue object with startrek issue
                self.Context.release_number with release number int
                self.Context.release_issue_key with issue key str
            Stops task if flow == stable and release_issue not in tested state
        """
        if self.Parameters.mode == "manual":
            self.release_issue = self.st.issues[self.Parameters.st.split("/")[-1]]
            for tag in self.release_issue.tags:
                if tag.startswith("release_number_"):
                    self.Context.release_number = tag[len("release_number_"):]
                    break
        else:
            self.release_issue = self.st.issues.find(
                "Queue: {} AND Tags: {} AND Tags: release_number_{}".format(
                    self.Parameters.startrek_queue, self.Parameters.component_name, self.Parameters.release_number))[0]
            self.Context.release_number = self.Parameters.release_number
        self.Context.release_issue_key = self.release_issue.key

    def __check_release_issue(self):
        """ AS-991 """
        if self.Parameters.flow not in [ReleaseStatus.prestable, ReleaseStatus.stable]:
            return

        #  Deploy to stable only if release task in tested|fix (hotfix) state
        if self.release_issue.status.key not in ["tested", "fix"]:
            raise common.errors.TaskError("https://st.yandex-team.ru/%s task "
                                          "not in tested state (%s)" % (
                                              self.Context.release_issue_key, self.release_issue.status.key))

        prefix = u"Релиз не может поехать в stable из-за задач в некорректных статустах (корректные: {}) " \
                 u"или без тега {}:\n".format(RELATED_STATUS_WHITE_LIST, self.skip_release_tag)
        bad_tasks = u""
        for related_link in self.release_issue.links:
            if not self.bad_status(related_link.object, self.Parameters.flow):
                continue
            bad_tasks += u"  * {}\n".format(related_link.object.key)
        if bad_tasks:
            self.release_issue.comments.create(text=prefix+bad_tasks)
            raise common.errors.TaskError("Bad statuses in related issues, check release task comments")

    def __get_registry_url(self):
        """
            Provides
                self.Context.registry_url with addr of image to release str
                self.Context.build_task with build task id int
        """
        if self.Parameters.mode == "auto":
            self.Context.registry_url = self.Parameters.registry_url
            self.Context.build_task = self.Parameters.build_task_id
            return
        for comment in self.release_issue.comments.get_all():
            if u"Релиз //{}// выкатился в **TESTING**".format(
                    self.Context.release_number) in comment.text or u"Релиз //{}// выкатился в **PRESTABLE**".format(self.Context.release_number) in comment.text:
                self.Context.build_task = comment.text.split("\n")[-1].split()[-1][:-2]
        self.Context.registry_url = sdk2.Task[self.Context.build_task].Parameters.registry_url

    def __deploy(self):
        """
            Starts multiple deployment tasks and provides
                self.Context.deploy_tasks with id of deployment tasks int
        """
        deploy_with_robot = False if self.Parameters.mode == "manual" else True
        result_tasks = []
        for app in self.Parameters.applications:
            task = self.task(
                DEPLOY_TASK,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description="Deploying {}".format(app),
                qloud_environments=app,
                registry_url=self.Context.registry_url,
                tgt_state=self.Parameters.target_state,
                comment="https://st.yandex-team.ru/{}".format(self.Context.release_issue_key),
                deploy_with_robot=deploy_with_robot)
            result_tasks.append(task)
        self.subtasks_run(result_tasks)

    def __get_deployed_envs(self):
        """
            Provides self.Context.deployed_envs list of dict of {url: url, task: task, envId: envId}
                with deployed envs
        """
        result = []
        for task_id in self.subtasks[DEPLOY_TASK]:
            if sdk2.Task[task_id].Context.result_envs is ctm.NotExists:
                result_envs = []
                for env_name in sdk2.Task[task_id].Context.qloud_environments.split(";"):
                    result_envs.append({"url": 000, "task": task_id, "endId": env_name})
            else:
                result_envs = sdk2.Task[task_id].Context.result_envs
                for env in result_envs:
                    env.update({"task": task_id})
            result.extend(result_envs)
        self.Context.deployed_envs = result

    def __check_if_deploy_success(self):
        if self.Context.deployed_envs_status is ctm.NotExists:
            return False
        if list(self.Context.deployed_envs_status.values()).count("DEPLOYED") == len(self.Context.deployed_envs):
            return True
        return False

    def __make_deploy_comment(self):
        from afisha.infra.libs.platform import PlatformEnvInfo, PlatformEnvStatus
        if self.__check_if_deploy_success():
            message = u"Релиз //{}// выкатился в **{}**".format(self.Context.release_number, self.Parameters.flow.upper())
        else:
            message = u"Релиз //{}// поехал в **{}** (->**{}**)\n".format(self.Context.release_number, self.Parameters.flow.upper(), self.Parameters.target_state)
        message += u"\n\n#|\n|| Проект | Приложение | Окружение | Задача на деплой | Статус ||\n"
        _prjs = []
        _apps = []
        for env_data in sorted(self.Context.deployed_envs, key=lambda i: i["envId"]):
            if isinstance(env_data["url"], int):
                env = PlatformEnvInfo.from_envid(env_data["envId"])
            else:
                env = PlatformEnvInfo.from_url(env_data["url"])
            message += u"||"

            if env.prj not in _prjs:
                message += u" {} |".format(env.prj)
                _prjs.append(env.prj)
            else:
                message += u" |"

            if env.app not in _apps:
                message += u" {} |".format(env.app)
                _apps.append(env.app)
            else:
                message += u" |"

            message += u" (({} {})) |".format(env.url, env.env)
            if isinstance(env_data["url"], int):
                message += u" !!!!**((https://sandbox.yandex-team.ru/task/{task}/view Failed))**!!!! ".format(
                    task=env_data["task"])
            else:
                message += u" **((https://sandbox.yandex-team.ru/task/{task}/view Ok))** ".format(
                    task=env_data["task"])

            if self.Context.deployed_envs_status is ctm.NotExists:
                message += u"| //waiting...// ||\n"
            else:
                try:
                    status = PlatformEnvStatus(self.Context.deployed_envs_status.get(env.url, "UNKNOWN"))
                except AttributeError:
                    status = PlatformEnvStatus("UNKNOWN")
                message += u"| !!({}){}!! ||\n".format(status.color, status.value)

        message += u"|#"

        if not self.__check_if_deploy_success() and self.Parameters.target_state == "COMMITTED":
            message += u"\n\n**Для перевода окружений из COMMITTED в DEPLOYED необходимо " \
                       u"второму ответственному пройтись по окружениям и прожать справа " \
                       u"вверху Operations -> Deploy**\n" \
                       u"//В комментарии можно писать что угодно//"
        message += u"\n\n//Release task:// ((https://sandbox.yandex-team.ru/task/{task}/view " \
                   u"{task}))".format(task=self.id)
        message += u"\n//Build task:// ((https://sandbox.yandex-team.ru/task/{task}/view " \
                   u"{task}))".format(task=self.Context.build_task)

        if self.Context.release_comment_id is ctm.NotExists:
            self.logger.debug("No release_comment_id in context, creating new")
            self.Context.release_comment_id = self.release_issue.comments.create(text=message).id
        else:
            release_comment = self.release_issue.comments.get(self.Context.release_comment_id)
            release_comment.update(text=message)

    def __wait_deploy_end(self):
        """
            Wait while all deploy finish
        """
        from afisha.infra.libs.platform import PlatformEnvInfo
        deploy_with_robot = False if self.Parameters.mode == "manual" else True
        result_tasks = []
        for env_data in self.Context.deployed_envs:
            env = PlatformEnvInfo.from_url(env_data["url"])
            task = self.task(
                CHECK_DEPLOY_TASK,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description="Waiting while {} get deployed".format(env.url),
                environment=env.id,
                version=env.version,
                wait=True,
                deploy_with_robot=deploy_with_robot
            )
            result_tasks.append(task)
        self.subtasks_run(result_tasks)

    def __get_deploy_status(self):
        """ Provides self.Context.deployed_envs_status dict {url: status} """
        result = {}
        for task_id in self.subtasks[CHECK_DEPLOY_TASK]:
            result.update(sdk2.Task[task_id].Context.result)
        self.Context.deployed_envs_status = result

    def __update_issue_status(self):
        """ Updates startrek release issue status """
        from startrek_client import exceptions as st_exceptions

        try:
            if self.Parameters.flow == ReleaseStatus.testing and not self.__check_if_deploy_success() and self.release_issue.status.key != "open":
                self.release_issue.transitions["open"].execute(resolution=None)
            elif self.Parameters.flow == ReleaseStatus.testing and self.__check_if_deploy_success() and self.release_issue.status.key != "readyForTest":
                self.release_issue.transitions["readyForTest"].execute(resolution=None)
            elif self.Parameters.flow == ReleaseStatus.stable and self.release_issue.status.key == "tested" and self.__check_if_deploy_success():
                self.release_issue.transitions["closed"].execute(resolution="fixed")
            elif self.Parameters.flow == ReleaseStatus.stable and self.release_issue.status.key == "fix" and self.__check_if_deploy_success():
                self.release_issue.transitions["testingInProd"].execute()
        except st_exceptions.StartrekError as error:
            comment = u"//Не могу сменить статус задаче: %%{}%%\nСмените, пожалуйста, сами (если требуется)//".format(
                error)
            self.release_issue.comments.create(text=comment)

    def __update_related_issue_stable(self, issue):
        from startrek_client import exceptions as st_exceptions
        prefix = u"Релиз **{}** выехал в продакшн".format(self.release_issue.summary)
        link_components_names = []
        try:
            link_components_names = [comp.display for comp in issue.components]
        except st_exceptions.Forbidden:
            self.logger.error("{} no access (forbidden)".format(issue.key))

        if issue.status.key == "tested":
            issue.transitions["closed"].execute(comment=prefix+u", закрываю", resolution="fixed")
            return

        if issue.status.key == "testingInProd" or ACTION_IN_PROD_COMPONENT in link_components_names:
            text = u", а задача в testingInProd или с компонентом {}. Посмотри, пожалуйста".format(ACTION_IN_PROD_COMPONENT)
            to_summon = issue.assignee if issue.assignee is not None else issue.createdBy
            issue.comments.create(text=prefix+text, summonees=[to_summon])

    def __update_related_issue_test(self, issue):
        to_summon = issue.assignee if issue.assignee is not None else issue.createdBy
        if issue.status.key == "readyForTest" and (issue.description is None or u"Как тестировать" not in issue.description):
            issue.comments.create(text=u"Задача выложена в тестинг в рамках **{}** и у задачи нет блока 'Как тестировать'. "
                                       u"Опиши его, пожалуйста".format(self.release_issue.summary), summonees=[to_summon])
            return
        if self.bad_status(issue, self.Parameters.flow):
            issue.comments.create(text=u"Задача блокирует тестирование и релиз **{}** тк не находится в разрешенном статусе "
                                       u"({}) или не имеет тега для пропуска этого релиза ({})".format(self.release_issue.summary, RELATED_STATUS_WHITE_LIST, self.skip_release_tag),
                                  summonees=[to_summon])
            try:
                self.Context.incorrect_status[to_summon.id].append(issue.key)
            except KeyError:
                self.Context.incorrect_status[to_summon.id] = [issue.key]
            return

    def __update_related_issue_prestable(self, issue):
        return

    def __update_related_issues(self):
        """ AS-991 """
        from startrek_client import exceptions as st_exceptions
        actions = {
            ReleaseStatus.stable: self.__update_related_issue_stable,
            ReleaseStatus.prestable: self.__update_related_issue_prestable,
            ReleaseStatus.testing: self.__update_related_issue_test
        }
        self.Context.incorrect_status = {}
        for related_link in self.release_issue.links:
            try:
                actions[self.Parameters.flow](related_link.object)
            except st_exceptions.Forbidden:
                self.logger.error("{} Forbidden, skipping".format(related_link.object.key))
                continue
            except st_exceptions.NotFound:
                self.logger.error("{} transition not found, skipping".format(related_link.object.key))
                continue
        bad_status_header = u"#|\n||Ответственный|Задача||\n"
        bad_status_footer = u"|#\n"
        bad_status_table = u""
        for user, tasks in sorted(self.Context.incorrect_status.items()):
            bad_status_table += u"||staff:{}|{}||\n".format(user, tasks[0])
            for task in tasks[1:]:
                bad_status_table += u"|| |{}||\n".format(task)
        if bad_status_table:
            text = u"Нашел задачи в несовместимых с релизом статусах:\n" + bad_status_header + bad_status_table + bad_status_footer
            self.release_issue.comments.create(text=text)

    def __release_build_resource(self):
        """
            Only for manual release to stable
        """
        release_to = {
            ReleaseStatus.stable: ctt.ReleaseStatus.STABLE,
            ReleaseStatus.prestable: ctt.ReleaseStatus.PRESTABLE,
            ReleaseStatus.testing: ctt.ReleaseStatus.TESTING
        }
        self.server.release(
            task_id=self.Context.build_task,
            type=release_to[self.Parameters.flow],
            subject=self.Context.release_st_task)

    def on_execute(self):
        self.logger = logging.getLogger(__name__)
        if self.Parameters.mode == "manual":
            user = self.author
        else:
            user = DEFAULT_DEPLOY_USER
        self._init_st(user)
        self._init_qc(user)

        self.__get_release_issue()
        self.__check_release_issue()
        self.__get_registry_url()
        with self.memoize_stage.deploy:
            self.__deploy()
        with self.memoize_stage.comment_start_deploy:
            self.__get_deployed_envs()
            self.__make_deploy_comment()
            self.__update_issue_status()
        with self.memoize_stage.wait_deploy_end:
            self.__wait_deploy_end()
        self.__get_deploy_status()
        self.__make_deploy_comment()
        self.__update_issue_status()
        self.__update_related_issues()
        self.__release_build_resource()
