# coding: utf-8
import random
import itertools
import logging
import urllib
from enum import Enum
from time import time

import sandbox.sdk2 as sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.common import errors
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask
from sandbox.projects.release_machine.core.const import ReleaseStatus


class StTransitions(Enum):
    close = 1


TRANSITIONS_MAP = {
    StTransitions.close: ["close", "closed"]
}


class AfishaRelease2(AfishaSandboxBaseTask):
    BINARY_TASK_ATTR_TARGET = "Afisha/deploy/common/AfishaRelease2"
    DEPLOY_TASK = "AFISHA_DEPLOY"
    CHECK_DEPLOY_TASK = "AFISHA_CHECK_DEPLOY"
    ST_STATUS_WHITE_LIST = {
        ReleaseStatus.stable: ["tested", "fix"],
        ReleaseStatus.prestable: ["tested", "fix"],
        ReleaseStatus.testing: ["open", "tested", "closed", "readyForTest", "fix"]
    }
    ST_RELATED_STATUS_WHITE_LIST = {
        ReleaseStatus.stable: ["testingInProd", "closed", "tested", "testedDev", "inProd"],
        ReleaseStatus.prestable: ["testingInProd", "closed", "tested", "testedDev", "inProd"],
        ReleaseStatus.testing: ["testingInProd", "closed", "tested", "readyForTest", "testing", "testedDev",
                                "readyForBuild", "inProd"]
    }
    ST_RELATED_LINK_BLACK_LIST = {
        ReleaseStatus.stable: ["depends"],
        ReleaseStatus.prestable: ["depends"],
        ReleaseStatus.testing: []
    }
    ACTION_IN_PROD_COMPONENT = "ProdAction"
    # https://infra.yandex-team.ru/services/154/events
    INFRA_DRILLS_CONFIG = {
        "enviroment_id": 204,
        "service_id": 154
    }

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(AfishaSandboxBaseTask.Parameters):
        kill_timeout = 300
        with sdk2.parameters.Group("Common settings") as common_settings_block:
            component_name = sdk2.parameters.String("Component name", required=True)
            version = sdk2.parameters.String("Version to deploy", required=True)
            with sdk2.parameters.RadioGroup("Release flow") as flow:
                flow.values[ReleaseStatus.stable] = flow.Value(ReleaseStatus.stable, default=True)
                flow.values[ReleaseStatus.prestable] = flow.Value(ReleaseStatus.prestable)
                flow.values[ReleaseStatus.testing] = flow.Value(ReleaseStatus.testing)
            with sdk2.parameters.RadioGroup("Target state") as target_state:
                target_state.values["COMMITTED"] = target_state.Value("COMMITTED", default=True)
                target_state.values["DEPLOYED"] = target_state.Value("DEPLOYED")

        with sdk2.parameters.Group("Specific settings", collapse=True) as specific_settings_block:
            use_robot = sdk2.parameters.Bool("Run with robot", default=True)
            allow_downgrade = sdk2.parameters.Bool("Allow downgrade", default=False)
            allow_minor_downgrade = sdk2.parameters.Bool("Allow minor downgrade", default=False)
            allow_drills_deploy = sdk2.parameters.Bool("Allow deploy during drills ", default=False)
            release_number_from_version = sdk2.parameters.Bool("Get release num from version", default=False)
            release_build_resource = sdk2.parameters.Bool("Release build resourse in SB after release", default=True)
            rollback = sdk2.parameters.Bool("Skip all checks for stable deploy (use only in rollback)", default=False)

        with sdk2.parameters.Group("Callbacks") as callbacks:
            teamcity_callback = sdk2.parameters.Bool("teamcity", default=False)
            with teamcity_callback.value[True]:
                teamcity_build_id = sdk2.parameters.String("Teamcity build ID", required=True)
                teamcity_build_parameters = sdk2.parameters.Dict("Teamcity build parameters", required=False)

    def on_enqueue(self):
        component_flow_lock = ctt.Semaphores.Acquire(
            name="{}_{}".format(self.Parameters.component_name, self.Parameters.flow),
            weight=1, capacity=1)
        release = (ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        self.Requirements.semaphores = ctt.Semaphores(acquires=[component_flow_lock], release=release)

    def on_failure(self, prev_status):
        self.setup()
        self.stop()
        self.release_log("failed")
        super(AfishaRelease2, self).on_failure(prev_status)

    def on_break(self, prev_status, status):
        self.setup()
        if status in ctt.Status.STOPPED:
            self.stop()
        self.release_log("failed")
        super(AfishaRelease2, self).on_break(prev_status, status)

    @property
    def stopped(self):
        if self.Context.stopped is ctm.NotExists or not self.Context.stopped:
            return False
        return True

    def stop(self):
        if not self.stopped:
            self.Context.stopped = True

    @property
    def user(self):
        if not self.Parameters.use_robot:
            return self.author
        return self.component.robot.robot_name

    def setup(self):
        self._init_rc()
        self._init_qc(self.user)
        self._init_st(self.user, useragent="SandboxAfishaRelease2")
        self._init_staff(self.user)
        self._init_tc(self.user)
        self._init_nyan()
        self._init_u()
        self._init_abc(self.user)
        self._init_infra(self.user)

    @property
    def component(self):
        if getattr(self, "_component", None):
            return self._component
        if self.Context.component is not ctm.NotExists:
            self._component = self.rc.components.restore(**self.Context.component)
            return self._component
        self._component = self.rc.components.find(name=self.Parameters.component_name)[0]
        self.Context.component = self._component.to_dict()
        return self._component

    @property
    def build_log(self):
        if getattr(self, "_build_log", None):
            return self._build_log
        if self.Context.build_log is not ctm.NotExists:
            self._build_log = self.rc.logs.restore(**self.Context.build_log)
            return self._build_log
        self._build_log = self.component.build_log(self.Parameters.version)
        self.Context.build_log = self._build_log.to_dict()
        return self._build_log

    @property
    def build_task(self):
        return self.build_log.build_task

    @property
    def release(self):
        if getattr(self, "_release", None):
            return self._release
        if self.Context.release is not ctm.NotExists:
            self._release = self.rc.releases.restore(**self.Context.release)
            return self._release
        if self.Parameters.release_number_from_version:
            release_number = self.Parameters.version.split('.')[0]
        else:
            release_number = self.build_log.release.number
        self._release = self.component.releases(number=release_number)[0]
        self.Context.release = self._release.to_dict()
        return self._release

    @property
    def release_issue(self):
        if not getattr(self, "_release_issue", None):
            self._release_issue = self.st.issues[self.release.issue]
        return self._release_issue

    @property
    def st_status_ok(self):
        return self.ST_STATUS_WHITE_LIST[self.Parameters.flow]

    @property
    def have_testing(self):
        for flow in self.component.flows:
            if flow.name == "testing" and flow.applications:
                return True
        return False

    @property
    def st_related_status_ok(self):
        return self.ST_RELATED_STATUS_WHITE_LIST[self.Parameters.flow]

    @property
    def st_related_link_bad(self):
        return self.ST_RELATED_LINK_BLACK_LIST[self.Parameters.flow]

    @property
    def skip_release_tag(self):
        return "skip_release_{}_{}".format(self.Parameters.component_name, self.release.number)

    @property
    def component_tag(self):
        return "component_{}".format(self.Parameters.component_name)

    @property
    def deployed_msg(self):
        return u"Выложена версия в **{}**: ((https://sandbox.yandex-team.ru/task/{}/view {}))".format(
            self.Parameters.flow.upper(), self.id, self.Parameters.version)

    @property
    def release_comment(self):
        if self.Context.release_comment is ctm.NotExists:
            return None
        return self.release_issue.comments.get(self.Context.release_comment)

    @release_comment.setter
    def release_comment(self, comment):
        self.Context.release_comment = comment.id

    def _get_applications(self):
        try:
            for flow in self.component.flows:
                if flow.name == self.Parameters.flow:
                    return flow.applications
        except KeyError:
            pass
        raise RuntimeError("Cant find applications for %s flow in component config" % self.Parameters.flow)

    @property
    def applications(self):
        if self.Context.applications is ctm.NotExists:
            self.Context.applications = self._get_applications()
        return self.Context.applications

    @property
    def deployed(self):

        def check_length(x, y):
            return len(x) == len(y)

        def check_status(x):
            return all([sdk2.Task[y].status in ctt.Status.Group.SUCCEED for y in x]) if x else False

        stages = []
        for stage_task in [self.DEPLOY_TASK, self.CHECK_DEPLOY_TASK]:
            subtasks = self.subtasks.get(stage_task, [])
            status = check_length(self.applications, subtasks) and check_status(subtasks)
            stages.append(status)
        return all(stages)

    @property
    def component_team_members(self):
        if self.Context.component_team_members is ctm.NotExists:
            if self.component.u_team:
                self.Context.component_team_members = list(self.u.team(self.component.u_team).keys())
            else:
                self.Context.component_team_members = []
        return self.Context.component_team_members

    @property
    def component_team_members_wo_duty(self):
        return [x for x in self.component_team_members if x != self.on_duty]

    @property
    def on_duty(self):
        if self.Context.component_on_duty is ctm.NotExists:
            if self.component.u_team:
                self.Context.component_on_duty = self.u.onduty(self.component.u_team)
            else:
                self.Context.component_on_duty = None
        return self.Context.component_on_duty

    def _get_duty_data_from_u(self, duty_id):
        return {
            "members": list(self.u.team(duty_id).keys()),
            "on_duty": self.u.onduty(duty_id)
        }

    def _get_duty_data_from_abc(self, duty_id):
        abc_slug, abc_duty_slug = duty_id.split(":")
        return {
            "members": self.abc.find_duty_members(abc_slug, abc_duty_slug),
            "on_duty": self.abc.find_duty(abc_slug, abc_duty_slug)
        }

    def _get_duty_data(self, duty_type, duty_id):
        if duty_type == "u":
            duty = self._get_duty_data_from_u(duty_id)
        elif duty_type == "abc":
            duty = self._get_duty_data_from_abc(duty_id)
        else:
            raise NotImplementedError()
        return duty

    def _run_teamcity_callback(self):
        bt = self.tc.build_types[self.Parameters.teamcity_build_id].run_build(properties=self.Parameters.teamcity_build_parameters)
        logging.info("Teamcity build started: %s (%s)", bt.id, bt.web_url)

    def run_callbacks(self):
        callback_map = {
            "teamcity": self._run_teamcity_callback,
        }
        for callback_type, callback_func in callback_map.items():
            if getattr(self.Parameters, "{}_callback".format(callback_type), False):
                logging.info("%s callback is set to True, executing", callback_type)
                callback_func()

    @property
    def duties(self):
        if self.Context.duties is ctm.NotExists:
            if self.component.duties:
                result = {}
                for duty in self.component.duties:
                    result[duty.duty_name] = self._get_duty_data(duty.duty_type, duty.duty_id)
                self.Context.duties = result
            elif self.component.u_team:
                self.Context.duties = {"dev": self._get_duty_data("u", self.component.u_team)}
            else:
                self.Context.duties = {}
        return self.Context.duties

    @staticmethod
    def sync_issue_transition(issue, transitions, **kwargs):
        # via ASD-482
        from startrek_client import exceptions as st_exceptions

        err = None
        transitions = TRANSITIONS_MAP.get(transitions, [transitions])
        success_flag = False
        for transition in transitions:
            try:
                issue.transitions[transition].execute(**kwargs)
                issue.update()
                success_flag = True
                break
            except st_exceptions.NotFound as error:
                err = error
        if not success_flag:
            logging.warning("Not found transition for issue: %s", issue.key)
            raise err

    def related_status_bad(self, issue):
        if issue.status.key in self.st_related_status_ok:
            return False
        if self.skip_release_tag in issue.tags:
            return False
        return True

    def related_link_bad(self, link, issue):
        if link.type.id not in self.st_related_link_bad:
            return False
        if self.skip_release_tag in issue.tags:
            return False
        return True

    def get_bad_related(self):
        bad_related = []
        bad_access = []
        bad_link = []
        for link in self.release_issue.links:
            issue = link.object
            if not self.check_issue_access(issue):
                bad_access.append(issue.key)
                continue
            if not self.issue_in_scope(issue):
                continue
            if self.related_link_bad(link, issue):
                bad_link.append(issue.key)
            if self.related_status_bad(issue):
                bad_related.append(issue.key)
        return bad_related, bad_access, bad_link

    def check_issue_access(self, issue):
        # https://st.yandex-team.ru/ASD-1733
        from startrek_client import exceptions as st_exceptions
        try:
            self.st.issues[issue.key]
            return True
        except st_exceptions.Forbidden as error:
            logging.warning("Couldn\'t get access to issue %s. Error: %s", issue, error)
        return False

    @staticmethod
    def _get_component_tags(issue):
        component_tags = []
        for tag in issue.tags:
            if tag.startswith("component_"):
                component_tags.append(tag)
        return component_tags

    def issue_in_scope(self, issue):
        # issue.queue.key triggers unwanted Forbidden error
        if issue.key.split("-")[0] not in self.component.st.queue_white_list:
            return False
        component_tags = self._get_component_tags(issue)
        if self.skip_release_tag not in issue.tags:
            if not component_tags or self.component_tag in component_tags:
                return True
        return False

    def _get_issues_ino_scope_st_filter(self):
        """
        Generate url for filters
        :return: in_scope filter url, out_of_scope filter url. Tuple
        """
        base_url = "https://st.yandex-team.ru/issues/?" \
                   "_f=type+priority+key+summary+status+resolution+updated+dueDate+assignee+project+parent&" \
                   "_q="
        base_filter = ['"Linked To": {}'.format(self.release_issue.key)]
        status_filter = [
            "Status: !{}".format(status) for status in self.ST_RELATED_STATUS_WHITE_LIST[ReleaseStatus.stable]
        ]
        queue_filter = ["({})".format(" OR ".join([
            "Queue: {}".format(queue) for queue in self.component.st.queue_white_list
        ]))]
        skip_filter = [
            "Tags: !{}".format(self.skip_release_tag)
        ]
        scope_filter = " AND ".join(itertools.chain(base_filter, status_filter, queue_filter, skip_filter))
        in_scope_filter = urllib.quote_plus(scope_filter + " AND Tags: {}".format(self.component_tag))
        out_scope_filter = urllib.quote_plus(scope_filter + " AND Tags: !{}".format(self.component_tag))
        return (
            base_url+in_scope_filter,
            base_url+out_scope_filter)

    def _add_filters_to_desc(self):
        pattern = "=====Фильтры по задачам:"
        if pattern in self.release_issue.description:
            return
        in_scope_url, out_scope_url = self._get_issues_ino_scope_st_filter()
        template = u"{pattern}\n" \
                   u"* (({in_scope_url} Задачи явно вошедшие в релиз)) (есть тег %%{component_tag}%%)\n" \
                   u"* (({out_scope_url} Задачи неявно вошедшие в релиз)) (нет тега %%{component_tag}%%)\n\n\n".format(
                       pattern=pattern, in_scope_url=in_scope_url, out_scope_url=out_scope_url,
                       component_tag=self.component_tag)
        self.release_issue.update(description=template + self.release_issue.description)

    def release_log(self, status):
        self.release.log(status, version=self.Parameters.version, flow=self.Parameters.flow, release_task=str(self.id))

    def get_deploying_app_env(self, application):
        from afisha.infra.libs.deployer import DeployerCloudEnvInfo
        subtasks = self.subtasks.get(self.DEPLOY_TASK, [])
        task = self.find_task(subtasks, "qloud_environments", application)
        return [DeployerCloudEnvInfo.from_url(data["url"]) for data in task.Context.result_envs]

    def _find_release_issues_in_links(self, issue):
        from startrek_client import exceptions as st_exceptions
        rissues = []
        for rlink in issue.links:
            try:
                if rlink.object.queue.key == self.component.st.release_queue:
                    rissues.append(rlink.object)
            except st_exceptions.Forbidden:
                logging.warning("No access to %s issue, cant check if it is release issue. Skipping", rlink.object.key)
        return rissues

    def scoped_releases_closed(self, issue):
        to_check = [x for x in issue.tags if x.startswith("component_")]
        rissues = self._find_release_issues_in_links(issue)
        for i in rissues:
            if list(set(to_check) & set(i.tags)) and i.status.key not in ["closed", "released"]:
                return False
        return True

    def check_release_number(self):
        # Allow manual downgrade
        if self.Parameters.allow_downgrade:
            return
        last_deployed = self.component.last_deployed_release(self.Parameters.flow)
        if not last_deployed:
            # First release ?
            return
        logging.info("Current release number %s", self.release.number)
        logging.info("Last deployed release number %s", last_deployed)
        if int(last_deployed) > int(self.release.number):
            raise errors.TaskFailure("Last deployed release > current (%s > %s)" % (last_deployed, self.release.number))
        if self.Parameters.allow_minor_downgrade:
            return
        last_minor_deployed = self.component.last_deployed_minor_release(self.Parameters.flow).split('.')[1]
        current_minor_version = self.Parameters.version.split('.')[1]
        logging.info("Current minor release number %s", current_minor_version)
        logging.info("Last deployed minor release number %s", last_minor_deployed)
        if int(last_deployed) == int(self.release.number) and int(last_minor_deployed) > int(current_minor_version):
            raise errors.TaskFailure("Last deployed minor version > current (%s > %s)" % (last_minor_deployed, current_minor_version))

    def check_drills(self):
        if not self.Parameters.allow_drills_deploy:
            current_timestamp = int(time())
            current_timestamp_plus_two_hours = current_timestamp + 7200
            drills = self.infra.get_events(self.INFRA_DRILLS_CONFIG["enviroment_id"],
                                           self.INFRA_DRILLS_CONFIG["service_id"],
                                           current_timestamp,
                                           current_timestamp_plus_two_hours)
            if drills:
                return True
        return False

    def check_requirements_testing(self):
        pass

    def check_requirements_stable(self):
        requirements = []
        if self.check_drills():
            message = u"""
            Релиз не может поехать в stable, так как запланированы учения в ДЦ в следующие 2 часа
            """
            requirements.append(message)

        if self.release_issue.status.key not in self.st_status_ok and self.have_testing:
            message = u"""
            Релиз не может поехать в stable, тк статус {} не в списке разрешенных: {}
            """.format(self.release_issue.status.key, ",".join(self.ST_STATUS_WHITE_LIST[ReleaseStatus.stable]))
            requirements.append(message)

        bad_related, bad_access, bad_link = self.get_bad_related()
        if bad_access:
            message = "Пользователь {}@ не имеет доступа к задачам:\n* {}\nОбработка этих задач не будет произведена.".format(self.user, "\n* ".join(bad_access))
            self.release_issue.comments.create(text=message)
        if bad_link:
            message = u"""
            Релиз не может поехать в stable, тк есть блокирующие связанные задачи:
            {}
            """.format("\n".join(["* {}".format(x) for x in bad_link]))
            requirements.append(message)
        if bad_related:
            message = u"""
            Релиз не может поехать в stable, тк есть связанные задачи не в разрешенных статусах:
            {}
            """.format("\n".join(["* {}".format(x) for x in bad_related]))
            requirements.append(message)
        if requirements:
            raise RuntimeError("\n".join(requirements))

    def check_requirements_default(self):
        pass

    def check_requirements(self):
        if self.Parameters.rollback:
            return

        self.check_release_number()

        check_fn = {
            ReleaseStatus.testing: self.check_requirements_testing,
            ReleaseStatus.prestable: self.check_requirements_stable,
            ReleaseStatus.stable: self.check_requirements_stable,
        }
        check_fn.get(self.Parameters.flow, self.check_requirements_default)()

    def sync_release_issue_related_testing(self, issue):
        to_summon = issue.assignee if issue.assignee is not None else issue.createdBy
        if not self.issue_in_scope(issue):
            return
        templates_map = {
            "how2test": u"* Задача выложена в тестинг в рамках **{}** "
                        u"и в теле задачи не блока 'Как тестировать'.".format(self.release_issue.summary),
            "can_test_tag": u"Задача выложена в тестинг в рамках **{}**, у задачи стоит тег %%{}%%\n"
                            u"Перевожу в статус \"Можно тестировать\"".format(
                                self.release_issue.summary, self.component_tag),
            "can_test_notags": u"Задача выложена в тестинг в рамках **{}**, у задачи нет ни одного тега компонента\n"
                               u"Перевожу в статус \"Можно тестировать\"".format(self.release_issue.summary),
            "bad_status": u"* Задача блокирует тестирование и релиз **{}**. Как разблокировать (**или**):\n"
                          u"  * Перевести задачу в один из статусов: %%{}%%\n"
                          u"  * Выставить тег для пропуска этого релиза: %%{}%%\n".format(
                              self.release_issue.summary,
                              ", ".join(self.ST_RELATED_STATUS_WHITE_LIST[self.Parameters.flow]),
                              self.skip_release_tag),
            "no_component_tag": u"* Необходимо проставить тег компонента (может быть несколько). "
                                u"Для связи с %%{}%% тег: %%{}%%\n".format(
                                    self.Parameters.component_name, self.component_tag)
        }

        component_tags = self._get_component_tags(issue)
        message = ""
        problems = ""
        if self.component.st.force_component_in_related and not component_tags:
            problems += templates_map["no_component_tag"]
        if issue.status.key == "readyForTest":
            if issue.description is None or u"как тестировать" not in issue.description.lower():
                problems += templates_map["how2test"]
        if issue.status.key in ["resolved", "readyForBuild"]:
            if self.component_tag in self._get_component_tags(issue):
                message = templates_map["can_test_tag"]
            else:
                message = templates_map["can_test_notags"]
            self.sync_issue_transition(issue, "readyForTest")
        if self.related_status_bad(issue):
            problems += templates_map["bad_status"]
        result_kwargs = {"text": message}
        if problems:
            result_kwargs["text"] = result_kwargs["text"] + u"\n!!Проблемы:!!\n{}".format(problems)
            result_kwargs["summonees"] = [to_summon]
        if result_kwargs["text"]:
            issue.comments.create(**result_kwargs)
        return

    def sync_release_issue_testing_pre(self):
        self._add_filters_to_desc()
        # ASD-370
        if self.release_issue.status.key == "fix":
            return
        if self.release_issue.status.key != "open":
            self.sync_issue_transition(self.release_issue, "open", resolution=None)

    def sync_release_issue_testing_post(self):
        from startrek_client import exceptions as st_exceptions

        self.release_issue.comments.create(text=self.deployed_msg)
        # ASD-370
        if self.release_issue.status.key not in ["readyForTest", "fix"]:
            self.sync_issue_transition(self.release_issue, "readyForTest", resolution=None)
        qa_duty = self.duties.get("qa", {}).get("on_duty", self.get_last_tester())
        if qa_duty:
            self.release_issue.update(assignee=qa_duty)

        for related in self.release_issue.links:
            issue = related.object
            if not self.issue_in_scope(issue):
                continue
            try:
                self.sync_release_issue_related_testing(issue)
            except st_exceptions.NotFound:
                logging.warning("Not found transition for issue: %s", issue.key)

    def sync_release_issue_related_stable(self, issue):
        if not self.issue_in_scope(issue) or issue.status.key == "closed":
            return
        if self.ACTION_IN_PROD_COMPONENT in [comp.display for comp in issue.components]:
            message = u"Релиз **{}** выехал, задача требует действия на продакшне.\nПосмотри, пожалуйста".format(
                self.release_issue.summary)
            to_summon = issue.assignee if issue.assignee is not None else issue.createdBy
            if self.component.st.related_tinprod_summon.startswith("on_duty_"):
                dtype = self.component.st.related_tinprod_summon.replace("on_duty_", "")
                if dtype in self.duties.keys():
                    to_summon = self.duties[dtype]["on_duty"]
            issue.comments.create(text=message, summonees=[to_summon])
            return
        if issue.status.key in ["testingInProd", "testedDev"]:
            message = u"Релиз **{}** выехал, задача требует проверки в проде.\nПосмотри, пожалуйста".format(
                self.release_issue.summary)
            to_summon = issue.assignee if issue.assignee is not None else issue.createdBy
            if self.component.st.related_tinprod_summon.startswith("on_duty_"):
                dtype = self.component.st.related_tinprod_summon.replace("on_duty_", "")
                if dtype in self.duties.keys():
                    to_summon = self.duties[dtype]["on_duty"]
            issue.comments.create(text=message, summonees=[to_summon])
            return
        if issue.status.key in ["tested", "inProd"]:
            if not self.scoped_releases_closed(issue):
                return
            message = u"Релиз **{}** выехал".format(self.release_issue.summary)
            if self.component.st.auto_close:
                message += u", закрываю"
                self.sync_issue_transition(issue, StTransitions.close, comment=message, resolution="fixed")
            elif issue.status.key == "tested":
                message += u", автозакрытие выключено - перевожу в inProd"
                self.sync_issue_transition(issue, "inProd", comment=message)
            else:
                issue.comments.create(text=message)
            return

    def get_last_tester(self):
        result = None
        for entry in self.release_issue.changelog.get_all():
            for field in entry.fields:
                if field["field"].id == "status" and field["to"].key == "tested":
                    result = entry.updatedBy.id
        return result

    def _action_after_release_wo_close(self, transition="released"):
        st_comment = u"Релиз выложен в прод. Посмотри, пожалуйста"
        st_kwargs = {"text": st_comment}
        t_comment = u"[{}](https://st.yandex-team.ru/{}) выложен в прод. Посмотри, пожалуйста\n".format(
            self.release_issue.summary, self.release_issue.key)
        qa_summon = self.duties.get("qa", {}).get("on_duty", self.get_last_tester())
        qa_staff = None
        if qa_summon:
            qa_staff = self.staff.find_by_login(qa_summon, params={})[0]
            st_kwargs["summonees"] = qa_summon
        self.release_issue.comments.create(**st_kwargs)
        self.sync_issue_transition(self.release_issue, transition, resolution=None)
        if self.component.telegram_chat_id:
            if qa_staff:
                telegram_summon = qa_summon
                if qa_staff.public_telegram:
                    telegram_summon = "@{}".format(self.nyan.escape(qa_staff.public_telegram))
                t_comment += "{}".format(telegram_summon)
            self.nyan.send(self.component.telegram_chat_id, text=t_comment, parse_mode="Markdown")

    def sync_release_issue_stable_post(self):
        from startrek_client import exceptions as st_exceptions

        if self.release_comment:
            self.release_comment.update(text=self.deployed_msg)
        else:
            self.release_comment = self.release_issue.comments.create(text=self.deployed_msg)

        if self.release_issue.status.key in ["testingInProd", "closed", "released"]:
            pass
        elif self.release_issue.status.key == "open" and not self.have_testing:
            self.sync_issue_transition(self.release_issue, StTransitions.close, resolution="fixed")
        elif self.release_issue.status.key == "fix":
            self._action_after_release_wo_close("testingInProd")
        elif self.release_issue.status.key == "tested":
            if not self.component.st.auto_close:
                self._action_after_release_wo_close()
            elif "testingInProd" in set([x.status.key for x in self.release_issue.links]) and self.component.st.disable_auto_close_on_related_testinprod:
                self._action_after_release_wo_close()
            else:
                self.sync_issue_transition(self.release_issue, StTransitions.close, resolution="fixed")
        else:
            raise RuntimeError("Cant sync release issue, unexpected state: %s" % self.release_issue.status.key)

        for related in self.release_issue.links:
            issue = related.object
            if not self.issue_in_scope(issue):
                continue
            try:
                self.sync_release_issue_related_stable(issue)
            except st_exceptions.NotFound:
                logging.warning("Not found transition for issue: %s", issue.key)

    def sync_release_issue(self, stage):
        sync_fns = {
            ReleaseStatus.testing: {
                "prepare": self.sync_release_issue_testing_pre,
                "finish": self.sync_release_issue_testing_post
            },
            ReleaseStatus.stable: {
                "finish": self.sync_release_issue_stable_post
            }
        }
        if stage == "prepare" and not self.deployed or stage == "finish" and self.deployed:
            sync_fns.get(self.Parameters.flow, {}).get(stage, lambda: True)()

    def prepare_for_deploy(self):
        self.sync_release_issue(stage="prepare")

    def start_deploy(self):
        result_tasks = []
        for application in self.applications:
            logging.debug("Starting deploy for %s", application)
            subtasks = self.subtasks.get(self.DEPLOY_TASK, [])
            logging.debug("Got deploy subtasks: %s", subtasks)
            task = self.find_task(subtasks, "qloud_environments", application)
            if task:
                if sdk2.Task[task].status in ctt.Status.Group.SUCCEED:
                    logging.debug("Found success deploy %s in %s", application, task.id)
                    continue
                logging.debug("Not success deploy for %s in %s, restarting", application, task.id)
            else:
                logging.debug("No deploy task for %s, creating new", application)
                task = self.task(
                    self.DEPLOY_TASK,
                    owner=self.Parameters.owner,
                    priority=self.Parameters.priority,
                    description="Deploying {}".format(application),
                    kill_timeout=120,
                    qloud_environments=application,
                    registry_url=":{}".format(self.Parameters.version),
                    tgt_state=self.Parameters.target_state,
                    comment="https://st.yandex-team.ru/{}".format(self.release_issue.key),
                    deploy_with_robot=self.Parameters.use_robot,
                    deployer_username=self.user)
            result_tasks.append(task)
        if result_tasks:
            logging.debug("New deploy tasks, enqueue: %s", result_tasks)
            self.subtasks_run(result_tasks, statuses=ctt.Status.Group.SUCCEED, timeout=300)

    def comment_committed(self):
        if self.Parameters.target_state != "COMMITTED":
            return

        if not self.component_team_members:
            staff = []
        elif not self.component_team_members_wo_duty:
            staff = [random.choice(self.staff.find_by_login(",".join(self.component_team_members)))]
        else:
            staff = [random.choice(self.staff.find_by_login(",".join(self.component_team_members_wo_duty)))]

        s_message = u"Закоммичена версия в **{}**: ((https://sandbox.yandex-team.ru/task/{}/view {}))\n" \
                    u"Её необходимо подтвердить:\n".format(self.Parameters.flow.upper(), self.id, self.Parameters.version)
        t_message = u"[{}](https://st.yandex-team.ru/{}) ([{}](https://sandbox.yandex-team.ru/task/{}/view)) -> *{}*\n" \
                    u"Необходимо подтверждение:\n".format(
                        self.release_issue.summary, self.release_issue.key, self.Parameters.version, self.id,
                        self.Parameters.flow.upper())
        for app in self.applications:
            app_envs = self.get_deploying_app_env(app)
            for app_env in app_envs:
                s_message += u"  * (({} {}))\n".format(app_env.url, app_env.id)
                t_message += u"  \\* [{}]({})\n".format(app_env.id, app_env.url)

        self.release_comment = self.release_issue.comments.create(text=s_message, summonees=[x.login for x in staff])
        if self.component.telegram_chat_id:
            telegram_summon = ["@{}".format(self.nyan.escape(x.public_telegram)) if x.public_telegram else x.login for x in staff]
            t_message += "{}".format(" ".join(telegram_summon))
            self.nyan.send(self.component.telegram_chat_id, text=t_message, parse_mode="Markdown")

    def wait_deploy(self):
        from afisha.infra.libs.deploy import DeployEnvInfo
        result_tasks = []
        for application in self.applications:
            subtasks = self.subtasks.get(self.CHECK_DEPLOY_TASK, [])
            task = self.find_task(subtasks, "environment", application)
            d_env = DeployEnvInfo.from_envid(application)
            if task:
                if sdk2.Task[task].status in ctt.Status.Group.SUCCEED:
                    continue
            else:
                task = self.task(
                    self.CHECK_DEPLOY_TASK,
                    owner=self.Parameters.owner,
                    priority=self.Parameters.priority,
                    description="Waiting while {} get deployed".format(d_env.url),
                    kill_timeout=120,
                    environment=application,
                    tag=self.Parameters.version,
                    wait=True,
                    deploy_with_robot=self.Parameters.use_robot,
                    deployer_username=self.user)
            result_tasks.append(task)
        if result_tasks:
            self.subtasks_run(result_tasks, statuses=ctt.Status.Group.SUCCEED, timeout=300)

    def check_deploy(self):
        if not self.deployed:
            raise RuntimeError("Some problems with deploy. Fix them and rerun me")

    def release_build_resource(self):
        """ Only for sandbox task """
        release_to = {
            ReleaseStatus.stable: ctt.ReleaseStatus.STABLE,
            ReleaseStatus.prestable: ctt.ReleaseStatus.PRESTABLE,
            ReleaseStatus.testing: ctt.ReleaseStatus.TESTING
        }
        try:
            self.server.release(task_id=int(self.build_task), type=release_to[self.Parameters.flow], subject=self.release_issue.key)
        except ValueError:
            logging.warning("Cant release build with such id. Skipping (id: %s)", self.build_task)

    def finish_deploy(self):
        if self.Parameters.release_build_resource:
            self.release_build_resource()
        if self.Parameters.rollback:
            return
        self.sync_release_issue(stage="finish")

    def on_execute(self):
        if self.stopped:
            raise errors.TaskFailure("Cant rerun stopped manually task")
        self.setup()

        with self.memoize_stage.log_deploy(commit_on_entrance=False):
            self.release_log("deploy")
        with self.memoize_stage.check_requirements(commit_on_entrance=False):
            self.check_requirements()

        self.prepare_for_deploy()
        self.start_deploy()

        with self.memoize_stage.log_wait_deploy(commit_on_entrance=False):
            self.release_log("wait_deploy")
        with self.memoize_stage.comment_committed(commit_on_entrance=False):
            self.comment_committed()

        self.wait_deploy()
        self.check_deploy()

        with self.memoize_stage.log_deployed(commit_on_entrance=False):
            self.release_log("deployed")

        self.finish_deploy()
        self.release_log("finished")
        with self.memoize_stage.run_callbacks():
            self.run_callbacks()
