# coding: utf-8
import sandbox.sdk2 as sdk2
import time
import logging
import requests
from requests.adapters import HTTPAdapter
from requests.exceptions import RequestException
from requests.packages.urllib3.util.retry import Retry
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask
from sandbox.projects.common.juggler import jclient
from jinja2 import Environment, BaseLoader

juggler_raw_hosts_to_alert = set()
juggler_raw_hosts = set()
TEMPLATE = template = """<table border="5">
        <caption>Balancer ITS weights</caption>
        <tr>
          <th rowspan="2">Balancer L7 dashboard</th>
          <th rowspan="2">Section</th>
          <th colspan="2">SAS</th>
          <th colspan="2">VLA</th>
          <th colspan="2">MAN</th>
          <th colspan="2">MYT</th>
          <th colspan="2">IVA</th>
        </tr>
        <tr>
          <td>current</td>
          <td>default</td>
          <td>current</td>
          <td>default</td>
          <td>current</td>
          <td>default</td>
          <td>current</td>
          <td>default</td>
          <td>current</td>
          <td>default</td>
        </tr>
        {% if balancers is defined %}
          {% for balancer, sections in balancers.items() %}
            {% for section, locations in sections.items() %}
            <tr>
              <td> <a href="https://nanny.yandex-team.ru/ui/#/l7heavy/{{ balancer }}"> {{ balancer }} </a> </td>
              <td>{{ section }}</td>
                {% for weights in locations.values() %}
                  {% if not weights.get('current_weight') == weights.get('default_weight') %}
                    <td style="background-color:#FF0000">{{ weights.get('current_weight') }}</td>
                    <td style="background-color:#FF0000">{{ weights.get('default_weight') }}</td>
                  {% else %}
                    <td>{{ weights.get('current_weight') }}</td>
                    <td>{{ weights.get('default_weight') }}</td>
                  {% endif %}
                {% endfor %}
              </tr>
            {% endfor %}
          {% endfor %}
        {% endif %}"""


class AfishaBalancerWeightMonitoring(AfishaSandboxBaseTask):
    BINARY_TASK_ATTR_TARGET = 'Afisha/infra/AfishaBalancerWeightMonitoring'
    INFRA_DC_OUTAGE_SERVICE_ID = 154
    INFRA_DC_OUTAGE_ENV_ID = 204

    class Parameters(AfishaSandboxBaseTask.Parameters):

        with sdk2.parameters.Group('Juggler parameters') as juggler_parameters:
            juggler_report_is_enabled = sdk2.parameters.Bool(
                'Enable Juggler reports', default=False)
            juggler_host_name = sdk2.parameters.String(
                'Host name', required=False)
            juggler_service_name = sdk2.parameters.String(
                'Service name', required=False)

        with sdk2.parameters.Group('Other parameters') as juggler_parameters:
            balancers_json_config = sdk2.parameters.JSON('Json config as string', required=True)
            nanny_token = sdk2.parameters.YavSecret("Yav secret with nanny oauth token", required=True)

    class Context(sdk2.Task.Context):
        map_balancers = {}

    @property
    def juggler_report_is_enabled(self):
        return self.Parameters.juggler_report_is_enabled

    @property
    def juggler_host_name(self):
        return self.Parameters.juggler_host_name

    @property
    def juggler_service_name(self):
        return self.Parameters.juggler_service_name

    @property
    def juggler_sandbox_url(self):
        return 'https://sandbox.yandex-team.ru/task/{}'.format(self.id)

    @property
    def juggler_author_staff_url(self):
        return 'https://staff.yandex-team.ru/{}'.format(self.author)

    @property
    def juggler_task_creation_date(self):
        return self.created.strftime('%Y-%m-%d %H:%M:%S')

    @property
    def juggler_task_description(self):
        return (
            '{sandbox_type}\n{sandbox_url}\n'
            'created at {create_date}\n'.format(
                sandbox_type=self.type,
                sandbox_url=self.juggler_sandbox_url,
                create_date=self.juggler_task_creation_date))

    def on_break(self, prev_status, status):
        description = (
            '{task_description}'
            '\n{previous_status} -> {current_status}'
            '\nCould not chek ITS weights for some reasons'.format(
                task_description=self.juggler_task_description,
                previous_status=prev_status, current_status=status))
        self.send_status_to_juggler(
            status='CRIT',
            description=description)

    def on_failure(self, prev_status):
        description = (
            '{task_description}'
            '\n{previous_status} -> FAILURE'
            '\nCould not chek ITS weights for some reasons'.format(
                task_description=self.juggler_task_description,
                previous_status=prev_status))
        self.send_status_to_juggler(
            status='CRIT',
            description=description)

    @sdk2.header(title="Balancer ITS weights")
    def result_table(self):

        environ = Environment(loader=BaseLoader)
        environ_data = environ.from_string(TEMPLATE)
        return environ_data.render(balancers=self.Context.map_balancers)

    def send_status_to_juggler(self, status, host=None, service=None, description=None):
        if self.juggler_report_is_enabled:
            jclient.send_events_to_juggler(
                host=(
                    host
                    if host is not None
                    else self.juggler_host_name),
                service=(
                    service
                    if service is not None
                    else self.juggler_service_name),
                status=status,
                description=(
                    description
                    if description is not None
                    else self.juggler_task_description))

    def check_drills(self):
        from afisha.infra.libs.infra import InfraApiClient
        unixtime_now = int(time.time())
        unixtime_check_threshold = unixtime_now + 3600 * 2
        infra_client = InfraApiClient(token="secret")
        events = infra_client.get_events(
            environment_id=self.INFRA_DC_OUTAGE_ENV_ID,
            service_id=self.INFRA_DC_OUTAGE_SERVICE_ID,
            search_from=unixtime_now,
            search_to=unixtime_check_threshold,
            severity='major'
        )
        for event in events:
            drills_start_unixtime = int(event['start_time'])
            if unixtime_now <= drills_start_unixtime <= unixtime_check_threshold:
                logging.info('drills are active: %s - %s %s' % (event['start_time_date'],
                                                                event['finish_time_date'],
                                                                event['description']))
                return True

    def get_balancer_weights(self):
        import jsonschema
        config = self.Parameters.balancers_json_config
        nanny_token = self.Parameters.nanny_token.data()

        config_schema = {
            "type": "object",
            "properties": {
                "max_closed_time": {"type": "number"},
                "balancer_id": {"type": "string"},
                "sections": {"type": "array",
                             "items": {"type": "object",
                                       "properties": {
                                           "section_id": {"type": "string"},
                                           "juggler_hostname": {"type": "string"}
                                            },
                                       "required": ["section_id", "juggler_hostname"]
                                       },
                             },
            },
            "required": ["max_closed_time", "balancer_id", "sections"]
        }

        result = {}
        for conf in config.values():
            balancer_id = conf["balancer_id"]
            if jsonschema.validate:
                jsonschema.validate(instance=conf, schema=config_schema)

            headers = {
                "Authorization": "OAuth %s" % (nanny_token["nanny.token"]),
            }
            its_url = 'https://ext.its.yandex-team.ru'
            its_uri = 'v2/l7/heavy/%s/weights/snapshots/?limit=1&skip=0' % balancer_id
            http = requests.Session()
            retries = Retry(total=3, backoff_factor=0.1, status_forcelist=[408, 429, 500, 502, 503, 504])
            http.mount('https://', HTTPAdapter(max_retries=retries))
            try:
                logging.info('Trying to connect %s/%s' % (its_url, its_uri))
                request = http.get('%s/%s' % (its_url, its_uri), headers=headers, verify=False, timeout=30)
                data = request.json()
                result[balancer_id] = data
            except RequestException as error:
                logging.error(
                    'Exception %s.Could not connect to %s/%s\n Error:\n%s' % (type(error), its_url, its_uri, error))
        return result

    def on_execute(self):
        balancer_check_config = self.Parameters.balancers_json_config
        drills_check_active = self.check_drills()
        balancer_its_configs = self.get_balancer_weights()
        ouput_list = []

        for config_item in balancer_check_config.values():
            balancer_id = config_item["balancer_id"]
            sections = config_item['sections']
            self.Context.map_balancers.update({balancer_id: {}})
            for section in sections:
                balancer_max_closed_time = config_item['max_closed_time']
                balancer_config_section_id = section["section_id"]
                self.Context.map_balancers[balancer_id].update({balancer_config_section_id: {}})
                juggler_raw_host = section['juggler_hostname']
                juggler_raw_hosts.add(juggler_raw_host)
                logging.info('checking balancer %s section %s' % (balancer_id, balancer_config_section_id))
                balancer_its_config = balancer_its_configs[balancer_id]
                weight_timestamp = balancer_its_config['result'][0]['change_info']['ctime'] / 1000
                unixtime_now = int(time.time())

                locations = balancer_its_config['result'][0]['content']['sections'][balancer_config_section_id][
                    'locations']
                for data_center in locations.items():
                    dc_name = data_center[0]
                    current_weight = data_center[1]['weight']
                    default_weight = data_center[1]['default_weight']
                    location_weight = {
                        dc_name: {
                            'current_weight': current_weight,
                            'default_weight': default_weight
                        }
                    }
                    self.Context.map_balancers[balancer_id][balancer_config_section_id].update(location_weight)

                    logging.info('checking ITS weights in dc: %s' % dc_name)
                    if (current_weight != default_weight) and drills_check_active:
                        message = 'drills alert! balancer %s section %s ITS weight skew detected: current %s default %s dc %s' % (
                            balancer_id,
                            balancer_config_section_id,
                            current_weight,
                            default_weight,
                            dc_name)
                        logging.info(message)
                        juggler_raw_hosts_to_alert.add(juggler_raw_host)

                    elif current_weight != default_weight and unixtime_now - weight_timestamp >= balancer_max_closed_time:
                        message = 'WARNING, current %s ' \
                                  'default %s dc %s for more than %d seconds' % (current_weight,
                                                                                 default_weight,
                                                                                 dc_name,
                                                                                 balancer_max_closed_time)
                        logging.info(message)
                        juggler_raw_hosts_to_alert.add(juggler_raw_host)
                    else:
                        logging.info(
                            "OK, current weight is - %s, default weight is - %s" % (current_weight, default_weight))
        juggler_raw_hosts.add(self.juggler_host_name)
        for host in juggler_raw_hosts - juggler_raw_hosts_to_alert:
            logging.info("OK status sent to juggler host %s", host)
            self.send_status_to_juggler(status='OK', host=host)
        if juggler_raw_hosts_to_alert:
            for host in juggler_raw_hosts_to_alert:
                logging.info("CRIT status sent to juggler host %s", host)
                self.send_status_to_juggler(status='CRIT', host=host, description=self.juggler_task_description)
