# coding: utf-8
import logging

import sandbox.sdk2 as sdk2
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask


TELEGRAM_MSG_TMPL = u"Привет! Сегодня дежурят:\n\n" \
                    u"{% for duty in duties_list %}\n" \
                    u"{% if duty.on_duty.public_telegram %}\n" \
                    u"{% set tg_str = '@%-15s'|format(escape(duty.on_duty.public_telegram)) %}\n" \
                    u"{% else %}\n" \
                    u"{% set tg_str = '%-15s'|format(escape(duty.on_duty.login)) %}\n" \
                    u"{% endif %}\n" \
                    u"[🔗]({{ duty.url }}) `{{ '%-25s'|format(escape(duty.name)) }}` " \
                    u"[staff]({{ duty.on_duty.url }}) | {{ tg_str }}\n" \
                    u"{% endfor %}"
WIKI_LINK = "https://nda.ya.ru/t/9b7HwYK14Hfrpt"
OAUTH_LINK = "https://nda.ya.ru/t/xdqfmgy74Hg2Xr"
ABC_DUTY_EXAMPLE = "https://abc.yandex-team.ru/services/afisha_infra/duty/?role=3882"


class Duties(object):

    def __init__(self, *args):
        """
        :param args: initial duties, list(AbcDuty)
        """
        self.duties = list(args)
        self._dict_duties = {}
        for duty in args:
            self._add_to_dict(duty)

    def _add_to_dict(self, duty):
        self._dict_duties[str(duty.id)] = duty

    def add(self, duty):
        """
        Add duty to duties
        :param duty: AbcDuty
        :return: None
        """
        self.duties.append(duty)
        self._add_to_dict(duty)

    def to_dict(self):
        return self._dict_duties


class AfishaDutyNotifier(AfishaSandboxBaseTask):

    BINARY_TASK_ATTR_TARGET = "Afisha/infra/AfishaDutyNotifier"

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(AfishaSandboxBaseTask.Parameters):
        kill_timeout = 120
        with sdk2.parameters.Group("Settings") as settings_block:
            token = sdk2.parameters.YavSecret(u"Секрет с токеном к staff/abc под ключем token\n"
                                              u"Можно получить здесь: {}".format(OAUTH_LINK), required=True)
            telegram_chat_id = sdk2.parameters.String(u"ID телеграм чата\n"
                                                      u"Уведомления приходят от @ugc_media_Bot "
                                                      u"(он должен быть в чате)", required=True)
            duties = sdk2.parameters.List(u"Ссылки на дежурства для уведомления\n"
                                          u"Пример: {}\n"
                                          u"(можно найти на странице дежурств в abc)".format(ABC_DUTY_EXAMPLE),
                                          required=True)
            telegram_msg_tmpl = sdk2.parameters.String(u"Темплейт сообщения для отправки\n",
                                                       default=TELEGRAM_MSG_TMPL,
                                                       multiline=True)

    def setup(self):
        self._init_nyan()
        self._init_yav_abc(self.Parameters.token, key="token")
        self._init_yav_staff(self.Parameters.token, key="token")

    def _render_jinja_template(self, template, *duties_list, **duties_dict):
        """
        Render jinja2 template with params from kwargs
        :param template: jinja template, str
        :param duties_list: list of AbcDuty objects as is
        :param duties_dict: AbcDuties as dict, k - duty id, v - AbcDuty object, dict
        :return: rendered data, str
        """
        from jinja2 import Environment, BaseLoader
        environ = Environment(loader=BaseLoader)
        environ.globals.update(escape=self.nyan.escape)
        environ.trim_blocks = True
        environ.lstrip_blocks = True
        environ.rstrip_blocks = True
        environ_data = environ.from_string(template)
        logging.info("Rendering message from template: '%s' with duties_list: %s, duties_dict: %s",
                     template, duties_list, duties_dict)
        return environ_data.render(duties_list=duties_list, duties_dict=duties_dict)

    def get_duties(self, *args):
        """
        Getting duty data from abc
        :param args: abc_duty_url, list
        :return: yields AbcDuty objects
        """
        from afisha.infra.libs.abc import AbcDuty, parse_abc_duty_url
        for schedule_url in args:
            logging.info("Getting duty data for %s", schedule_url)
            schedule_data = self.abc.get_schedule_by_id(*parse_abc_duty_url(schedule_url))
            logging.info("Got schedule data: %s", schedule_data)
            yield AbcDuty.from_dict(schedule_data, self.abc, self.staff)

    def send_telegram_notify(self, chat_id, msg_tmpl, duties):
        """
        Sends telegram message via self.nyan (@ugc_media_Bot)
        :param chat_id: telegram chat id, str
        :param msg_tmpl: message template, str
        :param duties: duties, Duties object
        :return: answer from nyan api, dict
        """
        message = self._render_jinja_template(msg_tmpl, *duties.duties, **duties.to_dict())
        message += u"\n_сгенерировано в_ [sb](https://sandbox.yandex-team.ru/task/{}/view) | " \
                   u"[doc]({})".format(self.id, WIKI_LINK)
        logging.info("Sending '%s' message to %s chat_id", message, chat_id)
        return self.nyan.send(chat_id, text=message, parse_mode="Markdown")

    def on_execute(self):
        self.setup()
        duties = Duties(*self.get_duties(*self.Parameters.duties))
        result = self.send_telegram_notify(
            self.Parameters.telegram_chat_id, self.Parameters.telegram_msg_tmpl, duties)
        logging.info("Message sent, returned id: %s", result)
