# coding: utf-8
import logging
import os
import shutil
import re

import sandbox.sdk2 as sdk2
from sandbox.sdk2.paths import get_logs_folder
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc

from sandbox.common import errors
from sandbox.common.types.misc import DnsType
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import decorators as deco


class AfishaFlywayMigration(AfishaSandboxBaseTask):

    BINARY_TASK_ATTR_TARGET = 'Afisha/infra/AfishaFlywayMigration'
    CHECKOUT_FOLDER = 'src'
    FLYWAY_FOLDER = 'flyway'
    FLYWAY_RESOURCE_ID = 1517110482
    FLYWAY_ERRORS = [
        "canceling statement due to lock timeout"
    ]

    class Requirements(sdk2.Requirements):
        dns = DnsType.DNS64
        disk_space = 1024
        client_tags = (
            ctc.Tag.GENERIC | ctc.Tag.LINUX_XENIAL
        )
        cores = 4

    class Parameters(AfishaSandboxBaseTask.Parameters):
        with sdk2.parameters.Group('Settings') as settings_block:
            pg_username = sdk2.parameters.String('PG user', required=True)
            robot_name = sdk2.parameters.String('Robot name', required=True)
            mdb_folder_id = sdk2.parameters.String('MDB folder ID', required=True)
            mdb_cluster_filter = sdk2.parameters.String('MDB cluster filter', required=True)
            github_repository = sdk2.parameters.String('GitHub repository', required=True)
            github_branch = sdk2.parameters.String('GitHub branch', default='master', required=True)
            migration_folder = sdk2.parameters.String('Migrations folder', default='migrations', required=True)
            db_blacklist = sdk2.parameters.List('List of blacklist DBs', required=True)
            secret_mdb = sdk2.parameters.YavSecret('Secret MDB', required=True)
            secret_pg = sdk2.parameters.YavSecret('Secret PG', required=True)
            secret_robot = sdk2.parameters.YavSecret('Secret Robot', required=True)
            validate_migration_files = sdk2.parameters.Bool('Validate migration files', required=False, default=False)
            restarts_number = sdk2.parameters.Integer('The number of times to restart migrations on error', required=False, default_value=20)
            with validate_migration_files.value[True]:
                validate_migration_files_regexp = sdk2.parameters.String("Validationg Regexp", required=True)

    @deco.retries(max_tries=3, delay=60)
    def _gh_checkout(self):

        robot_password = self.Parameters.secret_robot.data()['password']
        logging.info('Cloning the repository: {} and branch {}'.format(self.Parameters.github_repository,
                                                                       self.Parameters.github_branch))
        logging.info('Fetching data from repository: {}:'.format(self.Parameters.github_repository))

        repository = 'https://{robot_name}:{robot_password}@{repository}'.format(
            robot_name=self.Parameters.robot_name,
            robot_password=robot_password,
            repository=self.Parameters.github_repository)
        clone_command = 'git clone --branch {github_branch} {repository} {checkout_folder}'.format(
            github_branch=self.Parameters.github_branch, repository=repository,
            checkout_folder=self.CHECKOUT_FOLDER)
        with sdk2.helpers.ProcessLog(self, logger='git_checkout') as pl:
            os.mkdir(self.CHECKOUT_FOLDER)
            return_code = sdk2.helpers.subprocess.Popen(
                clone_command.split(),
                stdout=pl.stdout,
                stderr=pl.stderr
            ).wait()
        if return_code != 0:
            if os.path.exists(self.CHECKOUT_FOLDER):
                shutil.rmtree(self.CHECKOUT_FOLDER)
            raise Exception('Checkout process failed!')

    def _validate_migrations(self, path):
        if not self.Parameters.validate_migration_files:
            logging.info("Validate_migration_files - False, skipping")
            return
        migrations = os.listdir(path)
        for migr_file in migrations:
            if re.match(self.Parameters.validate_migration_files_regexp, migr_file):
                logging.info("Filename {} matchs pattern".format(migr_file))
            else:
                logging.error("Filename {} does not match pattern".format(migr_file))
                raise Exception("Check migration filesnames")

    def _generate_flyway_config(self, hosts, database):

        logging.info('Generating flyway config for DB: {}'.format(database))
        cluster_hosts = ','.join(['{}:6432'.format(host) for host in hosts])
        opts = 'targetServerType=master&ssl=true&prepareThreshold=0&sslrootcert=./mdb.pem'
        flyway_url = 'jdbc:postgresql://{}/{}?{}'.format(cluster_hosts, database, opts)
        flyway_config_file = 'flyway/conf/flyway.conf'
        fu.write_lines(flyway_config_file, ['flyway.url={}'.format(flyway_url),
                                            'flyway.baselineOnMigrate=true',
                                            'flyway.outOfOrder=true',
                                            'flyway.ignoreMissingMigrations=true',
                                            'flyway.locations=filesystem:{}/{}'.format(
                                                self.CHECKOUT_FOLDER,
                                                self.Parameters.migration_folder)])

    @deco.retries(max_tries=3, delay=60)
    def _get_certificate(self):
        logging.info('Getting CA certificate')
        curl_command = 'curl https://crls.yandex.net/allCAs.pem --output mdb.pem'
        with sdk2.helpers.ProcessLog(self, logger='curl') as pl:
            return_code = sdk2.helpers.subprocess.Popen(
                curl_command.split(),
                stdout=pl.stdout,
                stderr=pl.stderr
            ).wait()
        if return_code != 0:
            raise Exception('Get certificate process failed!')

    def _unpack_resource(self):
        flyway_resource = sdk2.resource.ResourceData(sdk2.Resource[self.FLYWAY_RESOURCE_ID])
        untar_command = 'tar xf {src_path} --strip-components=1 --directory {dst_path}'.format(
            src_path=str(flyway_resource.path),
            dst_path=self.FLYWAY_FOLDER)
        with sdk2.helpers.ProcessLog(self, logger='flyway_untar') as pl:
            os.mkdir(self.FLYWAY_FOLDER)
            return_code = sdk2.helpers.subprocess.Popen(
                untar_command.split(),
                stdout=pl.stdout,
                stderr=pl.stderr
            ).wait()
        if return_code != 0:
            raise Exception('Unpack process failed!')

    def _call_flyway(self, env, database, try_num):
        flyway_command = './flyway/flyway migrate'
        with sdk2.helpers.ProcessLog(self, logger='flyway_migrate_{}_{}'.format(database, try_num)) as pl:
            return_code = sdk2.helpers.subprocess.Popen(
                flyway_command.split(),
                stdout=pl.stdout,
                stderr=pl.stderr,
                env=env
            ).wait()
        with open(os.path.join(get_logs_folder(), 'flyway_migrate_{}_{}.err.log'.format(database, try_num))) as flyway_log:
            for line in flyway_log:
                for error in self.FLYWAY_ERRORS:
                    if error in line:
                        logging.info("Expected error occurred: %s" % error)
                        return False
        if return_code != 0:
            raise Exception('FlyWay process failed!')
        return True

    def on_enqueue(self):
        component_flow_lock = ctt.Semaphores.Acquire(
            name='{}_{}_{}'.format('afisha_flyway_migration', self.Parameters.mdb_folder_id,
                                   self.Parameters.github_branch),
            weight=1, capacity=1)
        release = (ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        self.Requirements.semaphores = ctt.Semaphores(acquires=[component_flow_lock], release=release)

    def on_execute(self):
        from afisha.infra.libs.mdb import MdbClientPostgres

        # Token expires in 1 year, to change it use doc below:
        # https://docs.yandex-team.ru/cloud/container-registry/operations/authentication#oauth

        mdb_token = self.Parameters.secret_mdb.data()['mdb-token']
        pg_password = self.Parameters.secret_pg.data()['mdb.pg.ticketsystem']
        mdb = MdbClientPostgres(token=mdb_token)

        env = {
            'FLYWAY_USER': self.Parameters.pg_username,
            'FLYWAY_PASSWORD': pg_password,
        }
        self._unpack_resource()
        self._get_certificate()
        self._gh_checkout()
        self._validate_migrations(self.CHECKOUT_FOLDER + "/" + self.Parameters.migration_folder)

        logging.info('Getting clusters from MDB folder: {}'.format(self.Parameters.mdb_folder_id))
        clusters = mdb.cluster_list(folder_id=self.Parameters.mdb_folder_id,
                                    filter=self.Parameters.mdb_cluster_filter)

        if not clusters:
            raise errors.TaskFailure('No clusters matched')

        logging.info('Got next clusters: {}'.format(','.join(['{}'.format(cluster.name) for cluster in clusters])))
        try:
            for cluster in clusters:
                hosts = mdb.cluster_hosts_list(cluster_id=cluster.id)
                databases = mdb.database_list(cluster_id=cluster.id, filter=self.Parameters.db_blacklist)
                for database in databases:
                    self._generate_flyway_config(hosts, database)
                    for try_num in range(self.Parameters.restarts_number):
                        if self._call_flyway(env, database, try_num):
                            break
                    else:
                        raise Exception('Migration restarts on error reached %s' % self.Parameters.restarts_number)
        except Exception as e:
            logging.error('Something went wrong: {}'.format(e))
            raise errors.TaskFailure('Something went wrong')
