# coding: utf-8
import logging
import sandbox.sdk2 as sdk2
from sandbox.common import errors
import sandbox.common.types.task as ctt
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask


class AfishaMdbRecovery(AfishaSandboxBaseTask):

    BINARY_TASK_ATTR_TARGET = 'Afisha/infra/AfishaMdbRecovery'
    DNS_MAPPING = {"afisha-qa": "mongo-qa.afisha.tst.yandex.net",
                   "afisha-editor": "mongo-editor.afisha.tst.yandex.net"
                   }

    CLUSTER_CONFIG = {"resourcePresetId": "m2.5xlarge"}

    class Parameters(AfishaSandboxBaseTask.Parameters):
        with sdk2.parameters.Group("Settings") as settings_block:
            src_cluster_id = sdk2.parameters.String("Source: cluster id", required=True)
            dst_cluster_folder = sdk2.parameters.String("Destination: cluster folder", required=True)
            with sdk2.parameters.String("Destination: cluster name") as dst_cluster_name:
                dst_cluster_name.values.afisha_qa = "afisha-qa"
                dst_cluster_name.values.afisha_editor = "afisha-editor"
            db_username = sdk2.parameters.String("Username: ", required=True, default="afisha")
            secret_id = sdk2.parameters.YavSecret("YAV: DNS/MDB tokens", required=True)

    def on_enqueue(self):
        component_flow_lock = ctt.Semaphores.Acquire(
            name="{}_{}".format("afisha_mdb_recovery", self.Parameters.dst_cluster_name),
            weight=1, capacity=1)
        release = (ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        self.Requirements.semaphores = ctt.Semaphores(acquires=[component_flow_lock], release=release)

    def recovery_mdb(self):
        from afisha.infra.libs.mdb import MdbClientMongo
        logging.info("Recovery MDB cluster")
        secret = self.Parameters.secret_id
        mdb_token = secret.data()["mdb-token"]

        mdb = MdbClientMongo(token=mdb_token)
        dst_cluster_name = self.Parameters.dst_cluster_name.replace("_", "-")

        # get config from source cluster
        src_cluster_config = mdb.cluster_get(cluster_id=self.Parameters.src_cluster_id)
        logging.info("Cluster configuration:\n%s", src_cluster_config)
        src_cluster_config.resources.update(self.CLUSTER_CONFIG)
        # get last backup id from source cluster
        src_cluster_last_backup = mdb.cluster_backups_list(cluster_id=self.Parameters.src_cluster_id)
        # get configs from folder by cluster name
        dst_configs = mdb.cluster_list(folder_id=self.Parameters.dst_cluster_folder,
                                       filter=dst_cluster_name)

        # define postfix for cluster name
        postfix = '-1'
        for config in dst_configs:
            if config.name.endswith('-1'):
                postfix = '-2'

        for config in dst_configs:
            if config.name.endswith('-1') or config.name.endswith('-2'):
                old_config = config

        new_dst_cluster_name = dst_cluster_name + postfix
        # get hosts from cluster name
        old_dst_cluster_hosts = mdb.cluster_hosts_list(cluster_id=old_config.id)
        # restore cluster
        operation_id = mdb.restore_mongo_cluster(src_cluster_last_backup, new_dst_cluster_name, src_cluster_config)
        # search cluster_by name
        new_dst_cluster_configs = mdb.cluster_list(folder_id=self.Parameters.dst_cluster_folder,
                                                   filter=new_dst_cluster_name)
        # get hosts from new cluster
        new_dst_cluster_hosts = mdb.cluster_hosts_list(cluster_id=new_dst_cluster_configs[0].id)

        return operation_id, old_dst_cluster_hosts[0], new_dst_cluster_hosts[0], old_config.id, new_dst_cluster_configs[0].id

    def change_dns(self, old_dns, new_dns):
        from afisha.infra.libs.dns import DnsClient
        logging.info("Change DNS name")

        secret = self.Parameters.secret_id
        dns_token = secret.data()["dns-token"]
        dns_username = secret.data()["dns-username"]

        dns = DnsClient(token=dns_token, username=dns_username)

        dst_cluster_name = self.Parameters.dst_cluster_name.replace("_", "-")
        cname_record = self.DNS_MAPPING.get(dst_cluster_name)

        # dns section
        logging.info("DNS: Old CNAME: {}".format(old_dns))
        dns.change_request(operation='delete', name=cname_record, type='CNAME', data=old_dns, ttl=60)
        logging.info("DNS: New CNAME: {}".format(new_dns))
        dns.change_request(operation='add', name=cname_record, type='CNAME', data=new_dns, ttl=60)

    def delete_old_cluster(self, cluster_id):
        from afisha.infra.libs.mdb import MdbClientMongo
        logging.info("Delete old MDB cluster: {}".format(cluster_id))
        secret = self.Parameters.secret_id
        mdb_token = secret.data()["mdb-token"]
        mdb = MdbClientMongo(token=mdb_token)

        return mdb.delete_mongo_cluster(cluster_id=cluster_id)

    def update_password(self, cluster_id):
        from afisha.infra.libs.mdb import MdbClientMongo
        logging.info("Changing password in MDB: {}")
        secret = self.Parameters.secret_id
        mdb_token = secret.data()["mdb-token"]
        db_password = secret.data()[self.Parameters.db_username]
        mdb = MdbClientMongo(token=mdb_token)

        return mdb.user_update(cluster_id=cluster_id,
                               username=self.Parameters.db_username,
                               password=db_password)

    def wait_deploy(self, operation_id):
        from afisha.infra.libs.mdb import MdbClientMongo
        logging.info("Wait deploy Operation ID: {}".format(operation_id))
        secret = self.Parameters.secret_id
        mdb_token = secret.data()["mdb-token"]
        mdb = MdbClientMongo(token=mdb_token)
        task_status = mdb.operation_status_get(id=operation_id)

        if task_status["done"]:
            if "error" in task_status:
                raise errors.TaskFailure("MDB task failed")
            else:
                return
        else:
            raise sdk2.WaitTime(300)

    def on_execute(self):

        with self.memoize_stage.first_step(commit_on_entrance=False):
            self.Context.OPERATION_ID, self.Context.OLD_DNS_CNAME, self.Context.NEW_DNS_CNAME, self.Context.OLD_CLUSTER_ID, self.Context.NEW_CLUSTER_ID = self.recovery_mdb()

        self.wait_deploy(self.Context.OPERATION_ID)

        with self.memoize_stage.second_step(commit_on_entrance=False):
            self.Context.OPERATION_ID = self.update_password(self.Context.NEW_CLUSTER_ID)

        self.change_dns(self.Context.OLD_DNS_CNAME, self.Context.NEW_DNS_CNAME)
        self.delete_old_cluster(self.Context.OLD_CLUSTER_ID)
