# coding: utf-8
import re
import logging
import sandbox.sdk2 as sdk2
import sandbox.common.types.misc as ctm
from requests.exceptions import HTTPError
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask


class AfishaPrCleanup(AfishaSandboxBaseTask):

    BINARY_TASK_ATTR_TARGET = "Afisha/infra/AfishaPrCleanup"

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    @property
    def seen_prs(self):
        if self.Context.seen_prs is ctm.NotExists:
            self.Context.seen_prs = []
        return self.Context.seen_prs

    @property
    def closed_prs(self):
        if self.Context.closed_prs is ctm.NotExists:
            self.Context.closed_prs = []
        return self.Context.closed_prs

    def is_pr_closed(self, pr_id):
        logging.info("Checking pr: %s", pr_id)
        if pr_id in self.seen_prs:
            if pr_id in self.closed_prs:
                logging.info("Got cached result: closed")
                return True
            logging.info("Got cached result: not closed")
            return False
        params = {"fields": "id,state"}
        pr = self.ac.get_review_request(pr_id, **params)
        pr_state = pr["data"]["state"]
        self.seen_prs.append(pr_id)
        if pr_state == "closed":
            logging.info("Closed")
            self.closed_prs.append(pr_id)
            return True
        logging.info("Not closed: %s", pr_state)
        return False

    def check_env(self, object_id, name):
        logging.info("Checking environment: %s", object_id)
        regexp = re.compile(r"pr-(\d+)")
        match = regexp.match(name)
        if not match:
            logging.info("Not matched, skip")
            return
        pr_id = match.group(1)
        if not self.is_pr_closed(pr_id):
            logging.info("PR %s not closed, skip", pr_id)
            return
        logging.info("PR %s closed, removing %s", pr_id, object_id)
        result = self.qc.delete_env(object_id)
        result.raise_for_status()
        return object_id

    def check_app(self, app_id, robot_name):
        self._init_qc(robot_name)
        self._init_ac(robot_name)
        result = []
        logging.info("Checking application: %s", app_id)
        prj, app = app_id.split(".")
        try:
            environments = self.qc.list_evnironments(prj, app)
        except HTTPError as error:
            if error.response.status_code != 404:
                raise
            logging.info("No such application, skip")
            return result
        for environment in environments:
            data = self.check_env(environment["objectId"], environment["name"])
            if data is not None:
                result.append(data)
        return result

    def check_envs(self):
        logging.info("Checking components envs from api")
        result = []
        components = self.rc.components.find()
        for component in components:
            if component.pr is None:
                logging.info("No pr settings in %s component", component.name)
                continue
            for application in component.pr.applications:
                data = self.check_app(application, component.robot.robot_name)
                if data:
                    result.extend(data)
        return result

    def check_prs(self):
        logging.info("Checking prs from api")
        result = []
        prs = self.rc.prs.find()
        for pr in prs:
            if not self.is_pr_closed(pr.number):
                continue
            pr.delete()
            result.append(pr.number)
        return result

    def on_execute(self):
        self._init_rc()
        removed_envs = self.check_envs()
        removed_prs = self.check_prs()
        logging.info("Removed envs: %s", removed_envs)
        logging.info("Removed prs: %s", removed_prs)
