# coding: utf-8
import logging
from requests.exceptions import HTTPError

import sandbox.sdk2 as sdk2
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask

WIKI_LINK = "https://nda.ya.ru/t/9b7HwYK14Hfrpt"
FORCE_REMOVE_FLAG = "[stage_expire:true]"
DONT_REMOVE_FLAG = "[stage_expire:false]"


class AfishaPrCleanup2(AfishaSandboxBaseTask):

    BINARY_TASK_ATTR_TARGET = "Afisha/infra/AfishaPrCleanup2"

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(AfishaSandboxBaseTask.Parameters):
        with sdk2.parameters.Group("Settings") as settings_block:
            ttl = sdk2.parameters.Integer("PR TTL in days for cleanup Stage", required=True, default=30)

    def setup(self):
        self._init_rc()
        self._init_ac(self.author)
        self._init_bbc(self.author)
        self._init_ghc(self.author)
        self._init_dc(self.author)
        self.vcs_clients_map = {
            "bitbucket": self.bbc,
            "arcadia": self.ac,
            "github": self.ghc
            }

    def post_pr_comment(self, component, pr, message):
        return self.vcs_clients_map[component.vcs.vcs_type].comment(component.vcs.vcs_path, pr.number, message)

    def is_pr_closed(self, component, pr):
        return self.vcs_clients_map[component.vcs.vcs_type].is_pr_closed(component.vcs.vcs_path, pr.number)

    def is_pr_old(self, component, pr):
        if self.Parameters.ttl <= 0:
            return False
        last_pr_update_day = self.vcs_clients_map[component.vcs.vcs_type].last_pr_update_days(component.vcs.vcs_path, pr.number)
        return True if last_pr_update_day > self.Parameters.ttl else False

    def force_remove_flag(self, component, pr):
        pr_summary = self.vcs_clients_map[component.vcs.vcs_type].get_pr_summary(component.vcs.vcs_path, pr.number)
        return FORCE_REMOVE_FLAG in pr_summary.split(" ")

    def dont_remove_flag(self, component, pr):
        pr_summary = self.vcs_clients_map[component.vcs.vcs_type].get_pr_summary(component.vcs.vcs_path, pr.number)
        return DONT_REMOVE_FLAG in pr_summary.split(" ")

    def remove_stages(self, component, pr):
        result = True
        logging.info("Processing applications for %s pr in %s component: ", pr.number, component.name)
        for application in component.pr.applications:
            logging.info("Now processing %s", application)
            env_id = "{}.{}{}".format(application, component.pr.pr_env_prefix, pr.number)
            try:
                stage_result = self.dc.remove(env_id).rstrip()
            except HTTPError as error:
                error_code = error.response.status_code
                if error_code == 404:
                    logging.info("Stage %s does not exist", env_id)
                    continue
                raise
            else:
                logging.info("Stage for application %s removed", application)
            if stage_result != "True":
                result = False
        return result

    def process_component_pr(self, component, pr):
        logging.info("Processing %s pr for %s component", pr.number, component.name)
        message = ""
        pr_checks_breaking = {
            "marked with \'[stage_expire:false]\' flag\n": self.dont_remove_flag(component, pr)
        }
        pr_checks = {
            "marked with \'[stage_expire:true]\' flag\n": self.force_remove_flag(component, pr),
            "closed\n": self.is_pr_closed(component, pr),
            "updated more then {} days ago\n".format(self.Parameters.ttl) : self.is_pr_old(component, pr)
        }
        for reason, condition in pr_checks_breaking.items():
            if condition:
                logging.info("Skipping. Stages for PR %s was NOT removed due to PR has not been %s", pr.number, reason)
                return
        if not any(pr_checks.values()):
            logging.info("Skipping. Stages for PR %s was NOT removed due to PR falied to pass remove checks", pr.number)
            return
        try:
            result = self.remove_stages(component, pr)
        except Exception as error:
            logging.error("Error while removing stages, not cleaning pr from base: %s", error)
        else:
            for reason, condition in pr_checks.items():
                if condition:
                    message += "Stages for PR {} WAS REMOVED due to PR has been {}".format(pr.number, reason)
            logging.info(message)
            if result:
                pr.delete()
                logging.info("Stages for PR was successfully removed from https://api.infra.afisha.yandex.net/releases/pr")
                message += u"\nсгенерировано в [sb](https://sandbox.yandex-team.ru/task/{}/view) | " \
                       u"[doc]({})".format(self.id, WIKI_LINK)
                self.post_pr_comment(component, pr, message)

    def process_component(self, component):
        logging.info("Processing component: %s", component.name)
        if not component.pr:
            logging.info("No pr settings in %s component", component.name)
            return
        if self.author != component.robot.robot_name:
            logging.info("Task started by %s, registered robot for component %s: %s. Skipping",
                         self.author, component.name, component.robot.robot_name)
            return
        for pr in component.prs():
            self.process_component_pr(component, pr)

    def on_execute(self):
        self.setup()
        components = self.rc.components.find()
        for component in components:
            self.process_component(component)
