# coding: utf-8
import logging
from datetime import datetime, timedelta
import sandbox.sdk2 as sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask


class AfishaReleasesMonitoring(AfishaSandboxBaseTask):

    BINARY_TASK_ATTR_TARGET = "Afisha/infra/AfishaReleasesMonitoring"

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(AfishaSandboxBaseTask.Parameters):
        kill_timeout = 120

    def on_enqueue(self):
        releases_monitoring_lock = ctt.Semaphores.Acquire(
            name="afisha_releases_monitoring",
            weight=1, capacity=1)
        release = (ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        self.Requirements.semaphores = ctt.Semaphores(acquires=[releases_monitoring_lock], release=release)

    @property
    def user(self):
        return self.author

    @property
    def problems(self):
        if self.Context.problems is ctm.NotExists:
            self.Context.problems = {}
        return self.Context.problems

    @problems.setter
    def problems(self, data):
        """
        Add problem
        :param data: tuple(ReleaseComponent, Exception)
        :return: None
        """
        component, problem = data
        if component.name in self.problems:
            self.problems[component.name].append(str(problem))
        else:
            self.problems[component.name] = [str(problem)]

    def _get_duty_data_from_u(self, duty_id):
        return {
            "members": list(self.u.team(duty_id).keys()),
            "on_duty": self.u.onduty(duty_id)
        }

    def _get_duty_data_from_abc(self, duty_id):
        abc_slug, abc_duty_slug = duty_id.split(":")
        return {
            "members": self.abc.find_duty_members(abc_slug, abc_duty_slug),
            "on_duty": self.abc.find_duty(abc_slug, abc_duty_slug)
        }

    def _get_duty_data(self, duty_type, duty_id):
        if duty_type == "u":
            duty = self._get_duty_data_from_u(duty_id)
        elif duty_type == "abc":
            duty = self._get_duty_data_from_abc(duty_id)
        else:
            raise NotImplementedError()
        return duty

    def get_duties(self, component):
        result = {}
        if component.duties:
            for duty in component.duties:
                result[duty.duty_name] = self._get_duty_data(duty.duty_type, duty.duty_id)
        elif component.u_team:
            result = {"dev": self._get_duty_data("u", component.u_team)}
        return result

    def setup(self):
        self._init_rc()
        self._init_nyan()
        self._init_u()
        self._init_abc(self.user)
        self._init_staff(self.user)

    def notify(self, component, *args):
        if not component.telegram_chat_id:
            logging.debug("No telegram_chat_id in component, skipping")
            return
        onduty = self.get_duties(component).get("dev", {}).get("on_duty", None)
        if onduty:
            onduty = self.staff.find_by_login(onduty, params={})[0]
        message = u"Проблемы с *{}*:\n".format(component.name)
        for problem in args:
            message += "\\* {}\n".format(problem)
        if onduty:
            message += u"Дежурный: [{user}](https://staff.yandex-team.ru/{user})".format(user=onduty.login)
            if onduty.public_telegram:
                message += u" (@{})".format(self.nyan.escape(onduty.public_telegram))
        self.nyan.send(component.telegram_chat_id, text=message, parse_mode="Markdown")

    def process(self, component):
        checks = [self.check_if_stuck, self.check_merges]
        messages = []
        for check in checks:
            logging.debug("Running check: %s", check.__name__)
            problem = check(component)
            if problem:
                logging.debug("Found problem: %s", problem)
                if isinstance(problem, list):
                    messages.extend(problem)
                else:
                    messages.append(problem)
        if messages:
            logging.debug("Problems found, notifying")
            self.notify(component, *messages)
        component.monitoring.save()

    @staticmethod
    def check_if_stuck(component):

        def get_task_link():
            link = "https://sandbox.yandex-team.ru/task/{}/view"
            mapping = {
                "built": "build_task",
                "build": "build_task",
                "build_failed": "build_task",
                "release": "release_task"
            }
            return link.format(getattr(component.last_log, mapping.get(component.last_log.status, mapping["release"])))

        max_lag = 3600  # seconds
        try:
            if component.max_notify_lag:
                max_lag = component.max_notify_lag
            else:
                logging.info("max_notify_lag is set to 3600s (value is not set in infra-api)")
        except AttributeError:
            logging.info("max_notify_lag is set to 3600s (field was not found in infra-api)")

        problem = None

        if not component.last_log or component.last_log.status == "finished":
            return
        last_processed = getattr(component.monitoring, "last_log_ts", None)
        if last_processed and last_processed >= component.last_log.timestamp:
            # Already processed entry, skip
            return

        elapsed_time = datetime.utcnow() - component.last_log.timestamp
        if elapsed_time.seconds > max_lag or "failed" in component.last_log.status:
            release = component.release(component.last_log.version)
            problem = u"[{issue} ({})](https://st.yandex-team.ru/{issue}) залип в [{}]({}) (>{}с)".format(
                release.number, component.last_log.status, get_task_link(), max_lag, issue=release.issue)

            component.monitoring.last_log_ts = component.last_log.timestamp
        return problem

    def check_merges(self, component):
        problems = []

        till_time = datetime.now()
        since_time = till_time - timedelta(hours=24)
        searchq = {
            "type": ["MERGE_TO_STABLE"],
            "input_parameters": {"component_name": component.name},
            "hidden": True,
            "created": "{}..{}".format(since_time.isoformat(), till_time.isoformat()),
            "status": ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
        }

        logging.debug("Searching tasks: %s", searchq)
        tasks = sdk2.Task.find(**searchq).limit(100)
        for task in tasks:
            logging.debug("Checking task: %s", task.id)
            if component.monitoring.last_merge_id and task.id <= int(component.monitoring.last_merge_id):
                continue
            if task.status not in ctt.Status.Group.SUCCEED:
                commit_user = self.staff.find_by_login(task.Parameters.commit_user, params={})[0]
                branches = ",".join([str(x) for x in task.Parameters.branch_nums_to_merge])
                revs = task.Parameters.revs.split(",")
                msg = u"[Мерж в {} ветку](https://sandbox.yandex-team.ru/task/{}/view) похоже упал. " \
                      u"Коммитер: [{user}](https://staff.yandex-team.ru/{user})\n ".format(
                          branches, task.id, user=commit_user.login)
                if commit_user.public_telegram:
                    msg += " (@{})".format(self.nyan.escape(commit_user.public_telegram))
                commits = [u"[{commit}](https://a.yandex-team.ru/arc/commit/{commit})".format(commit=rev) for rev in revs]
                msg += u"\n  Коммиты: {}".format(",".join(commits))
                problems.append(msg)
            component.monitoring.last_merge_id = str(task.id)
        return problems

    def on_execute(self):
        self.setup()
        components = self.rc.components.find()
        for component in components:
            logging.debug("Processing component: %s", component.name)
            self.process(component)
