# coding: utf-8
import logging
import sandbox.sdk2 as sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask


class AfishaS3Cleaner(AfishaSandboxBaseTask):

    BINARY_TASK_ATTR_TARGET = 'Afisha/infra/AfishaS3Cleaner'

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared cache

    class Parameters(AfishaSandboxBaseTask.Parameters):
        with sdk2.parameters.Group("Settings") as settings_block:
            s3_endpoint = sdk2.parameters.String("S3: endpoint", required=True)
            s3_bucket = sdk2.parameters.String("S3: bucket", required=True)
            s3_prefix = sdk2.parameters.String("S3: prefix", required=True)
            s3_pattern = sdk2.parameters.String("S3: pattern", required=True)
            keep_versions = sdk2.parameters.Integer("Versions: how many versions need to keep", required=True)
            secret_id = sdk2.parameters.YavSecret("YAV: S3 credentials", required=True)
            dry_run = sdk2.parameters.Bool("Dry run s3 cleaner", default=False)

    def on_enqueue(self):
        component_flow_lock = ctt.Semaphores.Acquire(
            name="{}_{}".format("afisha_s3_cleaner", self.Parameters.s3_bucket),
            weight=1, capacity=1)
        release = (ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        self.Requirements.semaphores = ctt.Semaphores(acquires=[component_flow_lock], release=release)

    @staticmethod
    def try_to_int(s):
        try:
            return int(s)
        except ValueError:
            return s

    def human_key(self, st):
        import re
        return [self.try_to_int(c) for c in re.split('([0-9]+)', st)]

    def human_sort(self, lst):
        """
        Sort a list of strings in the way that humans expect
        """
        lst.sort(key=self.human_key, reverse=True)

    def s3_cleaner(self):
        import boto3
        import re
        secret = self.Parameters.secret_id
        s3_access_key_id = secret.data()["S3-AccessKeyId"]
        s3_access_secret_key = secret.data()["S3-AccessSecretKey"]

        session = boto3.session.Session(
            aws_access_key_id=s3_access_key_id,
            aws_secret_access_key=s3_access_secret_key,
        )
        client = session.client(
            's3',
            endpoint_url=self.Parameters.s3_endpoint,
        )
        logging.info("Get list of objects from S3")
        list_of_objects = client.list_objects_v2(Bucket=self.Parameters.s3_bucket,
                                                 Prefix=self.Parameters.s3_prefix,
                                                 Delimiter='/')
        if list_of_objects['KeyCount'] != 0:
            folders = [o.get('Prefix') for o in list_of_objects.get('CommonPrefixes')]
        else:
            logging.info("List is empty, exiting...")
            return

        logging.info("Detected folders by prefix: %s" % ", ".join(folders))

        pattern = re.compile(self.Parameters.s3_pattern)
        folders_match = [i for i in folders if pattern.findall(i)]
        logging.info("Detected folders by s3_pattern regexp: %s" % ", ".join(folders_match))
        self.human_sort(folders_match)
        folders_remove = folders_match[self.Parameters.keep_versions:]
        if not folders_remove:
            logging.info("Nothing to remove, exiting...")
            return

        logging.info("Folders to remove: %s" % ", ".join(folders_remove))

        for folder in folders_remove:
            paginator = client.get_paginator('list_objects_v2')
            pages = paginator.paginate(Bucket=self.Parameters.s3_bucket, Prefix=folder)
            for objects_to_delete in pages:
                delete_keys = {'Objects': []}
                delete_keys['Objects'] = [{'Key': k} for k in [obj['Key'] for obj in objects_to_delete.get('Contents', [])]]
                if not self.Parameters.dry_run:
                    logging.info("Removing old data from S3... Current folder: %s" % folder)
                    client.delete_objects(Bucket=self.Parameters.s3_bucket, Delete=delete_keys)

    def on_execute(self):

        self.s3_cleaner()
