# coding: utf-8

from logging import getLogger, basicConfig, INFO
import urllib3
from copy import deepcopy
import os
import helpers
from helpers import Config
from errors import NotFoundError


import sandbox.sdk2 as sdk2
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.Afisha.base import AfishaSandboxBaseTask


class AfishaStPinger(AfishaSandboxBaseTask):

    BINARY_TASK_ATTR_TARGET = 'Afisha/infra/AfishaStPinger'

    class Parameters(AfishaSandboxBaseTask.Parameters):
        with sdk2.parameters.Group("Входные параметры"):
            UA = sdk2.parameters.String('UA', default_value='Alet StarTrek Pinger (https://github.yandex-team.ru/alet/st-pinger)')
            DRY_RUN = sdk2.parameters.Bool('DRY_RUN', default_value=True)
            ONE_TASK = sdk2.parameters.String('ONE_TASK', default_value="")
            CONFIG_PATH = sdk2.parameters.String('CONFIG_PATH', default_value="sandbox/projects/Afisha/infra/AfishaStPinger/config.yaml")
            ROBOT_FOR_ST = sdk2.parameters.String('ROBOT_FOR_ST', default_value="robot-afisha")
            YAV_SECRET = sdk2.parameters.YavSecret(u"Секрет с токеном к StarTrek под ключем startrek\n",
                                                        default="sec-01cqnt01f7crq55y7334e6xk6y", required=True)
            USE_YAV = sdk2.parameters.Bool('USE_YAV', default_value=True)

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 256
        disk_space = 100

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    """
    Issues check
    Function name must start with 'check_' prefix
    Function args:
    issue: startrek issue object
    startrek: startrek client object
    **kwargs: queue check options from config
    Function return: tuple
    processed , stop
        processed: is issue processed? Required
        stop: break checking loop? By default False
    """

    def check_outdated_normal(self, issue, wait_days, max_summons):
        """
            Find issues without any activity for wait_days
            summon author for max_summons times
            if there are still no activity - close
            And break cycle of checks processing
        """

        self.LOG.info("Processing {} with outdated_normal check".format(issue.key))
        last_update = helpers.days_since(issue.updatedAt)

        if last_update < wait_days:
            self.LOG.info("Seems issue is actual (%s < %s)", last_update, wait_days)
            return False

        template = 'summon_task.tpl'
        to_summon = [issue.assignee or issue.createdBy]  # old task without assigne can summon None (helpers.can_close_soon always False because helpers.summon have None arg) -> add or

        # If people don't care, we'll close the issue
        if helpers.can_close_soon(issue, self.st, max_summons):
            if "close_soon" in issue.tags:
                self.LOG.info("close_soon in tags, max_summon reached. Closing")

                if not self.DRY_RUN:
                    helpers.close_issue(issue)

                return True, True
            else:
                self.LOG.info("Max summon reached, but no close_soon tag. Last call")
                to_summon = [issue.createdBy]

                template = "last_call.tpl"

                if not self.DRY_RUN:
                    issue.update(tags=issue.tags + ["close_soon"],
                                 ignore_version_change=True)

        self.LOG.info("Summon %s", to_summon)

        if not self.DRY_RUN:
            helpers.summon(issue, self.st, template, users=to_summon, config_path=self.Parameters.CONFIG_PATH)

        return True, True

    def check_outdated_critical(self, issue, **kwargs):
        """
            Find not closed issues with critical/blocker priority
            Summon users from config, choice based on days since issue was created
            And break cycle of checks processing
        """

        self.LOG.info("Processing {} with outdated_critical check".format(issue.key))

        status_black_list = ["closed", "tested"]
        template = "summon_critical.tpl"
        tag_prefix = "outdated_critical_"
        priority_white_list = kwargs.get("priority_white_list", ["blocker"])
        escalation_days = kwargs["days"]

        if issue.status.key in status_black_list:
            self.LOG.info("Task status {} in black list".format(issue.status.key))
            return False

        if issue.priority.key not in priority_white_list:
            self.LOG.info("Task priority {} not in white list".format(issue.priority.key))
            return False

        escalation_steps = [int(x) for x in sorted(escalation_days.keys())]
        days = helpers.days_since(issue.updatedAt)

        last = helpers.get_tag_id(issue, tag_prefix)

        if last is None:
            last = -1
        #  Already summoned everybody
        if last+1 >= len(escalation_steps):
            return False

        self.LOG.info("Days since update %s, threshold %s (last: %s)",
                      days, escalation_steps[last+1], last)

        if days >= escalation_steps[last+1]:

            self.LOG.info("Adding %s%s tag, trying to remove %s%s tag, summon: %s",
                          tag_prefix, last+1, tag_prefix, last,
                          escalation_days[str(escalation_steps[last+1])])

            if not self.DRY_RUN:
                try:
                    issue.tags.remove("{}{}".format(tag_prefix, last))
                except ValueError:
                    pass

                issue.update(tags=issue.tags + ["{}{}".format(tag_prefix, last+1)],
                             ignore_version_change=True)
                helpers.summon(issue, self.st, template,
                               users=escalation_days[str(escalation_steps[last+1])],
                               config_path=self.Parameters.CONFIG_PATH,
                               **kwargs)

            return True, True
        return False

    def process_issue(self, issue, **kwargs):
        """ Process issue with checks """

        processed = False

        if "no_st_pinger" in issue.tags:
            return processed

        for func_name in kwargs.keys():
            check_config = kwargs.get(func_name)

            result = getattr(self, func_name)(issue, **check_config)

            if isinstance(result, tuple):
                _processed, stop = result
            else:
                _processed = result
                stop = False

            processed = processed | _processed
            if stop is True:
                break

        return processed

    def walk_st(self, queue, default_config):
        """ Process issues from queue """

        config = deepcopy(default_config)
        config.update(queue.get("checks", {}))
        processed_issues = []

        search_query = "Queue: {} ".format(queue["name"]) + \
                       "Status: !closed " + \
                       '"Sort By": updated ASC'

        issues = self.st.issues.find(search_query, per_page=10000)
        for issue in issues:

            if self.ONE_TASK and issue.key != self.ONE_TASK:
                self.LOG.info("ONE_TASK is {} -> skip task {}".format(self.ONE_TASK, issue.key))
                continue

            if self.process_issue(issue, **config) is True:
                processed_issues.append(issue.key)

        return processed_issues

    def on_execute(self):
        if self.Parameters.USE_YAV:
            self._init_yav_st(self.Parameters.YAV_SECRET, useragent=self.Parameters.UA, key="startrek")
        else:
            self._init_st(self.Parameters.ROBOT_FOR_ST, useragent=self.Parameters.UA)
        self.DRY_RUN = self.Parameters.DRY_RUN
        self.ONE_TASK = self.Parameters.ONE_TASK

        self.LOG = getLogger()

        # Disable warning for urllib3
        urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
        basicConfig(level=INFO)
        with arcadia_sdk.mount_arc_path("arcadia-arc:/#trunk") as arcadia:
            try:
                with open(os.path.join(arcadia, self.Parameters.CONFIG_PATH)) as config_resource:
                    config_object = Config(config_resource)
                    config = config_object.config_yaml
            except Exception as exception:
                raise NotFoundError("Couldn't open config %s. Details: %s" % (self.Parameters.CONFIG_PATH, exception))

        for queue in config['queues']:

            if self.ONE_TASK:
                q_name = queue['name']
                if q_name != self.ONE_TASK.split("-")[0]:
                    self.LOG.info("ONE_TASK is {} -> skip queue {}".format(self.ONE_TASK, q_name))
                    continue

            done_issues = self.walk_st(queue, config['defaults'])
            self.LOG.info("Done for {}: {}".format(queue['name'], done_issues))
