""" Helper function """
import yaml
import datetime
from collections import deque
from logging import getLogger
from jinja2 import Template

LOG = getLogger()


class MySafeLoader(yaml.SafeLoader):

    def construct_mapping(self, node, deep=False):
        """ Load dict keys as strings in yaml"""
        data = super(MySafeLoader, self).construct_mapping(node, deep)
        return {
            (str(key) if isinstance(key, int) else key): data[key] for key in data
        }


class ValidationError(Exception):
    """ Exception for config validation error """
    pass


class Config(object):

    CONFIG_KEYS = {"queues", "defaults"}
    QUEUE_KEYS = {"name", "checks"}
    CHECKS_KEYS = {"check_outdated_normal", "check_outdated_critical"}
    OUTDATED_NORMAL_KEYS = {"wait_days": int, "max_summons": int}
    OUTDATED_CRITICAL_KEYS = {"days": dict}

    def validate_check_outdated_normal(self, check):
        for field, field_type in self.OUTDATED_NORMAL_KEYS.items():
            if field not in check.keys():
                raise ValidationError(
                    "Error validating check_outdated_normal: you must specify %s" % field)
            if not isinstance(check[field], field_type):
                raise ValidationError(
                    "Error validating check_outdated_normal: %s must be %s" % (field, str(field_type)))

    def validate_check_outdated_critical(self, check):
        for field, field_type in self.OUTDATED_CRITICAL_KEYS.items():
            if field not in check.keys():
                raise ValidationError(
                    "Error validating check_outdated_critical: you must specify %s" % field)
            if not isinstance(check[field], field_type):
                raise ValidationError(
                    "Error validating check_outdated_critical: %s must be %s" % (field, str(field_type)))

    def validate_checks(self, checks):
        if not set(checks.keys()).issubset(self.CHECKS_KEYS):
            raise ValidationError(
                "Error validating checks %s: not existing checks. Expected checks: %s" % (checks.keys(), str(self.CHECKS_KEYS)))
        for check, args in checks.items():
            getattr(self, "validate_" + check)(args)

    def validate_queue(self, queue):
        if set(queue.keys()) != self.QUEUE_KEYS:
            raise ValidationError(
                "Error validating queue %s: invalid fields. Expected: %s" % (queue, str(self.QUEUE_KEYS)))
        self.validate_checks(queue["checks"])

    def validate_config(self, config_yaml):
        if set(config_yaml.keys()) != self.CONFIG_KEYS:
            raise ValidationError(
                "Error validating config %s: invalid fields. Expected: %s" % (self.config_file, str(self.CONFIG_KEYS)))
        self.validate_checks(config_yaml['defaults'])
        for queue in config_yaml['queues']:
            self.validate_queue(queue)

    def __init__(self, config_file):
        try:
            config_yaml = yaml.load(config_file, Loader=MySafeLoader)
        except Exception as error:
            raise ValidationError("Error while loading config from %s: %s" %
                                    (config_file.buffer.name, error))
        self.validate_config(config_yaml)
        self.config_yaml = config_yaml


def days_since(date, format="%Y-%m-%dT%H:%M:%S.%f+0000"):
    return (
        datetime.datetime.now() - datetime.datetime.strptime(date, format)
    ).days


def can_close_soon(issue, startrek, max_summons):
    comments = issue.comments.get_all()
    queue = deque(maxlen=max_summons)
    for comment in comments:
        queue.append(comment.createdBy.login)

    if queue.count(startrek.myself.login) >= max_summons:
        return True
    return False


def close_issue(issue, resolution="won'tFix", template="autoclose.tpl"):
    issue.tags.append('auto_closed')
    issue.update(tags=issue.tags, ignore_version_change=True)

    from library.python import resource
    t = Template(resource.find("sandbox/projects/Afisha/infra/AfishaStPinger/helpers/"+template))

    # fix 404 error
    # curl  -H "Authorization: OAuth <TOKEN>" https://st-api.yandex-team.ru/v2/issues/<TASK_ID>/transitions
    my_action = [action.id for action in issue.transitions.get_all() if action.id in ("close", "closed")][0]

    try:
        issue.transitions[my_action].execute(comment=t.render(),
                                             resolution=resolution)
    except Exception as e:
        LOG.info("skip issue: {} with reason: {}'  ".format(issue.key, e))


def summon(issue, startrek, template, users=None, config_path=None, **kwargs):
    if users is None:
        users = [issue.createdBy if issue.createdBy else issue.assignee]

    to_summon = []

    for user in list(users):
        if isinstance(user, str):
            user = startrek.users.get(user)
        if isinstance(user, classmethod):
            user = startrek.users.get(user.login)
        # Only humans are allowed
        if user.login.startswith(('robot-', 'zomb-')):
            continue
        to_summon.append(user)

    if not to_summon:
        # If we can not collect users from Created/Assigned, so we need to get queue lead login
        to_summon = [startrek.queues.get(issue.queue.key).lead.login]

    from library.python import resource
    t = Template(resource.find("sandbox/projects/Afisha/infra/AfishaStPinger/helpers/"+template))

    issue.comments.create(summonees=to_summon, text=t.render(context=kwargs) +
                                                "\n((https://a.yandex-team.ru/arc_vcs/{} st-pinger config))".format(config_path))
    return to_summon


def get_tag_id(issue, prefix, delimiter="_"):
    for tag in issue.tags:
        if tag.startswith(prefix):
            return int(tag.split(delimiter)[-1])
